/*
 * Copyright (C) 2004, 2006-2010, Gostai S.A.S.
 *
 * This software is provided "as is" without warranty of any kind,
 * either expressed or implied, including but not limited to the
 * implied warranties of fitness for a particular purpose.
 *
 * See the LICENSE file for more information.
 */

/// \file urbi/uabstractclient.hh
/// \brief Definition of the URBI interface class

#ifndef URBI_UABSTRACTCLIENT_HH
# define URBI_UABSTRACTCLIENT_HH

# include <libport/cstdio>
# include <libport/sys/types.h>
# include <libport/cstring>
# include <libport/cstdlib>
# include <cstdarg>

# include <list>
# include <iostream>
# include <string>

# include <libport/compiler.hh>
# include <libport/fwd.hh>
# include <libport/lockable.hh>
# include <libport/traits.hh>

# include <urbi/fwd.hh>
# include <urbi/export.hh>
# include <urbi/ubinary.hh>
# include <urbi/umessage.hh>

/**
\mainpage
This document is the autogenerated documentation for Urbi SDK Remote.

Classes of interest:
- Urbi connections: urbi::UAbstractClient, urbi::UClient, urbi::USyncClient.
- Data types: urbi::UValue, urbi::UBinary.
- UObject API: urbi::UObject, urbi::UVar.
*/
namespace urbi
{
  /// Return values for the callcack functions.
  /*! Each callback function, when called, must return with either URBI_CONTINUE
    or URBI_REMOVE:
    - URBI_CONTINUE means that the client should continue to call this callbak
    function.
    - URBI_REMOVE means that the client should never call this callback again.
  */
  enum UCallbackAction
    {
      URBI_CONTINUE=0,
      URBI_REMOVE
    };

  /// Maximum length of an URBI tag.
  enum { URBI_MAX_TAG_LENGTH = 64 };

  typedef unsigned int UCallbackID;

# define UINVALIDCALLBACKID 0

  /// Callback prototypes.
  typedef UCallbackAction (*UCallback)		   (const UMessage&msg);

  typedef UCallbackAction (*UCustomCallback)	   (void * callbackData,
						    const UMessage&msg);


  //used internaly
  class UCallbackInfo
  {
  public:
    UCallbackInfo(UCallbackWrapper &w);
    bool operator==(UCallbackID id) const;

    char tag[URBI_MAX_TAG_LENGTH];
    UCallbackWrapper& callback;
    UCallbackID id;
  };

  //used internaly
  class UClientStreambuf;


  /// Interface for an URBI wrapper object.
  /*! Implementations of this interface are wrappers around the URBI protocol.
    It handles URBI messages parsing, callback registration and various
    formatting functions.
    Implementations of this interface should:
    - Redefine errorNotify() as a function able to notify the user of eventual
    errors.
    - Redfine the four mutual exclusion functions.
    - Redefine effectiveSend().
    - Fill recvBuffer, update recvBufferPosition and call processRecvBuffer()
    when new data is available.
    - Provide an execute() function in the namespace urbi, that never returns,
    and that will be called after initialization.
    - Call onConnection when the connection is established.

   See the liburbi-cpp documentation for more informations on
   how to use this class.
  */
  class URBI_SDK_API UAbstractClient : public std::ostream
  {
  public:
    /// Connection Buffer size.
    enum { URBI_BUFLEN = 128000 };
    /// Standard port of URBI server.
    enum { URBI_PORT = 54000 } ;
    /// Default host.
    static const char* default_host();

    /// Error code.
    /// 0 iff no error.
    /// Other values are unspecified.
    typedef int error_type;

    /// Create a new instance and connect to the Urbi server.
    /*! Initializes sendBuffer and recvBuffer, and copy host and port.
      \param host    IP address or name of the robot to connect to.
      \param port    TCP port to connect to.
      \param buflen  size of send and receive buffers.
      Implementations should establish the connection in their constructor.
    */
    UAbstractClient(const std::string& host = default_host(),
                    unsigned port = URBI_PORT,
		    size_t buflen = URBI_BUFLEN,
		    bool server = false);

    virtual ~UAbstractClient();

    bool init() const;
    /// Return current error status, or zero if no error occurred.
    error_type error() const;

    /*----------.
    | Sending.  |
    `----------*/
    /**
    * \defgroup Sending urbiscript code.
    *
    * All the send functions are expected to be asynchronous by default: they
    * return immediately, and
    * send the data in a separate thread.
    */
    //@{


    /// Send an Urbi command. The syntax is similar to the printf()
    /// function.
    /// Passing `0' is supported as means `""', but with no warning.
    error_type send(const char* format, ...)
      __attribute__((__format__(printf, 2, 3)));

    /// Send the value without any prefix or terminator
    error_type send(const urbi::UValue& v);

    /// Send the remainder of the stream.
    error_type send(std::istream& is);

    /// Send an urbi Binary value.
    error_type sendBinary(const void* data, size_t len,
                          const std::string& header);

    /// Send binary data.
    error_type sendBin(const void*, size_t len);

    /// Send an Urbi header followed by binary data.
    error_type sendBin(const void*, size_t len, const char* header, ...)
      __attribute__((__format__(printf, 4, 5)));

    /// Lock the send buffer (for backward compatibility, will be
    /// removed in future versions).
    error_type startPack();

    /// Unlock the send buffer (for backward compatibility, will be
    /// removed in future versions).
    error_type endPack();

    /// Append Urbi commands to the send buffer (for backward
    /// compatibility, will be removed in future versions).
    /// Passing `0' is supported as means `""', but with no warning.
    error_type pack(const char* format, ...)
      __attribute__((__format__(printf, 2, 3)));

    /// va_list version of pack.
    /// Passing `0' is supported as means `""', but with no warning.
    error_type vpack(const char* format, va_list args);

    /// Send urbi commands contained in a file.
    /// The file "/dev/stdin" is recognized as referring to std::cin.
    error_type sendFile(const std::string& f);

    /// Send a command, prefixing it with a tag, and associate the
    /// given callback with this tag.
    UCallbackID sendCommand(UCallback, const char*, ...)
      __attribute__((__format__(printf, 3, 4)));

    /// Send a command, prefixing it with a tag, and associate the
    /// given callback with this tag.
    UCallbackID sendCommand(UCustomCallback, void *, const char*, ...)
      __attribute__((__format__(printf, 4, 5)));

    /// Send sound data to the robot for immediate playback.
    error_type sendSound(const char* device,
		  const urbi::USound& sound, const char* tag = 0);

    /// Put a file on the robot's mass storage device.
    error_type putFile(const char* localName, const char* remoteName = 0);

    /// Save a buffer to a file on the robot.
    error_type putFile(const void* buffer, size_t length,
                       const char* remoteName);

    //@}

    //@{
    /**
    * \defgroup Setting read callbacks.
    *
    * All those functions register function callbacks that are called on
    * some events. The UClient subclass calls those callbacks synchronously
    * in the main read thread, so no other callback will be called until
    * the first one returns.
    * The USyncClient subclass uses a separate thread per instance.
    */

    /// Associate a callback function with a tag. New style.
    /*!
    \param callback a callback function wrapper, generated by callback()
    \param tag the tag to associate the callback with
    */
    UCallbackID setCallback(UCallbackWrapper& callback, const char* tag);

    /// Associate a callback function with all error messages from the server
    UCallbackID setErrorCallback(UCallbackWrapper& callback);

    /// Associate a callback with all messages
    UCallbackID setWildcardCallback(UCallbackWrapper& callback);

    /// Associate a callback with local connection errors
    UCallbackID setClientErrorCallback(UCallbackWrapper& callback);

    /// OLD-style callbacks
    UCallbackID setCallback(UCallback, const char* tag);

    /// Associate a callback function with a tag, specifiing a
    /// callback custom value that will be passed back to the callback
    /// function.
    UCallbackID setCallback(UCustomCallback, void* callbackData,
                            const char* tag);

    //@}
    //@{
    /// \deprecated{ Callback to class member functions(old-style).}

    template<class C>
    UCallbackID
    setCallback(C& ref, UCallbackAction (C::*)(const UMessage&),
                            const char * tag);
    template<class C, class P1>
    UCallbackID
    setCallback(C& ref, UCallbackAction (C::*)(P1, const UMessage&),
                P1, const char * tag);
    template<class C, class P1, class P2>
    UCallbackID
    setCallback(C& ref, UCallbackAction (C::*)(P1 , P2, const UMessage&),
                P1, P2,	   const char * tag);

    template<class C, class P1, class P2, class P3>
    UCallbackID
      setCallback(C& ref, UCallbackAction (C::*)(P1 , P2, P3, const UMessage&),
                  P1, P2, P3, const char* tag);

    template<class C, class P1, class P2, class P3, class P4>
    UCallbackID
    setCallback(C& ref, UCallbackAction(C::*)(P1 , P2, P3, P4, const UMessage&),
                P1, P2, P3, P4, const char* tag);
    //@}

    /// Get the tag associated with a registered callback.
    /// \return 1 and fill tag on success, 0 on failure.
    int getAssociatedTag(UCallbackID id, char* tag);

    /// Delete a callback.
    /// \return 0 if no callback with this id was found, 1 otherwise.
    int deleteCallback(UCallbackID id);

    /// Return a identifier, for tags for instance.
    std::string fresh();

    /// Fill tag with a unique tag for this client.
    /// Obsolete, use fresh() instead.
    ATTRIBUTE_DEPRECATED
    void makeUniqueTag(char* tag);

    /// Pass the given UMessage to all registered callbacks with the
    /// corresponding tag, as if it were comming from the URBI server.
    virtual void notifyCallbacks(const UMessage& msg);

    /// Notify of an error.
    virtual void printf(const char* format, ...)
      __attribute__((__format__(printf, 2, 3)))
      = 0;

    /// Get time in milliseconds since an unspecified but constant
    /// reference time.
    virtual unsigned int getCurrentTime() const = 0;

    /// Active KeepAlive functionality
    /// Sends an URBI message at specified interval, if no anwser is received
    /// close the connection and notify 'URBI_ERROR_CONNECTION_TIMEOUT'.
    ///
    /// \param pingInterval  interval between ping messages
    ///                      in milliseconds, 0 to disable.
    /// \param pongTimeout   timeout in milliseconds to wait answer.
    virtual void setKeepAliveCheck(unsigned pingInterval,
                                   unsigned pongTimeout) = 0;

    /// Return the server name or IP address.
    const std::string& getServerName() const;

    /// Return the server port.
    unsigned getServerPort() const;
    /// Called each time new data is available in recvBuffer.
    void processRecvBuffer();

  private:
    /// New binary data is available.
    /// \return true if there is still data to process.
    bool process_recv_buffer_binary_();

    /// New text data is available.
    /// \return true if there is still data to process.
    bool process_recv_buffer_text_();

  public:
    /// This, as a stream.
    std::ostream& stream_get();

    /// dummy tag for client error callback.
    static const char* CLIENTERROR_TAG;

  protected:
    /// Must be called by subclasses when the connection is established.
    void onConnection();

    /// Executed when closing connection.
    virtual error_type onClose();
    bool closed_;

    /// Queue data for sending, returns zero on success, nonzero on failure.
    virtual error_type effectiveSend(const void* buffer, size_t size) = 0;

    /// Bounce to effectiveSend() using strlen.
    error_type effective_send(const char* buffer);

    /// Bounce to effectiveSend() using c_str().
    error_type effective_send(const std::string& buffer);

    libport::Lockable sendBufferLock;
    libport::Lockable listLock;

    /// Add a callback to the list.
    UCallbackID addCallback(const char* tag, UCallbackWrapper& w);

    /// Generate a client error message and notify callbacks.
    /// \param message an optional string describing the error.
    ///                The possible prefix "!!! " is skipped if present.
    /// \param code    an optional system error code on which strerror is called
    void clientError(std::string msg, int code = 0);
    void clientError(const char* msg = 0, int code = 0);

    /// Host name.
    std::string host_;
    /// Urbi Port.
    unsigned port_;

    /// Server mode
    bool server_;

    /// Buffer sizes.
    size_t sendBufSize;
    size_t recvBufSize;

    /// System calls return value storage.
    /// 0 iff no error.
    /// Other values are unspecified.
    error_type rc;

    /// Reception buffer.
    char* recvBuffer;
    /// Current position in reception buffer.
    size_t recvBufferPosition;
    /// Temporary buffer for send data.
    char* sendBuffer;

    /// \name Kernel Version.
    /// \{
  public:
    /// Kernel version string.
    /// Call waitForKernelVersion to make sure it is defined (beware
    /// that there are two signatures, one for UAbstractClient,
    /// another for USyncClient).
    const std::string& kernelVersion() const;
    /// Major kernel version.  Dies if unknown yet.
    int kernelMajor() const;
    /// Minor kernel version.  Dies if unknown yet.
    int kernelMinor() const;
    /** Block until kernel version is available or an error occurrs.
     * Message processing must not depend on this thread.
     */
    virtual void waitForKernelVersion() const = 0;

  protected:
    /// The full kernel version as answered by the server.
    /// Empty if not available yet.
    std::string kernelVersion_;
    /// The major version.  -1 if not yet available.
    int kernelMajor_;
    /// The minor version.  -1 if not yet available.
    int kernelMinor_;

    /// A callback, installed by onConnection(), that reads an answer
    /// from the server to know if it's k1 or k2.
    virtual UCallbackAction setVersion(const UMessage& msg);
    /// \}

  public:
    const std::string& connectionID() const;

  protected:
    std::string connectionID_;
    /// A callback, installed by setVersion, that computes
    /// connectionID_.
    virtual UCallbackAction setConnectionID(const UMessage& msg);

  private:
    /// Bin object for this command.
    typedef std::list<urbi::BinaryData> binaries_type;
    binaries_type bins;

    /// Empty bins.
    void bins_clear();

    /// Temporary storage of binary data.
    void* binaryBuffer;
    /// Current position in binaryBuffer.
    size_t binaryBufferPosition;
    /// Size of binaryBuffer.
    size_t binaryBufferLength;

    /// Position of parse in recvBuffer.
    size_t parsePosition;
    /// True if preparsing is in a string.
    bool inString;
    /// Current depth of bracket.
    size_t nBracket;
    /// Start of command, after [ts:tag] header.
    char* currentCommand;

    /// Currently parsing binary
    bool binaryMode;

    /// Parsing a system message
    bool system;

    /// Position of end of header.
    size_t endOfHeaderPosition;
    char currentTag[URBI_MAX_TAG_LENGTH];

    int	currentTimestamp;

  protected:
    /// Client fully created
    bool init_;

  public:
    int getCurrentTimestamp() const;

  private:
    typedef std::list<UCallbackInfo> callbacks_type;
    callbacks_type callbacks_;

    /// A counter, used to generate unique (tag) identifiers.
    unsigned int counter_;

    /// Ourself as a stream.
    /// It looks useless, accorded, but VC++ wants it.
    std::ostream* stream_;

    friend class UClientStreambuf;
  };

  /// Wrapper around a callback function. Use callback() to create them.
  class UCallbackWrapper
  {
  public:
    virtual UCallbackAction operator ()(const UMessage&)=0;
    virtual ~UCallbackWrapper() {}
  };


  ///@{
  /// \internal
  class UCallbackWrapperF
    : public UCallbackWrapper
  {
    UCallback cb;
  public:
    UCallbackWrapperF(UCallback cb) : cb(cb) {}
    virtual UCallbackAction operator ()(const UMessage& msg)
    {
      return cb(msg);
    }
    virtual ~UCallbackWrapperF() {}
  };

  class UCallbackWrapperCF
    : public UCallbackWrapper
  {
    UCustomCallback cb;
    void* cbData;
  public:
    UCallbackWrapperCF(UCustomCallback cb, void* cbData)
      : cb(cb), cbData(cbData)
    {}
    virtual UCallbackAction operator ()(const UMessage& msg)
    {
      return cb(cbData, msg);
    }
    virtual ~UCallbackWrapperCF() {}
  };

  template<class C>
  class UCallbackWrapper0 :
    public UCallbackWrapper
  {
    C& instance;
    UCallbackAction (C::*func)(const UMessage&);
  public:
    UCallbackWrapper0(C& instance, UCallbackAction (C::*func)(const UMessage&))
      : instance(instance), func(func)
    {}
    virtual UCallbackAction operator ()(const UMessage& msg)
    {
      return (instance.*func)(msg);
    }
    virtual ~UCallbackWrapper0() {}
  };

  template<class C, class P1>
  class UCallbackWrapper1
    : public UCallbackWrapper
  {
    C& instance;
    UCallbackAction (C::*funcPtr)(P1, const UMessage&);
    typename libport::traits::remove_reference<P1>::type p1;
  public:
    UCallbackWrapper1(C& instance,
                      UCallbackAction (C::*func)(P1, const UMessage&), P1 p1)
      : instance(instance), funcPtr(func), p1(p1)
    {}
    virtual UCallbackAction operator ()(const UMessage& msg)
    {
      return (instance.*funcPtr)(p1, msg);
    }
    virtual ~UCallbackWrapper1()
    {}
  };

  template<class C, class P1, class P2>
  class UCallbackWrapper2 : public UCallbackWrapper
  {
    C& instance;
    UCallbackAction (C::*func)(P1, P2, const UMessage&);
    typename libport::traits::remove_reference<P1>::type p1;
    typename libport::traits::remove_reference<P2>::type p2;
  public:
    UCallbackWrapper2(
      C& instance, UCallbackAction (C::*func)(P1, P2, const UMessage&),
      P1 p1, P2 p2)
      : instance(instance), func(func), p1(p1), p2(p2)
    {}
    virtual UCallbackAction operator ()(const UMessage& msg)
    {
      return (instance.*func)(p1, p2, msg);
    }
    virtual ~UCallbackWrapper2()
    {}
  };


  template<class C, class P1, class P2, class P3>
  class UCallbackWrapper3 : public UCallbackWrapper
  {
    C& instance;
    UCallbackAction (C::*func)(P1, P2, P3, const UMessage&);
    typename libport::traits::remove_reference<P1>::type p1;
    typename libport::traits::remove_reference<P2>::type p2;
    typename libport::traits::remove_reference<P3>::type p3;
  public:
    UCallbackWrapper3(
      C& instance, UCallbackAction (C::*func)(P1, P2, P3, const UMessage&),
      P1 p1, P2 p2, P3 p3)
      : instance(instance), func(func), p1(p1), p2(p2), p3(p3)
    {}
    virtual UCallbackAction operator ()(const UMessage& msg)
    {
      return (instance.*func)(p1, p2, p3, msg);
    }
    virtual ~UCallbackWrapper3()
    {}
  };


  template<class C, class P1, class P2, class P3, class P4>
  class UCallbackWrapper4 : public UCallbackWrapper
  {
    C& instance;
    UCallbackAction (C::*func)(P1, P2, P3, P4, const UMessage&);
    typename libport::traits::remove_reference<P1>::type p1;
    typename libport::traits::remove_reference<P2>::type p2;
    typename libport::traits::remove_reference<P3>::type p3;
    typename libport::traits::remove_reference<P4>::type p4;
  public:
    UCallbackWrapper4(
      C& instance, UCallbackAction(C::*func)(P1, P2, P3, P4, const UMessage&),
      P1 p1, P2 p2, P3 p3, P4 p4)
      : instance(instance), func(func), p1(p1), p2(p2), p3(p3), p4(p4)
    {}
    virtual UCallbackAction operator ()(const UMessage& msg)
    {
      return (instance.*func)(p1, p2, p3, p4, msg);
    }
    virtual ~UCallbackWrapper4()
    {}
  };
  //@}
  //overloaded callback generators

  //@{
  /// Generate a callback wrapper used by UAbstractClient::setCallback().
  /*!
  You can pass to this function:
    - A function pointer, with signature UCallbackAction(*)(const UMessage&)
      \code setCallback(callback(&myFunction), "tag"); \endcode
    - A reference to a class instance and a pointer to a member function.
      \code setCallback(callback(*this, &MyClass::myMemberFunction)); \endcode
  */
  inline UCallbackWrapper& callback(UCallback cb)
  {
    return *new UCallbackWrapperF(cb);
  }
  inline UCallbackWrapper& callback(UCustomCallback cb, void* data)
  {
    return *new UCallbackWrapperCF(cb, data);
  }


  template<class C> UCallbackWrapper&
  callback(C& ref, UCallbackAction (C::*func)(const UMessage&))
  {
    return *new UCallbackWrapper0<C>(ref, func);
  }

  template<class C, class P1> UCallbackWrapper&
  callback(C& ref, UCallbackAction (C::*func)(P1, const UMessage&), P1 p1)
  {
    return *new UCallbackWrapper1<C, P1>(ref, func, p1);
  }

  template<class C, class P1, class P2>	 UCallbackWrapper&
  callback(C& ref, UCallbackAction (C::*func)(P1, P2, const UMessage&),
	   P1 p1, P2 p2)
  {
    return *new UCallbackWrapper2<C, P1, P2>(ref, func, p1, p2);
  }

  template<class C, class P1, class P2, class P3> UCallbackWrapper&
  callback(C& ref, UCallbackAction (C::*func)(P1, P2, P3, const UMessage&),
	   P1 p1, P2 p2, P3 p3)
  {
    return *new UCallbackWrapper3<C, P1, P2, P3>(ref, func, p1, p2, p3);
  }

  template<class C, class P1, class P2, class P3, class P4>  UCallbackWrapper&
  callback(C& ref, UCallbackAction (C::*func)(P1, P2, P3, P4, const UMessage&),
	   P1 p1, P2 p2, P3 p3, P4 p4)
  {
    return *new UCallbackWrapper4<C, P1, P2, P3, P4>(ref, func, p1, p2, p3, p4);
  }

 //@}


  //old-style addCallback, deprecated
  template<class C>
  UCallbackID
  UAbstractClient::setCallback(C& ref,
                               UCallbackAction (C::*func)(const UMessage&),
                               const char* tag)
  {
    return addCallback(tag,*new UCallbackWrapper0<C>(ref, func));
  }

  template<class C, class P1>
  UCallbackID
  UAbstractClient::setCallback(C& ref,
                               UCallbackAction (C::*func)(P1 , const UMessage&),
                               P1 p1, const char* tag)
  {
    return addCallback(tag, *new UCallbackWrapper1<C, P1>(ref, func, p1));
  }

  template<class C, class P1, class P2>
  UCallbackID
  UAbstractClient::setCallback(
    C& ref, UCallbackAction (C::*func)(P1, P2, const UMessage&),
    P1 p1, P2 p2, const char* tag)
  {
    return addCallback(tag, *new UCallbackWrapper2<C, P1, P2>
                       (ref, func, p1, p2));
  }

  template<class C, class P1, class P2, class P3>
  UCallbackID
  UAbstractClient::setCallback(
    C& ref, UCallbackAction (C::*func)(P1, P2, P3, const UMessage&),
    P1 p1 , P2 p2, P3 p3, const char* tag)
  {
    return addCallback(tag, *new UCallbackWrapper3<C, P1, P2, P3>
                       (ref, func, p1, p2, p3));
  }

  template<class C, class P1, class P2, class P3, class P4>
  UCallbackID
  UAbstractClient::setCallback(
    C& ref, UCallbackAction (C::*func)(P1, P2, P3, P4, const UMessage&),
    P1 p1, P2 p2, P3 p3, P4 p4, const char* tag)
  {
    return addCallback(tag, *new UCallbackWrapper4<C, P1, P2, P3, P4>
                       (ref, func, p1, p2, p3, p4));
  }


  /// Conveniant macro for easy insertion of URBI code in C
  /**
     With this macro, the following code is enough to send a simple
     command to a robot using URBI:

     int main()
     {
       urbi::connect("robot");
       URBI(headPan.val'n = 0 time:1000 |
            headTilt.val'n = 0 time:1000,
            speaker.play("test.wav"),
            echo "test";
          );
     }

     The following construct is also valid:
     URBI(()) << "headPan.val="<<12<<";";
  */

# ifdef URBI
#  undef URBI
# endif

# define URBI(A)				\
  ::urbi::unarmorAndSend(#A)

  static const char semicolon = ';';
  static const char pipe = '|';
  static const char parallel = '&';
  static const char comma = ',';

  /// Must be called at the last line of your main() function.
  URBI_SDK_API
  void execute(void);

  /// Terminate your URBI program.
  URBI_SDK_API
  void exit(int code);

  /// Create a new UClient object
  URBI_SDK_API
  UClient& connect(const std::string& host);

  /// Destroy an UClient object
  /// Be careful: don't use client after called this function
  URBI_SDK_API
  void disconnect(UClient &client);

  /*-----------------.
  | Default client.  |
  `-----------------*/

  /// Return the first UClient created by the program.
  URBI_SDK_API
  UClient* getDefaultClient();

  /// Same as getDefaultClient(), but as a reference.
  URBI_SDK_API
  UClient& get_default_client();

  /// Redefine the default client.
  URBI_SDK_API
  void setDefaultClient(UClient* cl);

  URBI_SDK_API
  std::string getClientConnectionID(const UAbstractClient* cli);

# ifndef DISABLE_IOSTREAM
  /// Send a possibly armored string to the default client
  URBI_SDK_API
  std::ostream&
  unarmorAndSend(const char* str,
                 UAbstractClient* c = (UAbstractClient*)getDefaultClient());
# endif

  URBI_SDK_API
  extern UClient* defaultClient;

  /// Return a stream for error, preferrably the one the defaultClient.
  URBI_SDK_API
  std::ostream& default_stream();

} // namespace urbi

# include <urbi/uabstractclient.hxx>

#endif // URBI_UABSTRACTCLIENT_HH
