/*
 * Copyright (C) 2007, 2008, 2009, 2010, Gostai S.A.S.
 *
 * This software is provided "as is" without warranty of any kind,
 * either expressed or implied, including but not limited to the
 * implied warranties of fitness for a particular purpose.
 *
 * See the LICENSE file for more information.
 */

/// \file urbi/uvar.hxx
#ifndef URBI_UVAR_HXX
# define URBI_UVAR_HXX

# include <stdexcept>
# include <libport/cassert>
# include <urbi/uvar.hh>

namespace urbi
{
  inline
  UVar::operator bool() const
  {
    return static_cast<int>(*this) != 0;
  }

  inline
  UVar::UVar()
    : owned(false)
    , VAR_PROP_INIT
    , impl_(0)
    , vardata(0)
    , name("noname")
  {
  }

  inline
  void
  UVar::init(const std::string& objname, const std::string& varname,
             impl::UContextImpl* ctx)
  {
    init(objname + '.' + varname, ctx);
  }


  inline
  void
  UVar::setOwned()
  {
    check_();
    impl_->setOwned();
  }

  inline
  UDataType
  UVar::type() const
  {
    check_();
    return impl_->type();
  }

  inline
  void
  UVar::syncValue()
  {
    check_();
    impl_->sync();
  }

  inline const UValue&
  UVar::val() const
  {
    check_();
    return impl_->get();
  }

  inline
  void
  UVar::keepSynchronized()
  {
    check_();
    impl_->keepSynchronized();
  }

  inline
  UVar&
  UVar::operator=(const UValue& v)
  {
    check_();
    impl_->set(v);
    return *this;
  }

# define SET(Type)                         \
  inline UVar& UVar::operator=(Type tv)    \
  {                                        \
    /*no need to copy, impl will do it*/   \
    check_();                              \
    UValue v(tv, false);                   \
    impl_->set(v);                         \
    return *this;                          \
  }

  SET(const UBinary&)
  SET(const UImage&)
  SET(const UList&)
  SET(const UDictionary&)
  SET(const USound&)
  SET(const std::string&)
  SET(ufloat)
# undef SET

# define GET(Type)                    \
  inline UVar::operator Type() const  \
  {                                   \
    check_();                         \
    return impl_->get();              \
  }

  GET(UImage)
  GET(UList)
  GET(UDictionary)
  GET(USound)
  GET(const UBinary&)
  GET(int)
  GET(std::string)
  GET(ufloat)
# undef GET

  inline UVar::operator UBinary*() const
  {
    return new UBinary(static_cast<const UBinary&>(*this));
  }

  inline void UVar::setProp(UProperty prop, const UValue &v)
  {
    check_();
    impl_->setProp(prop, v);
  }

  inline void UVar::setProp(UProperty prop, const char* v)
  {
    check_();
    UValue tv(v);
    impl_->setProp(prop, tv);
  }

  inline void UVar::setProp(UProperty prop, ufloat v)
  {
    check_();
    UValue tv(v);
    impl_->setProp(prop, tv);
  }

  inline bool
  UVar::setBypass(bool enable)
  {
    check_();
    return impl_->setBypass(enable);
  }

  inline UValue
  UVar::getProp(UProperty prop)
  {
    check_();
    return impl_->getProp(prop);
  }

  inline void
  UVar::unnotify()
  {
    check_();
    impl_->unnotify();
  }

  inline void
  UVar::check_() const
  {
    if (!impl_)
      throw std::runtime_error("Invalid use of unbound UVar");
  }
} // end namespace urbi

#endif // ! URBI_UVAR_HXX
