
/*
//
//                 INTEL CORPORATION PROPRIETARY INFORMATION
//    This software is supplied under the terms of a license agreement or
//   nondisclosure agreement with Intel Corporation and may not be copied
//    or disclosed except in accordance with the terms of that agreement.
//        Copyright (c) 1998 Intel Corporation. All Rights Reserved.
//
//  Project:
//    Intel JPEG Library
//
//  Purpose:
//    Sample application
//
//  Author(s):
//    Vladimir V. Dudnik
//
*/

#include "stdafx.h"

#include <dlgs.h>     // common dialog symbols
#include "JPGView.rh" // main symbols

#ifndef __FILEOPENDLG_H__
#include "FileOpenDlg.h"
#endif
#ifndef __IJL_H__
#include "ijl.h"
#endif


#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif




/////////////////////////////////////////////////////////////////////////////
// CFileDialogEx

IMPLEMENT_DYNAMIC(CFileDialogEx, CFileDialog)


BEGIN_MESSAGE_MAP(CFileDialogEx, CFileDialog)
  //{{AFX_MSG_MAP(CFileDialogEx)
  ON_BN_CLICKED(IDC_PREVIEW_CHECK,OnClickedPreview)
  ON_WM_PAINT()
  //}}AFX_MSG_MAP
END_MESSAGE_MAP()


CFileDialogEx::CFileDialogEx(
  BOOL      bOpenFileDialog,
  LPCTSTR   lpszDefExt,
  LPCTSTR   lpszFileName,
  DWORD     dwFlags,
  LPCTSTR   lpszFilter,
  CWnd*     pParentWnd) :

  CFileDialog(
    bOpenFileDialog,
    lpszDefExt,
    lpszFileName,
    dwFlags,
    lpszFilter,
    pParentWnd)
{
  //{{AFX_DATA_INIT(CFileDialog)
  m_bChanged    = FALSE;
  m_bPreview    = TRUE;
  m_comments    = "";
  m_color       = "";
  m_subsampling = "";
  m_size        = "";
  m_imageData   = NULL;
  //}}AFX_DATA_INIT

  return;
} // CFileDialogEx ctor


CFileDialogEx::~CFileDialogEx(void)
{
  Clean();
  m_memDC.DeleteDC();
  return;
} // dtor

void CFileDialogEx::DoDataExchange(CDataExchange* pDX)
{
  CDialog::DoDataExchange(pDX);

  //{{AFX_DATA_MAP(CFileDialogEx)
  DDX_Check(pDX,IDC_PREVIEW_CHECK,m_bPreview);
  DDX_Text(pDX,IDC_JPEG_COMMENT,m_comments);
  DDX_Text(pDX,IDC_JPEG_COLOR,m_color);
  DDX_Text(pDX,IDC_JPEG_SS,m_subsampling);
  DDX_Text(pDX,IDC_JPEG_SIZE,m_size);
  //}}AFX_DATA_MAP

  return;
} // CFileDialogEx::DoDataExchange()


BOOL CFileDialogEx::OnInitDialog(void)
{
  int   height;
  int   width;
  CDC*  previewDC  = NULL;
  CWnd* previewWnd = NULL;

  CFileDialog::OnInitDialog();

  previewWnd = GetDlgItem(IDC_PREVIEW);

  if(NULL == previewWnd)
  {
    goto Exit;
  }

  previewDC = previewWnd->GetWindowDC();

  if(NULL == previewDC)
  {
    goto Exit;
  }

  previewWnd->GetWindowRect(PreviewRect);

  width = PreviewRect.Width();
  height = PreviewRect.Height();

  m_bkbrush.CreateStockObject(LTGRAY_BRUSH);

  m_memDC.CreateCompatibleDC(NULL);

  m_memDC.SelectObject(m_bkbrush);
  m_memDC.PatBlt(0,0,width,height,PATCOPY);

  m_bitmap.CreateCompatibleBitmap(previewDC,width,height);
  m_memDC.SelectObject(m_bitmap);

Exit:

  UpdateData(FALSE);

  if(NULL != previewDC)
  {
    ReleaseDC(previewDC);
  }

  return TRUE;
} // CFileDialogEx::OnInitDialog()


BOOL CFileDialogEx::OnNotify(
  WPARAM   wParam,
  LPARAM   lParam,
  LRESULT* pResult)
{
  ASSERT(pResult != NULL);

  // allow message map to override

  OFNOTIFY* pNotify = (OFNOTIFY*)lParam;

  switch(pNotify->hdr.code)
  {
  case CDN_SELCHANGE:
    OnFileNameChange();
    return TRUE;

  case CDN_FOLDERCHANGE:
    OnFolderChange();
    return TRUE;
  }

  return CFileDialog::OnNotify(wParam, lParam, pResult);
} // CFileDialogEx::OnNotify()


void CFileDialogEx::OnFolderChange(void)
{
  m_bChanged = TRUE;
  m_comments    = "";
  m_color       = "";
  m_subsampling = "";
  m_size        = "";

  UpdateData(FALSE);

  int width = PreviewRect.Width();
  int height = PreviewRect.Height();

//  CBrush m_bkbrush;
//  m_bkbrush.CreateStockObject(LTGRAY_BRUSH);
  m_memDC.SelectObject(m_bkbrush);
  m_memDC.PatBlt(2,2,width-4,height-4,PATCOPY);

  SetPreviewFile(CString(""));
  InvalidateRect(&PreviewRect);

  return;
} // CFileDialogEx::OnFolderChange()


void CFileDialogEx::OnFileNameChange(void)
{
  RedrawWindow();
  m_comments    = "";
  m_color       = "";
  m_subsampling = "";
  m_size        = "";

  Clean();
  UpdateData(FALSE);

  int width = PreviewRect.Width();
  int height = PreviewRect.Height();

  m_memDC.SelectObject(m_bkbrush);
  m_memDC.PatBlt(2,2,width-4,height-4,PATCOPY);

  if(!m_bChanged && m_bPreview)
  {
    SetPreviewFile(GetPathName());

    if(NULL != m_imageData)
    {
      HBITMAP hBmp;
      LPBITMAPINFO pbmi = (LPBITMAPINFO)m_imageData;
      LPBITMAPINFOHEADER pbmih = (LPBITMAPINFOHEADER)m_imageData;
      BYTE* bits = GetImageBits();
      hBmp = CreateDIBitmap(m_memDC,pbmih,CBM_INIT,bits,pbmi,DIB_RGB_COLORS);
      m_memDC.SelectObject(hBmp);
      DeleteObject(hBmp);

      DrawPreview();
    }
  }

  m_bChanged = FALSE;

  UpdateData(FALSE);

  return;
} // CFileDialogEx::OnFileNameChange()


void CFileDialogEx::OnClickedPreview(void)
{
  UpdateData(TRUE);

  if(FALSE != m_bPreview)
  {
    InvalidateRect(&PreviewRect);
  }

  return;
} // CFileDialogEx::OnClickedPreview()


void CFileDialogEx::OnPaint(void)
{
  CFileDialog::OnPaint();

  if(m_bPreview)
  {
    if(m_imageData)
    {
      DrawPreview();
    }
  }

  return;
} // CFileDialogEx::OnPaint()


void CFileDialogEx::SetPreviewFile(const CString& csFileName)
{
  CString ext3;
  CString ext4;
  BOOL    bIsJPEG = FALSE;

  Clean();

  ext3 = csFileName.Right(3);
  ext4 = csFileName.Right(4);

  if(0 == ext3.CompareNoCase("JPG"))
  {
    bIsJPEG = TRUE;
  }

  if(0 == ext3.CompareNoCase("JPE"))
  {
    bIsJPEG = TRUE;
  }

  if(0 == ext4.CompareNoCase("JPEG"))
  {
    bIsJPEG = TRUE;
  }

  if(FALSE != bIsJPEG)
  {
    GetImageFromJPG(csFileName);
  }

  return;
} // CFileDialogEx::SetPreviewFile()


BOOL CFileDialogEx::GetImageFromJPG(const CString& csFileName)
{
  int                  pad_bytes;
  int                  dib_line_size;
  int                  size;
  BOOL                 bres = TRUE;
  char                 comments[1024];
  IJLERR               jerr;
  JPEG_CORE_PROPERTIES jcprops;

  comments[0] = 0;

  jerr = ijlInit(&jcprops);

  if(IJL_OK != jerr)
  {
    bres = FALSE;
    goto Exit;
  }

  TRACE("File %s\n",csFileName);

  jcprops.JPGFile = csFileName;

  jcprops.jprops.jpeg_comment      = &comments[0];
  jcprops.jprops.jpeg_comment_size = sizeof(comments);

  jerr = ijlRead(&jcprops,IJL_JFILE_READPARAMS);

  if(IJL_OK != jerr && IJL_ERR_COM_BUFFER != jerr)
  {
    TRACE("ijlRead() failed\n");
    m_size.Format("%s",ijlErrorStr(jerr));
    bres = FALSE;
    goto Exit;
  }

  m_color = GetColorStr(jcprops.JPGColor);
  m_subsampling = GetSubsamplingStr(jcprops.JPGSubsampling);

  if(jcprops.JPGThumbWidth != 0 && jcprops.JPGThumbHeight != 0)
  {
    TRACE("File contains embedded thumbnail, size %d X %d\n",
      jcprops.JPGThumbWidth,jcprops.JPGThumbHeight);

    jcprops.DIBWidth  =  jcprops.JPGThumbWidth;
    jcprops.DIBHeight = -jcprops.JPGThumbHeight;

    pad_bytes = IJL_DIB_PAD_BYTES(jcprops.DIBWidth,3);
    dib_line_size = jcprops.DIBWidth * jcprops.DIBChannels + pad_bytes;
    size = dib_line_size * abs(jcprops.DIBHeight) + sizeof(BITMAPINFO);

    m_imageData = new BYTE [size];

    if(NULL == m_imageData)
    {
      bres = FALSE;
      goto Exit;
    }

    TRACE("allocate %d bytes at 0x%08X for thumbnail image\n",
      size,m_imageData);

    BYTE* buffer = m_imageData + sizeof(BITMAPINFO);

    jcprops.DIBBytes    = buffer;
    jcprops.DIBPadBytes = pad_bytes;

    jerr = ijlRead(&jcprops,IJL_JFILE_READTHUMBNAIL);

    if(IJL_OK != jerr)
    {
      TRACE("ijlRead() failed\n");
      m_size.Format("%s",ijlErrorStr(jerr));
      bres = FALSE;
      goto Exit;
    }

    BITMAPINFO* bmi = (BITMAPINFO*)m_imageData;

    bmi->bmiHeader.biSize          = sizeof(BITMAPINFOHEADER);
    bmi->bmiHeader.biWidth         = jcprops.DIBWidth;
    bmi->bmiHeader.biHeight        = abs(jcprops.DIBHeight);
    bmi->bmiHeader.biCompression   = BI_RGB;
    bmi->bmiHeader.biBitCount      = (jcprops.DIBChannels == 3) ? (WORD)24 : (WORD)32;
    bmi->bmiHeader.biPlanes        = 1;
    bmi->bmiHeader.biClrUsed       = 0;
    bmi->bmiHeader.biClrImportant  = 0;
    bmi->bmiHeader.biSizeImage     = 0;
    bmi->bmiHeader.biXPelsPerMeter = 0;
    bmi->bmiHeader.biYPelsPerMeter = 0;
    bmi->bmiColors[0].rgbRed       = 0;
    bmi->bmiColors[0].rgbGreen     = 0;
    bmi->bmiColors[0].rgbBlue      = 0;
    bmi->bmiColors[0].rgbReserved  = 0;
  }
  else
  {
    TRACE("File have not embedded thumbnail\n");
    TRACE("IJL will do scaled decoding\n");

    jcprops.DIBWidth  =  IJL_DIB_SCALE_SIZE(jcprops.JPGWidth,8);
    jcprops.DIBHeight = -IJL_DIB_SCALE_SIZE(jcprops.JPGHeight,8);

    TRACE("size %d X %d\n",jcprops.DIBWidth,jcprops.DIBHeight);

    switch(jcprops.JPGChannels)
    {
    case 1:
      jcprops.DIBChannels = 3;
      jcprops.DIBColor    = IJL_BGR;
      break;

    case 3:
      jcprops.DIBChannels = 3;
      jcprops.DIBColor    = IJL_BGR;
      break;

    case 4:
      jcprops.DIBChannels = jcprops.JPGChannels;
      jcprops.DIBColor    = IJL_RGBA_FPX;
      break;
    }

    pad_bytes = IJL_DIB_PAD_BYTES(jcprops.DIBWidth,jcprops.DIBChannels);
    dib_line_size = jcprops.DIBWidth * jcprops.DIBChannels + pad_bytes;
    size = dib_line_size * abs(jcprops.DIBHeight) + sizeof(BITMAPINFO);

    m_imageData = new BYTE [size];

    if(NULL == m_imageData)
    {
      bres = FALSE;
      goto Exit;
    }

    TRACE("allocate %d bytes at 0x%08X for thumbnail image\n",
      size,m_imageData);

    BYTE* buffer = m_imageData + sizeof(BITMAPINFO);

    jcprops.DIBBytes    = buffer;
    jcprops.DIBPadBytes = pad_bytes;

    jerr = ijlRead(&jcprops,IJL_JFILE_READONEEIGHTH);

    if(IJL_OK != jerr)
    {
      TRACE("ijlRead() failed\n");
      m_size.Format("%s",ijlErrorStr(jerr));
      bres = FALSE;
      goto Exit;
    }

    BITMAPINFO* bmi = (BITMAPINFO*)m_imageData;

    bmi->bmiHeader.biSize          = sizeof(BITMAPINFOHEADER);
    bmi->bmiHeader.biWidth         = jcprops.DIBWidth;
    bmi->bmiHeader.biHeight        = abs(jcprops.DIBHeight);
    bmi->bmiHeader.biCompression   = BI_RGB;
    bmi->bmiHeader.biBitCount      = (jcprops.DIBChannels == 3) ? (WORD)24 : (WORD)32;
    bmi->bmiHeader.biPlanes        = 1;
    bmi->bmiHeader.biClrUsed       = 0;
    bmi->bmiHeader.biClrImportant  = 0;
    bmi->bmiHeader.biSizeImage     = 0;
    bmi->bmiHeader.biXPelsPerMeter = 0;
    bmi->bmiHeader.biYPelsPerMeter = 0;
    bmi->bmiColors[0].rgbRed       = 0;
    bmi->bmiColors[0].rgbGreen     = 0;
    bmi->bmiColors[0].rgbBlue      = 0;
    bmi->bmiColors[0].rgbReserved  = 0;
  }

  m_size.Format("%d X %d X %d",jcprops.JPGWidth,jcprops.JPGHeight,jcprops.JPGChannels);

  if(comments[0] != 0)
  {
    int tail = min(sizeof(comments),jcprops.jprops.jpeg_comment_size);
    comments[tail] = 0;
    m_comments.Format("%s",&comments[0]);
  }

Exit:

  ijlFree(&jcprops);

  if(FALSE == bres)
  {
    if(NULL != m_imageData)
    {
      TRACE("free memory at 0x%08X for thumbnail image\n",m_imageData);
      delete[] m_imageData;
      m_imageData = NULL;
    }
  }

  return bres;
} // CFileDialogEx::GetImageFromJPG()


void CFileDialogEx::Clean(void)
{
  if(NULL != m_imageData)
  {
    TRACE("free memory at 0x%08X for thumbnail image\n",m_imageData);
    delete[] m_imageData;
    m_imageData = NULL;
  }
  return;
} // CFileDialogEx::Clean()


CString CFileDialogEx::GetColorStr(IJL_COLOR color)
{
  CString str;

  switch(color)
  {
  case IJL_G:          str = "Gray";    break;
  case IJL_RGB:        str = "RGB";     break;
  case IJL_YCBCR:      str = "YCbCr";   break;
  case IJL_RGBA_FPX:   str = "CMYK";    break;
  case IJL_YCBCRA_FPX: str = "YCCK";    break;
  default:             str = "Unknown"; break;
  }

  return str;
} // CFileDialogEx::GetColorStr()


CString CFileDialogEx::GetSubsamplingStr(IJL_JPGSUBSAMPLING ss)
{
  CString str;

  switch(ss)
  {
  case IJL_NONE: str = "None";  break;
  case IJL_422:  str = "422";   break;
  case IJL_411:  str = "411";   break;
  case IJL_4224: str = "4224";  break;
  case IJL_4114: str = "4114";  break;
  default:       str = "Other"; break;
  }

  return str;
} // CFileDialogEx::GetColorStr()


int CFileDialogEx::GetImageWidth(void)
{
  LPBITMAPINFOHEADER bmih = (LPBITMAPINFOHEADER)m_imageData;
  return bmih == NULL ? 0 : bmih->biWidth;
} // CFileDialogEx::GetImageWidth()


int CFileDialogEx::GetImageHeight(void)
{
  LPBITMAPINFOHEADER bmih = (LPBITMAPINFOHEADER)m_imageData;
  return bmih == NULL ? 0 : abs(bmih->biHeight);
} // CFileDialogEx::GetImageHeight()


int CFileDialogEx::GetImageWidthBytes(void)
{
  LPBITMAPINFOHEADER bmih = (LPBITMAPINFOHEADER)m_imageData;
  return bmih == NULL ? 0 : bmih->biWidth * 3;
} // CFileDialogEx::GetImageWidthBytes()


BYTE* CFileDialogEx::GetImageBits(void)
{
  BYTE* bits = m_imageData + sizeof(BITMAPINFO);
  return bits;
} // CFileDialogEx::GetImageBits()


void CFileDialogEx::DrawPreview(void)
{
  int   width;
  int   height;
  int   img_width;
  int   img_height;
  int   x1, y1, dx, dy;
  CDC*  previewDC  = NULL;
  CWnd* previewWnd = NULL;

  previewWnd = GetDlgItem(IDC_PREVIEW);

  if(NULL == previewWnd)
  {
    goto Exit;
  }

  previewDC = previewWnd->GetWindowDC();

  if(NULL == previewDC)
  {
    goto Exit;
  }

  width = PreviewRect.Width();
  height = PreviewRect.Height();

  img_width = GetImageWidth();
  img_height = GetImageHeight();

  x1 = img_width >= width ? 2 : (width - img_width) / 2;
  y1 = img_height >= height ? 2 : (height - img_height) / 2;

  dx = img_width >= width ? 4 : 0;
  dy = img_height >= height ? 4 : 0;

  previewDC->SelectObject(m_bkbrush);
  previewDC->PatBlt(2,2,width-4,height-4,PATCOPY);

  previewDC->BitBlt(x1,y1,width - dx,height - dy,&m_memDC,0,0,SRCCOPY);

Exit:

  if(NULL != previewDC)
  {
    ReleaseDC(previewDC);
  }

  return;
} // CFileDialogEx::DrawPreview()


///////////////////////////////////////////////////////////////////////////////
//                       End of file 'FileOpenDlg.cpp'                       //
///////////////////////////////////////////////////////////////////////////////
