                             // JPGViewView.cpp //
///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//                 INTEL CORPORATION PROPRIETARY INFORMATION                 //
//    This software is supplied under the terms of a license agreement or    //
//   nondisclosure agreement with Intel Corporation and may not be copied    //
//    or disclosed except in accordance with the terms of that agreement.    //
//        Copyright (c) 1998 Intel Corporation. All Rights Reserved.         //
//                                                                           //
//  Project:                                                                 //
//    Intel JPEG Library                                                     //
//  Purpose:                                                                 //
//    Sample application                                                     //
//  Author(s):                                                               //
//    Paul V. Pervov                                                         //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////
#include "stdafx.h"

#ifndef __JPGVIEW_H__
#include "JPGView.h"
#endif
#include "JPGView.rh"

#ifndef __JPGVIEWDOC_H__
#include "JPGViewDoc.h"
#endif
#ifndef __JPGVIEWVIEW_H__
#include "JPGViewView.h"
#endif
#ifndef __IJL_H__
#include "ijl.h"
#endif
#ifndef __FOLDERBROWSER_H__
#include "FolderBrowser.h"
#endif

#include <math.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define INT2bool( a )   (a != 0)

///////////////////////////////////////////////////////////////////////////////
//                              CImageInfoDlg dialog                         //
///////////////////////////////////////////////////////////////////////////////

CImageInfoDlg::CImageInfoDlg(CWnd* pParent /*=NULL*/)
: CDialog(CImageInfoDlg::IDD, pParent)
{
    //{{AFX_DATA_INIT(CImageInfoDlg)
    //}}AFX_DATA_INIT
}


void CImageInfoDlg::DoDataExchange(CDataExchange* pDX)
{
    CDialog::DoDataExchange(pDX);
    if( pDX->m_bSaveAndValidate ) return;
    //{{AFX_DATA_MAP(CImageInfoDlg)
    DDX_Text( pDX, IDC_IMAGEWIDTH, m_width );
    DDX_Text( pDX, IDC_IMAGEHEIGHT, m_height );
    DDX_Text( pDX, IDC_IMAGECOLORS, m_colors );
    //}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CImageInfoDlg, CDialog)
    //{{AFX_MSG_MAP(CImageInfoDlg)
    //}}AFX_MSG_MAP
END_MESSAGE_MAP()

// CImageInfoDlg message handlers


///////////////////////////////////////////////////////////////////////////////
//                            COptionsDlg dialog                             //
///////////////////////////////////////////////////////////////////////////////

COptionsDlg::COptionsDlg(CWnd* pParent /*=NULL*/)
    : CDialog(COptionsDlg::IDD, pParent)
{
    //{{AFX_DATA_INIT(COptionsDlg)
    m_bCenterToPrev = FALSE;
    m_bResizeWithImage = FALSE;
    m_bScaleToFit = FALSE;
    m_bCenterInWindow = FALSE;
    m_bCenterOnPage = FALSE;
    //}}AFX_DATA_INIT
}


void COptionsDlg::DoDataExchange(CDataExchange* pDX)
{
    CDialog::DoDataExchange(pDX);
    //{{AFX_DATA_MAP(COptionsDlg)
    DDX_Check( pDX, IDC_RESIZEWITHIMAGE, m_bResizeWithImage );
    DDX_Check( pDX, IDC_CENTERTOPREV, m_bCenterToPrev );
    DDX_Check( pDX, IDC_CENTERINWINDOW, m_bCenterInWindow );
    DDX_Check( pDX, IDC_SCALETOFIT, m_bScaleToFit );
    DDX_Check( pDX, IDC_CENTERONPAGE, m_bCenterOnPage );
    //}}AFX_DATA_MAP

    CWnd* item = GetDlgItem( IDC_CENTERTOPREV );
    ASSERT_VALID( item );
    item->EnableWindow( m_bResizeWithImage );

    item = GetDlgItem( IDC_CENTERINWINDOW );
    ASSERT_VALID( item );
    item->EnableWindow( !m_bResizeWithImage );

    item = GetDlgItem( IDC_CENTERONPAGE );
    ASSERT_VALID( item );
    item->EnableWindow( !m_bScaleToFit );
}


BEGIN_MESSAGE_MAP(COptionsDlg, CDialog)
    //{{AFX_MSG_MAP(COptionsDlg)
    ON_BN_CLICKED(IDC_RESIZEWITHIMAGE, OnResizeWithImage)
    ON_BN_CLICKED(IDC_SCALETOFIT, OnScaleToFit)
    //}}AFX_MSG_MAP
END_MESSAGE_MAP()

// COptionsDlg message handlers

void COptionsDlg::OnResizeWithImage()
{
    UpdateData( TRUE );
}


void COptionsDlg::OnScaleToFit()
{
    UpdateData( TRUE );
}


///////////////////////////////////////////////////////////////////////////////
//                               CJPGViewView                                //
///////////////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNCREATE( CJPGViewView, CScrollView )

BEGIN_MESSAGE_MAP( CJPGViewView, CScrollView )
    //{{AFX_MSG_MAP(CJPGViewView)
    ON_COMMAND(ID_FILE_DIRECTORY, OnFileSelectDirectory)
    ON_COMMAND(ID_VIEW_NEXT, OnViewNextImage)
    ON_COMMAND(ID_VIEW_PREV, OnViewPrevImage)
    ON_COMMAND(ID_VIEW_IMAGEINFO, OnViewImageInfo)
    ON_WM_CREATE()
    ON_WM_DESTROY()
    ON_COMMAND(ID_VIEW_OPTIONS, OnViewOptions)
    ON_UPDATE_COMMAND_UI(ID_VIEW_NEXT, OnUpdateViewNext)
    ON_UPDATE_COMMAND_UI(ID_VIEW_PREV, OnUpdateViewPrev)
    //}}AFX_MSG_MAP
    ON_COMMAND(ID_FILE_SAVE_AS, CJPGViewView::OnSaveAs)
    ON_COMMAND(ID_FILE_PRINT, CScrollView::OnFilePrint)
    ON_COMMAND(ID_FILE_PRINT_DIRECT, CScrollView::OnFilePrint)
    ON_COMMAND(ID_FILE_PRINT_PREVIEW, CScrollView::OnFilePrintPreview)
END_MESSAGE_MAP()


CJPGViewView::CJPGViewView()
{
    ZeroMemory( &m_bmi, sizeof( BITMAPINFO ) );
    m_bmp = NULL;
    m_bResizeWithImage = false;
    m_bCenterToPrev = false;
    m_bCenterInWindow = false;
    m_bScaleToFit = false;

    m_imageIndex = 0;
    m_bViewFolder = false;
    m_directory = _T("");
}


CJPGViewView::~CJPGViewView()
{
    if( m_bmp != NULL ) delete[] m_bmp;
    ZeroMemory( &m_bmi, sizeof( BITMAPINFO ) );
    m_bmp = NULL;
}


BOOL CJPGViewView::PreCreateWindow( CREATESTRUCT& cs )
{
    cs.style |= CS_BYTEALIGNCLIENT;

    return CView::PreCreateWindow(cs);
}


void CJPGViewView::OnDraw( CDC* pDC )
{
    if( m_bmp == NULL ) return;

    CJPGViewDoc* pDoc = GetDocument();
    ASSERT_VALID( pDoc );

    SIZE sizes = pDoc->GetImageDims();

    CRect rectClient;
    GetClientRect( &rectClient );

    CRect rectTo;
    rectTo.left = rectTo.top = 0;

    if( m_bCenterInWindow && !m_bResizeWithImage )
    {
        rectTo.left = rectClient.Width() >> 1;
        rectTo.left -= sizes.cx >> 1;
        if( rectTo.left < 0 ) rectTo.left = 0;

        rectTo.top = rectClient.Height() >> 1;
        rectTo.top -= sizes.cy >> 1;
        if( rectTo.top < 0 ) rectTo.top = 0;
    }

    rectTo.right = rectTo.left + sizes.cx;
    rectTo.bottom = rectTo.top + sizes.cy;

    TRACE( "Client rectangle: %d %d\nFinal rectangle: %d %d %d %d w=%d h=%d\n",
            rectClient.Width(), rectClient.Height(),
            rectTo.left, rectTo.top, rectTo.right, rectTo.bottom,
            rectTo.Width(), rectTo.Height() );

    if( pDC->IsPrinting() )
    {
        int cxPage = pDC->GetDeviceCaps( HORZRES );
        int cyPage = pDC->GetDeviceCaps( VERTRES );

        int cxInch = pDC->GetDeviceCaps( LOGPIXELSX );
        int cyInch = pDC->GetDeviceCaps( LOGPIXELSY );

        if( m_bScaleToFit )
        {
            rectTo.left = rectTo.top = 0;
            rectTo.right = cxPage;
            rectTo.bottom = (int)( ((double)sizes.cy * cxPage * cyInch) /
                                   ((double)sizes.cx * cxInch) );
        }
        else
        {
            CDC dc;
            BOOL bResult = dc.CreateCompatibleDC( NULL );
            if( bResult )
            {
                int cxDisplay = dc.GetDeviceCaps( LOGPIXELSX );
                int cyDisplay = dc.GetDeviceCaps( LOGPIXELSY );
                dc.DeleteDC();

                rectTo.right = rectTo.left +
                               (int)((double)rectTo.Width()*cxInch/cxDisplay);
                rectTo.bottom = rectTo.top +
                                (int)((double)rectTo.Height()*cyInch/cyDisplay);

                if( m_bCenterOnPage )
                {
                    int prtcx = (int)((double)sizes.cx*cxInch/cxDisplay);
                    int prtcy = (int)((double)sizes.cy*cyInch/cyDisplay);

                    rectTo.left = cxPage>>1;
                    rectTo.left -= prtcx>>1;

                    rectTo.top = cyPage>>1;
                    rectTo.top -= prtcy>>1;

                    rectTo.right = rectTo.left + prtcx;
                    rectTo.bottom = rectTo.top + prtcy;
                }
            }
        }
    }

    ::StretchDIBits( pDC->m_hDC,
                     rectTo.left, rectTo.top, rectTo.Width(), rectTo.Height(),
                     0, 0, sizes.cx, sizes.cy,
                     m_bmp, &m_bmi, DIB_RGB_COLORS, SRCCOPY );

} // CJPGViewView::OnDraw


// byte offsets
#define BO_BLUE     0
#define BO_GREEN    1
#define BO_RED      2

void CJPGViewView::OnInitialUpdate()
{
    CScrollView::OnInitialUpdate();

    CJPGViewDoc* pDoc = GetDocument();
    ASSERT_VALID( pDoc );

    BYTE* data      = pDoc->GetImageData();
    SIZE  sizes     = pDoc->GetImageDims();
    int   nChannels = pDoc->GetImageChannels();

    // Hide scroll bars
    SetScrollSizes( MM_TEXT, CSize( 1, 1 ) );

    // If no document - quit
    if( pDoc->IsEmpty() )
      return;

    if( m_bResizeWithImage )
    {
        ResizeImageWindow();
    }

    SetScrollSizes( MM_TEXT, sizes );

    BITMAPINFOHEADER& bih = m_bmi.bmiHeader;

    ::ZeroMemory( &bih, sizeof( BITMAPINFOHEADER ) );

    bih.biSize        = sizeof( BITMAPINFOHEADER );
    bih.biWidth       = sizes.cx;
    bih.biHeight      = -sizes.cy;
    bih.biCompression = BI_RGB;
    bih.biPlanes      = 1;

    switch(nChannels)
    {
    case 3:
      bih.biBitCount = 24;
      break;

    case 4:
      bih.biBitCount = 32;
      break;

    default:
      TRACE("Unsupported number of channels!\n");
      break;
    }

    if( m_bmp != NULL )
      delete[] m_bmp;

    int pad = IJL_DIB_PAD_BYTES(sizes.cx,nChannels);

    int imageSize = (sizes.cx * nChannels + pad) * sizes.cy;

    m_bmp = new BYTE[ imageSize ];

    if( m_bmp == NULL )
      return;

    ::ZeroMemory(m_bmp,imageSize);
    ::CopyMemory(m_bmp,data,imageSize);

    return;
}


BOOL CJPGViewView::OnPreparePrinting( CPrintInfo* pInfo )
{
    return DoPreparePrinting( pInfo );
}


void CJPGViewView::OnBeginPrinting( CDC* /*pDC*/, CPrintInfo* /*pInfo*/ )
{
}


void CJPGViewView::OnEndPrinting( CDC* /*pDC*/, CPrintInfo* /*pInfo*/ )
{
}


#ifdef _DEBUG
void CJPGViewView::AssertValid() const
{
    CView::AssertValid();
}


void CJPGViewView::Dump( CDumpContext& dc ) const
{
    CView::Dump(dc);
}


CJPGViewDoc* CJPGViewView::GetDocument()
{
    ASSERT( m_pDocument->IsKindOf( RUNTIME_CLASS( CJPGViewDoc ) ) );
    return (CJPGViewDoc*)m_pDocument;
}
#endif //_DEBUG


void CJPGViewView::OnFileSelectDirectory()
{
    CFolderBrowser browser( this );

    if( browser.Browse( _T("*.JPG") ) )
    {
        m_directory = browser.GetDirectory();

        CreateImageList();
        m_imageIndex = 0;
        m_bViewFolder = true;
    }
}


int ImagePathCompare( LPCVOID item1, LPCVOID item2 )
{
    CString* string1 = reinterpret_cast<CString*>( const_cast<LPVOID>(item1) );
    CString* string2 = reinterpret_cast<CString*>( const_cast<LPVOID>(item2) );

    ASSERT_POINTER( string1, CString );
    ASSERT_POINTER( string2, CString );

    return string1->Compare( *string2 );
}


void CJPGViewView::CreateImageList()
{
    WIN32_FIND_DATA find;
    HANDLE hSearch;

    m_list.RemoveAll();

    int pos = m_directory.ReverseFind( _T('\\') );
    if( pos != m_directory.GetLength()-1 )
    {
        m_directory += _T('\\');
    }
    CString path = m_directory;
    m_directory += _T( "*.jpg" );

    hSearch = FindFirstFile( m_directory, &find );
    if( hSearch == INVALID_HANDLE_VALUE )
    {
        AfxMessageBox( "No JPEG images found in specified folder" );
        return;
    }
    CString ref = path + find.cFileName;
    m_list.Add( ref );
    while( FindNextFile( hSearch, &find ) )
    {
        ref = path + find.cFileName;
        m_list.Add( ref );
    }
    qsort( m_list.GetData(), m_list.GetSize(), sizeof( CString ),
        ImagePathCompare );

    FindClose( hSearch );

    m_bViewFolder = true;
    m_imageIndex = -1;
    OpenValidDocument( m_list[0], true );
}


void CJPGViewView::ResizeImageWindow()
{
    CWnd* pWndParent = GetParent();
    WINDOWPLACEMENT wndPlace;
    if( !pWndParent->GetWindowPlacement( &wndPlace ) ) return;
    if( wndPlace.showCmd == SW_SHOWMAXIMIZED ) return;

    CJPGViewDoc* pDoc = GetDocument();
    ASSERT_VALID( pDoc );

    if( pDoc->IsEmpty() ) return;

    SIZE sizes = GetDocument()->GetImageDims();

    SetScrollSizes( MM_TEXT, CSize( 1, 1 ) );

    CRect rectCur;
    CRect rectClient;

    pWndParent->GetWindowRect( &rectCur );
    GetClientRect( &rectClient );

    if( !m_bCenterToPrev ) {
        rectCur.right += sizes.cx - rectClient.Width();
        rectCur.bottom += sizes.cy - rectClient.Height();
    } else {
        int dx2 = (sizes.cx - rectClient.Width())>>1;
        int dy2 = (sizes.cy - rectClient.Height())>>1;
        rectCur.left -= dx2;
        rectCur.right += dx2;
        rectCur.top -= dy2;
        rectCur.bottom += dy2;
    }

    CRect rectDesktop;
    SystemParametersInfo( SPI_GETWORKAREA, 0, &rectDesktop, 0 );
    if( rectCur.left < rectDesktop.left )
    {
        rectCur.right += rectDesktop.left - rectCur.left;
        rectCur.left = rectDesktop.left;
    }
    if( rectCur.top < rectDesktop.top )
    {
        rectCur.bottom += rectDesktop.top - rectCur.top;
        rectCur.top = rectDesktop.top;
    }
    if( rectCur.right > rectDesktop.right )
    {
        rectCur.left -= rectCur.right - rectDesktop.right;
        if( rectCur.left < 0 ) rectCur.left = 0;
        rectCur.right = rectDesktop.right;
    }
    if( rectCur.bottom > rectDesktop.bottom )
    {
        rectCur.top -= rectCur.bottom - rectDesktop.bottom;
        if( rectCur.top < 0 ) rectCur.top = 0;
        rectCur.bottom = rectDesktop.bottom;
    }
    pWndParent->MoveWindow( &rectCur );
    SetScrollSizes( MM_TEXT, sizes );
}


void CJPGViewView::OpenValidDocument( CString name, bool direction )
{
    int idx = m_imageIndex;
    bool bFound = false;    // have some image opened successfully
    bool bLast = false;     // ran out of list
    CDocument* pDoc;

    // if last image - skip operation
    if( ( (direction == true) && (idx == m_list.GetUpperBound()) ) ||
        ( (direction == false) && (idx == 0) ) )
    {
        return;
    }

    while( !bLast && !bFound )
    {
        if( direction ) {
            idx++;
        } else {
            idx--;
        }

        pDoc = AfxGetApp()->OpenDocumentFile( m_list[ idx ] );
        if( pDoc != NULL )
        {
            bFound = true;
        }
        else
        {
            CString message;
            message.Format( "Can't open file \"%s\"", m_list[idx] );
            if( ( direction && idx == m_list.GetUpperBound() ) ||
                ( (!direction) && idx == 0 ) )
            {
                message.Format( "There are no more images to view"
                    " in %s direction", direction?"forward":"backward" );
                AfxMessageBox( message );
                bLast = true;
            }
        }
    }

    if( bFound ) m_imageIndex = idx;
}


static LPCTSTR lpszDefExt  = "jpg";
static LPCTSTR lpszDefName = "noname";
static LPCTSTR lpszFilter = "JPEG Files (*.jpg)|*.jpg|Windows Bitmap Files (*.bmp)|*.bmp|All Files (*.*)|*.*||";
static DWORD dwFlags = OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT;

void CJPGViewView::OnSaveAs(void)
{
    CString fname;

    CFileDialog fdlg(FALSE,lpszDefExt,lpszDefName,dwFlags,lpszFilter,this);

    if(IDOK == fdlg.DoModal())
    {
        fname = fdlg.GetPathName();
        CJPGViewDoc* pDoc = GetDocument();
        ASSERT_VALID( pDoc );
        pDoc->OnSaveDocument( fname );
    }

    return;
} // CJPGViewView::OnSaveAs()


void CJPGViewView::OnViewNextImage()
{
    TRACE( "CJPGViewView::OnViewNextImage()\n" );
    if( m_bViewFolder && m_imageIndex < m_list.GetUpperBound() )
    {
        OpenValidDocument( m_list[ m_imageIndex ], true );
    }
}


void CJPGViewView::OnViewPrevImage()
{
    TRACE( "CJPGViewView::OnViewPrevImage()\n" );
    if( m_bViewFolder && m_imageIndex > 0 )
    {
        OpenValidDocument( m_list[ m_imageIndex ], false );
    }
}


void CJPGViewView::OnUpdateViewNext(CCmdUI* pCmdUI)
{
    int bound = m_list.GetUpperBound();
    if( m_imageIndex == bound || bound == -1 )
        pCmdUI->Enable( FALSE );
    else
        pCmdUI->Enable( TRUE );
}


void CJPGViewView::OnUpdateViewPrev(CCmdUI* pCmdUI)
{
    if( m_imageIndex == 0 ) {
        pCmdUI->Enable( FALSE );
    } else {
        pCmdUI->Enable( TRUE );
    }
}


void CJPGViewView::OnViewImageInfo()
{
    CJPGViewDoc* pDoc = GetDocument();
    ASSERT_VALID( pDoc );

    if( pDoc->IsEmpty() )
    {
        AfxMessageBox( "No image in view" );
        return;
    }
    SIZE size = pDoc->GetImageDims();

    CImageInfoDlg dlg;
    dlg.m_width = size.cx;
    dlg.m_height = size.cy;
    dlg.m_colors = static_cast<DWORD>(pow( 2, 3*8 ));
    // 3 is the default number of DIB channels for IJL

    dlg.DoModal();
}


void CJPGViewView::OnViewOptions()
{
    COptionsDlg dlg;
    bool bCentered = m_bCenterInWindow;
    dlg.m_bResizeWithImage = m_bResizeWithImage;
    dlg.m_bCenterToPrev = m_bCenterToPrev;
    dlg.m_bCenterInWindow = m_bCenterInWindow;
    dlg.m_bScaleToFit = m_bScaleToFit;
    dlg.m_bCenterOnPage = m_bCenterOnPage;

    int result = dlg.DoModal();

    if( result == IDOK )
    {
        m_bResizeWithImage = INT2bool(dlg.m_bResizeWithImage);
        m_bCenterToPrev = INT2bool(dlg.m_bCenterToPrev);
        m_bCenterInWindow = INT2bool(dlg.m_bCenterInWindow);
        m_bScaleToFit = INT2bool(dlg.m_bScaleToFit);
        m_bCenterOnPage = INT2bool(dlg.m_bCenterOnPage);

        if( m_bResizeWithImage ) ResizeImageWindow();
        if( !(m_bCenterInWindow && bCentered) && !m_bResizeWithImage ){
            Invalidate( TRUE );
        }
    }
}

///////////////////////////////////////////////////////////////////////////////

static LPCTSTR g_section        = _T("Settings");
static LPCTSTR g_resizeFlag     = _T("ResizeWithImage");
static LPCTSTR g_centerToFlag   = _T("CenterToPrevious");
static LPCTSTR g_centerInFlag   = _T("CenterInWindow" );
static LPCTSTR g_scaleFlag      = _T("ScaleToFit");
static LPCTSTR g_centerOnFlag   = _T("CenterOnPage");

static const UINT  g_default    = FALSE;


int CJPGViewView::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
    if (CScrollView ::OnCreate(lpCreateStruct) == -1) return -1;

    CWinApp* pwa = AfxGetApp();
    ASSERT_VALID( pwa );

    // Load options from Regisry
    UINT data;

    data = pwa->GetProfileInt( g_section, g_resizeFlag, g_default );
    m_bResizeWithImage = INT2bool(data);

    data = pwa->GetProfileInt( g_section, g_centerToFlag, g_default );
    m_bCenterToPrev = INT2bool(data);

    data = pwa->GetProfileInt( g_section, g_centerInFlag, g_default );
    m_bCenterInWindow = INT2bool(data);

    data = pwa->GetProfileInt( g_section, g_scaleFlag, g_default );
    m_bScaleToFit = INT2bool(data);

    data = pwa->GetProfileInt( g_section, g_centerOnFlag, g_default );
    m_bCenterOnPage = INT2bool(data);

    return 0;
}


void CJPGViewView::OnDestroy()
{
    CWinApp* pwa = AfxGetApp();
    ASSERT_VALID( pwa );

    // Write current options to registry
    pwa->WriteProfileInt( g_section, g_resizeFlag, m_bResizeWithImage );
    pwa->WriteProfileInt( g_section, g_centerToFlag, m_bCenterToPrev );
    pwa->WriteProfileInt( g_section, g_centerInFlag, m_bCenterInWindow );
    pwa->WriteProfileInt( g_section, g_scaleFlag, m_bScaleToFit );
    pwa->WriteProfileInt( g_section, g_centerOnFlag, m_bCenterOnPage );

    CScrollView ::OnDestroy();
}


///////////////////////////////////////////////////////////////////////////////
//                       End of file 'JPGViewView.cpp'                       //
///////////////////////////////////////////////////////////////////////////////
