%**************************************************************************
% Function Name  : VNplotADOR
% Description    : 
%  Plot the data captured by the VNrecordADOR function. Pass the data
%  received along with the ADOR register type, and this function will plot
%  the results along with labeled and titled figures.
%
% Input(s)       : ADOR       -> register acronym
%                : sampleRate -> sample rate in Hz
%                : numSec     -> number of seconds to record
%                : titleName  -> Prefix for plot titles (optional)
% Output(s)      : None
%**************************************************************************
% Examples:
%
%   s = VNserial('COM8');
%
%   data = VNrecordADOR(s, 'YPR', 100, 10); %record @ 100Hz for 10s
%
%   VNplotADOR(data, 'YPR', 100);
%
% ------------- VectorNav Technologies, LLC -------------------------------
% This file is property of VectorNav Technologies and cannot be used,
% copied or distributed without the written consent of VectorNav
% Technologies. 
% -------------------------------------------------------------------------
function VNplotADOR(data, ADOR, sampleRate, titleName)

if(isstr(ADOR))
    ADOR = VNgetADORID(ADOR);
end

if nargin < 4
    titleName = '';
end

%Plot results
if ADOR == 1 %YPR
    plotYPR(data, sampleRate, titleName);
elseif ADOR == 2 %QTN
    plotQuat(data, sampleRate, titleName);
elseif ADOR == 3 %QTM
    plotQuat(data(:,1:4), sampleRate, titleName);
    plotMagnetic(data(:,5:7), sampleRate, titleName);
elseif ADOR == 4 %QTA
    plotQuat(data(:,1:4), sampleRate, titleName);
    plotAccel(data(:,5:7), sampleRate, titleName);
elseif ADOR == 5 %QTR
    plotQuat(data(:,1:4), sampleRate, titleName);
    plotGyro(data(:,5:7), sampleRate, titleName);
elseif ADOR == 6 %QMA
    plotQuat(data(:,1:4), sampleRate, titleName);
    plotMagnetic(data(:,5:7), sampleRate, titleName);
    plotAccel(data(:,8:10), sampleRate, titleName);
elseif ADOR == 7 %QAR
    plotQuat(data(:,1:4), sampleRate, titleName);
    plotAccel(data(:,5:7), sampleRate, titleName);
    plotGyro(data(:,8:10), sampleRate, titleName);    
elseif ADOR == 8 %QMR
    plotQuat(data(:,1:4), sampleRate, titleName);
    plotMagnetic(data(:,5:7), sampleRate, titleName);    
    plotAccel(data(:,8:10), sampleRate, titleName);
    plotGyro(data(:,11:13), sampleRate, titleName); 
elseif ADOR == 10 %MAG
    plotMagnetic(data, sampleRate, titleName); 
elseif ADOR == 11 %ACC
    plotAccel(data, sampleRate, titleName);
elseif ADOR == 12 %'GYR'
    plotGyro(data, sampleRate, titleName); 
elseif ADOR == 13 %'MAR'
    plotMagnetic(data(:,1:3), sampleRate, titleName); 
    plotAccel(data(:,4:6), sampleRate, titleName);
    plotGyro(data(:,7:9), sampleRate, titleName);    
elseif ADOR == 14 %'YMR'
    plotYPR(data(:,1:3), sampleRate, titleName);
    plotMagnetic(data(:,4:6), sampleRate, titleName); 
    plotAccel(data(:,7:9), sampleRate, titleName);
    plotGyro(data(:,10:12), sampleRate, titleName); 
elseif ADOR == 252 %RAW
    figure;
    t = (1:size(data,1))./sampleRate;
    stitle = sprintf('%s Uncalibrated magnetometer sensor voltages at %i Hz', titleName, sampleRate); 
    plot(t,data(:,1:3)); title(stitle);
    xlabel('Time [s]'); ylabel('Voltages [deg]');
    legend('X', 'Y', 'Z');    
    figure;
    t = (1:size(data,1))./sampleRate;
    stitle = sprintf('%s Uncalibrated accelerometer sensor voltages at %i Hz', titleName, sampleRate); 
    plot(t,data(:,4:6)); title(stitle);
    xlabel('Time [s]'); ylabel('Voltages [deg]');
    legend('X', 'Y', 'Z');    
    figure;
    t = (1:size(data,1))./sampleRate;
    stitle = sprintf('%s Uncalibrated gyro sensor voltages at %i Hz', titleName, sampleRate); 
    plot(t,data(:,7:9)); title(stitle);
    xlabel('Time [s]'); ylabel('Voltages [deg]');
    legend('X', 'Y', 'Z');    
elseif ADOR == 253 %CMV
    figure;
    t = (1:size(data,1))./sampleRate;
    stitle = sprintf('%s Calibrated magnetometer measurements at %i Hz', titleName, sampleRate); 
    plot(t,data(:,1:3)); title(stitle);
    xlabel('Time [s]'); ylabel('User dimensions');
    legend('X', 'Y', 'Z');    
    figure;
    t = (1:size(data,1))./sampleRate;
    stitle = sprintf('%s Calibrated accelerometer sensor measurements at %i Hz', titleName, sampleRate); 
    plot(t,data(:,4:6)); title(stitle);
    xlabel('Time [s]'); ylabel('Acceleration [m/s^2]');
    legend('X', 'Y', 'Z');    
    figure;
    t = (1:size(data,1))./sampleRate;
    stitle = sprintf('%s Calibrated gyro sensor measurements at %i Hz', titleName, sampleRate); 
    plot(t,data(:,7:9).*(180/pi)); title(stitle);
    xlabel('Time [s]'); ylabel('Angular Rates [deg/s]');
    legend('X', 'Y', 'Z');    
elseif ADOR == 254 %STV
    plotQuat(data(:,1:4), sampleRate, titleName);
    figure;
    t = (1:size(data,1))./sampleRate;
    stitle = sprintf('%s Gyro bias estimates at %i Hz', titleName, sampleRate); 
    plot(t,data(:,5:7).*(180/pi)); title(stitle);
    xlabel('Time [s]'); ylabel('Bias estimate [deg/s]');
    legend('X', 'Y', 'Z');    
end
end

function plotYPR(data, sampleRate, titleName)
    figure;
    t = (1:size(data,1))./sampleRate;
    stitle = sprintf('%s Yaw / Pitch / Roll at %i Hz', titleName, sampleRate); 
    plot(t,data); title(stitle);
    xlabel('Time [s]'); ylabel('Angle [deg]');
    legend('Yaw', 'Pitch', 'Roll');
end

function plotQuat(data, sampleRate, titleName)
    figure;
    t = (1:size(data,1))./sampleRate;
    stitle = sprintf('%s Quaterion at %i Hz', titleName, sampleRate); 
    plot(t,data); title(stitle);
    xlabel('Time [s]'); ylabel('quaternion component');
    legend('Q_0', 'Q_1', 'Q_2', 'Q_3');
end

function plotMagnetic(data, sampleRate, titleName)
    figure;
    t = (1:size(data,1))./sampleRate;
    stitle = sprintf('%s Magnetic at %i Hz', titleName, sampleRate); 
    plot(t,data); title(stitle);
    xlabel('Time [s]'); ylabel('Magnetic [custom units]');
    legend('X', 'Y', 'Z');
end

function plotAccel(data, sampleRate, titleName)
    figure;
    t = (1:size(data,1))./sampleRate;
    stitle = sprintf('%s Acceleration at %i Hz', titleName, sampleRate); 
    plot(t,data); title(stitle);
    xlabel('Time [s]'); ylabel('Acceleration [m/s^2]');
    legend('X', 'Y', 'Z');    
end

function plotGyro(data, sampleRate, titleName)
    figure;
    t = (1:size(data,1))./sampleRate;
    stitle = sprintf('%s Angular Rates at %i Hz', titleName, sampleRate); 
    plot(t,data.*(180/pi)); title(stitle);
    xlabel('Time [s]'); ylabel('Angular Rates [deg/s]');
    legend('X', 'Y', 'Z');    
end