                             // JPGViewDoc.cpp //
///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//                 INTEL CORPORATION PROPRIETARY INFORMATION                 //
//    This software is supplied under the terms of a license agreement or    //
//   nondisclosure agreement with Intel Corporation and may not be copied    //
//    or disclosed except in accordance with the terms of that agreement.    //
//        Copyright (c) 1998 Intel Corporation. All Rights Reserved.         //
//                                                                           //
//  Project:                                                                 //
//    IJL                                                                    //
//  Purpose:                                                                 //
//    JPEG viewer                                                            //
//  Author(s):                                                               //
//    Paul V. Pervov                                                         //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////
#include "stdafx.h"

#ifndef __JPGVIEW_H__
#include "JPGView.h"
#endif

#ifndef __JPGVIEWDOC_H__
#include "JPGViewDoc.h"
#endif

#ifndef __FILEOPENDLG_H__
#include "FileOpenDlg.h"
#endif

#ifndef __IJL_H__
#include "ijl.h"
#endif

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

///////////////////////////////////////////////////////////////////////////////
//                                CJPGViewDoc                                //
///////////////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNCREATE(CJPGViewDoc, CDocument)

BEGIN_MESSAGE_MAP(CJPGViewDoc, CDocument)
  //{{AFX_MSG_MAP(CJPGViewDoc)
  ON_COMMAND(ID_FILE_OPEN, OnFileOpen)
  //}}AFX_MSG_MAP
END_MESSAGE_MAP()


CJPGViewDoc::~CJPGViewDoc()
{
    DeleteContents();
    return;
}

/*
BOOL CJPGViewDoc::OnNewDocument()
{
    if( !CDocument::OnNewDocument() )
        return FALSE;

    DeleteContents();

    return TRUE;
}
*/

void CJPGViewDoc::Serialize( CArchive& ar )
{
    if( ar.IsStoring() )
    {
    }
    else
    {
    }
    return;
}


void CJPGViewDoc::RGBA_FPX_to_BGRA(BYTE* data,int width,int height)
{
  int   i;
  int   j;
  int   pad;
  int   line_width;
  BYTE  r, g, b, a;
  BYTE* ptr;

  ptr = data;
  pad = IJL_DIB_PAD_BYTES(width,4);
  line_width = width * 4 + pad;

  for(i = 0; i < height; i++)
  {
    ptr = data + line_width*i;
    for(j = 0; j < width; j++)
    {
      r = ptr[0];
      g = ptr[1];
      b = ptr[2];
      a = ptr[3];
      ptr[2] = (r*a+1) >> 8;
      ptr[1] = (g*a+1) >> 8;
      ptr[0] = (b*a+1) >> 8;
      ptr += 4;
    }
  }

  return;
} // CJPGViewDoc::RGBA_FPX_to_BGRA()


void CJPGViewDoc::BGRA_to_RGBA(BYTE* data,int width,int height)
{
  int   i;
  int   j;
  int   pad;
  int   line_width;
  BYTE  r, g, b, a;
  BYTE* ptr;

  ptr = data;
  pad = IJL_DIB_PAD_BYTES(width,4);
  line_width = width * 4 + pad;

  for(i = 0; i < height; i++)
  {
    ptr = data + line_width*i;
    for(j = 0; j < width; j++)
    {
      b = ptr[0];
      g = ptr[1];
      r = ptr[2];
      a = ptr[3];
      ptr[0] = r;
      ptr[1] = g;
      ptr[2] = b;
      ptr += 4;
    }
  }

  return;
} // CJPGViewDoc::BGRA_to_RGBA()


BOOL CJPGViewDoc::OnOpenDocument( LPCTSTR lpszPathName )
{
    if (!CDocument::OnOpenDocument(lpszPathName))
        return FALSE;

    JPEG_CORE_PROPERTIES image;
    ZeroMemory( &image, sizeof( JPEG_CORE_PROPERTIES ) );
    BYTE* imageData;

    BeginWaitCursor();

    TRY
        if( ijlInit( &image ) != IJL_OK )
        {
            TRACE( "Cannot initialize Intel JPEG library\n" );
            AfxThrowUserException();
        }

        image.JPGFile = const_cast<char*>(lpszPathName);
        if( ijlRead( &image, IJL_JFILE_READPARAMS ) != IJL_OK )
        {
            TRACE( "Cannot read JPEG file header from %s file\n",
              image.JPGFile );
            AfxThrowUserException();
        }

        // !dudnik: to fix bug case 584680, [OT:287A305B]
        // Set the JPG color space ... this will always be
        // somewhat of an educated guess at best because JPEG
        // is "color blind" (i.e., nothing in the bit stream
        // tells you what color space the data was encoded from).
        // However, in this example we assume that we are
        // reading JFIF files which means that 3 channel images
        // are in the YCbCr color space and 1 channel images are
        // in the Y color space.
        switch(image.JPGChannels)
        {
        case 1:
          image.JPGColor    = IJL_G;
          image.DIBChannels = 3;
          image.DIBColor    = IJL_BGR;
          break;

        case 3:
          image.JPGColor    = IJL_YCBCR;
          image.DIBChannels = 3;
          image.DIBColor    = IJL_BGR;
          break;

        case 4:
          image.JPGColor    = IJL_YCBCRA_FPX;
          image.DIBChannels = 4;
          image.DIBColor    = IJL_RGBA_FPX;
          break;

        default:
          // This catches everything else, but no
          // color twist will be performed by the IJL.
          image.DIBColor = (IJL_COLOR)IJL_OTHER;
          image.JPGColor = (IJL_COLOR)IJL_OTHER;
          image.DIBChannels = image.JPGChannels;
          break;
        }

        image.DIBWidth    = image.JPGWidth;
        image.DIBHeight   = image.JPGHeight;
        image.DIBPadBytes = IJL_DIB_PAD_BYTES(image.DIBWidth,image.DIBChannels);

        int imageSize = (image.DIBWidth * image.DIBChannels + image.DIBPadBytes) *
          image.DIBHeight;

        imageData = new BYTE[ imageSize ];
        if( imageData == NULL )
        {
            TRACE( "Cannot allocate memory for image\n" );
            AfxThrowUserException();
        }

        image.DIBBytes = imageData;

        if( ijlRead( &image, IJL_JFILE_READWHOLEIMAGE ) != IJL_OK )
        {
            TRACE( "Cannot read image data from %s file\n", image.JPGFile );
            delete[] imageData;
            AfxThrowUserException();
        }

        if( ijlFree( &image ) != IJL_OK )
        {
            TRACE( "Cannot free Intel(R) JPEG library" );
        }

        if(image.DIBColor == IJL_RGBA_FPX)
        {
          RGBA_FPX_to_BGRA(imageData,image.DIBWidth,image.DIBHeight);
        }

    CATCH_ALL( e )

        EndWaitCursor();
        ijlFree( &image );

        AfxMessageBox( "Error opening JPEG file" );
        return FALSE;

    END_CATCH_ALL

    // initializing incapsulated image with correct values
    m_imageData     = imageData;
    m_imageDims.cx  = image.DIBWidth;
    m_imageDims.cy  = image.DIBHeight;
    m_imageChannels = image.DIBChannels;

    EndWaitCursor();
    SetModifiedFlag( FALSE );

    // now we have
    //   m_imageData  containing image data, and
    //   m_imageDims  with image dimensions, and
    //   m_imageChannels with image number of channels

    return TRUE;
} // CJPGViewDoc::OnOpenDocument()



BOOL CJPGViewDoc::SaveAsJPEG(LPCTSTR lpszPathName)
{
    BOOL bres = TRUE;
    JPEG_CORE_PROPERTIES image;
    ZeroMemory( &image, sizeof( JPEG_CORE_PROPERTIES ) );

    BeginWaitCursor();

    TRY

        if( ijlInit( &image ) != IJL_OK )
        {
            TRACE( "Can't initialize Intel(R) JPEG library\n" );
            AfxThrowUserException();
        }

        // Setup DIB
        image.DIBWidth         = m_imageDims.cx;
        image.DIBHeight        = m_imageDims.cy;
        image.DIBBytes         = m_imageData;

        // Setup JPEG
        image.JPGFile          = const_cast<char*>(lpszPathName);
        image.JPGWidth         = m_imageDims.cx;
        image.JPGHeight        = m_imageDims.cy;

        switch(m_imageChannels)
        {
        case 3:
          image.DIBColor       = IJL_BGR;
          image.DIBChannels    = 3;
          image.DIBPadBytes    = IJL_DIB_PAD_BYTES(image.DIBWidth,3);
          image.JPGColor       = IJL_YCBCR;
          image.JPGChannels    = 3;
          image.JPGSubsampling = IJL_411;
          break;

        case 4:
          image.DIBColor       = IJL_RGBA_FPX;
          image.DIBChannels    = 4;
          image.DIBPadBytes    = IJL_DIB_PAD_BYTES(image.DIBWidth,4);
          image.JPGColor       = IJL_YCBCRA_FPX;
          image.JPGChannels    = 4;
          image.JPGSubsampling = IJL_4114;
          break;

        default:
          break;
        }

        if(image.DIBColor == IJL_RGBA_FPX)
        {
          BGRA_to_RGBA(image.DIBBytes,image.DIBWidth,image.DIBHeight);
        }


        if( ijlWrite( &image, IJL_JFILE_WRITEWHOLEIMAGE ) != IJL_OK )
        {
            TRACE( "Can't write image\n" );
            AfxThrowUserException();
        }

        if( ijlFree( &image ) != IJL_OK )
        {
            TRACE( "Can't free Intel(R) JPEG library\n" );
        }

    CATCH_ALL( e )

        bres = FALSE;
        EndWaitCursor();
        ijlFree( &image );

        AfxMessageBox( "Error storing JPEG file" );

    END_CATCH_ALL

    EndWaitCursor();

    return bres;
} // CJPGViewDoc::SaveAsJPEG()


BOOL CJPGViewDoc::SaveAsBMP(LPCTSTR lpszPathName)
{
    int  pad_bytes;
    int  bwidth;
    int  width_step;
    int  img_size;
    int  file_size;
    BOOL bres = TRUE;
    BITMAPFILEHEADER bmfh;
    BITMAPINFOHEADER bmih;

    pad_bytes = IJL_DIB_PAD_BYTES(m_imageDims.cx,m_imageChannels);
    bwidth = m_imageDims.cx * m_imageChannels;
    width_step = bwidth + pad_bytes;
    img_size = width_step * m_imageDims.cy;
    file_size = img_size + sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER);

    CFile dib(lpszPathName,CFile::modeCreate|CFile::modeWrite|CFile::typeBinary);

    bmfh.bfType      = 'MB';
    bmfh.bfSize      = file_size;
    bmfh.bfReserved1 = 0;
    bmfh.bfReserved2 = 0;
    bmfh.bfOffBits   = sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER);

    dib.Write(&bmfh,sizeof(BITMAPFILEHEADER));

    bmih.biSize          = sizeof(BITMAPINFOHEADER);
    bmih.biWidth         = m_imageDims.cx;
    bmih.biHeight        = -m_imageDims.cy;
    bmih.biPlanes        = 1;

    switch(m_imageChannels)
    {
    case 3:
      bmih.biBitCount = 24;
      break;

    case 4:
      bmih.biBitCount = 32;
      break;

    default:
      break;
    }

    bmih.biCompression   = BI_RGB;
    bmih.biSizeImage     = img_size;
    bmih.biXPelsPerMeter = 0;
    bmih.biYPelsPerMeter = 0;
    bmih.biClrUsed       = 0;
    bmih.biClrImportant  = 0;

    dib.Write(&bmih,sizeof(BITMAPINFOHEADER));

    dib.Write(m_imageData,img_size);

    return bres;
} // CJPGViewDoc::SaveAsBMP()


void CJPGViewDoc::OnFileOpen(void) 
{
  CFileDialogEx dlg(
    TRUE,
    "jpg",
    "*.jpg",
    OFN_EXPLORER | OFN_ENABLETEMPLATE | OFN_HIDEREADONLY | OFN_FILEMUSTEXIST | OFN_NOCHANGEDIR | OFN_PATHMUSTEXIST,
    "JPEG Files (*.jpg)|*.jpg|"
    "All Files (*.*)|*.*||",
    AfxGetMainWnd());

  dlg.m_ofn.hInstance = AfxGetInstanceHandle();
  dlg.m_ofn.lpTemplateName = MAKEINTRESOURCE(IDD_FILEOPENEX);

  if(IDOK == dlg.DoModal())
  {
    AfxGetApp()->OpenDocumentFile(dlg.GetPathName());
  }

  return;
} // CJPGViewDoc::OnFileOpen()


BOOL CJPGViewDoc::OnSaveDocument(LPCTSTR lpszPathName)
{
    BOOL bres = TRUE;
    TCHAR ext[_MAX_EXT];

    _tsplitpath(lpszPathName,NULL,NULL,NULL,&ext[0]);

    if(0 == _tcsicmp(ext,".jpg"))
    {
      bres = SaveAsJPEG(lpszPathName);
    }
    else
    {
      bres = SaveAsBMP(lpszPathName);
    }

    return bres;
} // CJPGViewDoc::OnSaveDocument()


#ifdef _DEBUG
void CJPGViewDoc::AssertValid() const
{
    CDocument::AssertValid();
    ASSERT( m_imageDims.cx >= 0 && m_imageDims.cy >= 0 );
    return;
}


void CJPGViewDoc::Dump( CDumpContext& dc ) const
{
    CDocument::Dump( dc );
    return;
}
#endif


CJPGViewDoc::CJPGViewDoc()
{
    CleanDocument();
    return;
}


void CJPGViewDoc::CleanDocument()
{
    m_imageData = NULL;
    m_imageDims.cx = m_imageDims.cy = 1;
    m_imageChannels = 0;
    return;
}


void CJPGViewDoc::DeleteContents()
{
    CDocument::DeleteContents();

    if( m_imageData != NULL )
    {
        delete[] m_imageData;
    }
    CleanDocument();
    return;
}

//{{AFX_INSERT_LOCATION}}

///////////////////////////////////////////////////////////////////////////////
//                       End of file 'JPGViewDoc.cpp'                        //
///////////////////////////////////////////////////////////////////////////////
