#include <stdlib.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <string.h>
#include <linux/if.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/reboot.h>

#ifdef CONFIG_FEATURE_MULTICAST_INFO_GROUP
#define INFO_MULTICAST_GROUP CONFIG_FEATURE_MULTICAST_INFO_GROUP
#else
#define INFO_MULTICAST_GROUP "233.89.188.1"
#endif


#ifdef CONFIG_FEATURE_MULTICAST_INFO_PORT
#define INFO_MULTICAST_PORT CONFIG_FEATURE_MULTICAST_INFO_PORT
#else
#define INFO_MULTICAST_PORT 10001
#endif


#ifdef CONFIG_FEATURE_MULTICAST_INFO_IFNAME
#define INFO_IFNAME CONFIG_FEATURE_MULTICAST_INFO_IFNAME
#else
#define INFO_IFNAME "eth0"
#endif

#ifdef CONFIG_FEATURE_MULTICAST_INFO_VERSION_FILE
#define INFO_VERSION_FILE CONFIG_FEATURE_MULTICAST_INFO_VERSION_FILE
#else
#define INFO_VERSION_FILE "/usr/lib/version"
#endif

#define LO_IFNAME "lo"

#define PROTO_VERSION 0x00
#define PROTO_VERSION1 0x01
#define PROTO_CMD_INFO 0x00
#define PROTO_CMD_REBOOT 0x01

#define DEF_MAC_ENTRY 0x01
#define MAC_IP_ENTRY 0x2
#define FW_VERSION_ENTRY 0x03
#define IP_ADDR_ENTRY 0x04
#define MAC_ENTRY 0x05

static int
init_mcast_sock(void) {
	struct ip_mreq mreq;
	struct sockaddr_in addr;
	int sock;

	sock = socket(PF_INET, SOCK_DGRAM, 0);
	if (sock < 0)
		return -1;

	inet_aton(INFO_MULTICAST_GROUP, &mreq.imr_multiaddr);
	mreq.imr_interface.s_addr = INADDR_ANY;

	if (setsockopt(sock, SOL_IP, IP_ADD_MEMBERSHIP, &mreq, sizeof(mreq))) {
		close(sock);
		return -1;
	}

	addr.sin_family = AF_INET;
	addr.sin_port = htons(INFO_MULTICAST_PORT);
	addr.sin_addr.s_addr = htonl(INADDR_ANY);

	if (bind(sock, (struct sockaddr*)&addr, sizeof(addr))) {
		close(sock);
		return -1;
	}

	return sock;
}

static int
init_bcast_sock(void)
{
	int val = 1;
	int sock;

	sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
	if (sock < 0)
	    return -1;
                             
	if (setsockopt(sock, SOL_SOCKET, SO_BROADCAST, &val, sizeof(val))) {
	    close(sock);
	    return -1;
	}
	return sock;
}


typedef struct mcast_request {
	unsigned char version;
	unsigned char cmd;
}  __attribute__((__packed__)) mcast_request_t;

typedef struct mcast_request_new {
    	mcast_request_t request;
        unsigned int magic;
	unsigned short data_len;
        unsigned char data[0];
}  __attribute__((__packed__)) mcast_request_new_t;

typedef struct request_data_entry {
        unsigned char type;
    	unsigned short len;
	unsigned char data[0];
}  __attribute__((__packed__)) request_data_entry_t;

typedef struct mcast_info_response {
	unsigned char version;
	unsigned char mac[6];
	unsigned int ip;
	unsigned int fw_version_len;
	unsigned char fw_version[0];
}  __attribute__((__packed__)) mcast_info_response_t;

static char fw_version[128];
static size_t fw_version_len = 0;

static int 
do_ioctl(int fd, const char* ifname, int req, struct ifreq* ifr) {
        if (fd == -1) {
		return -1;
        }

        strncpy(ifr->ifr_name, ifname, IFNAMSIZ);
        return ioctl(fd, req, ifr);
}

static int 
get_ipaddr(int fd, const char* ifname, struct in_addr* addr) {
        struct ifreq ifr;
        int rc;

        memset(&ifr, 0, sizeof(ifr));
        rc = do_ioctl(fd, ifname, SIOCGIFADDR, &ifr);
        if (addr)
                memcpy(addr, &((struct sockaddr_in*)&ifr.ifr_addr)->sin_addr, sizeof(struct in_addr));
        return 0;
}

static int 
get_hwaddr(int fd, const char* ifname, unsigned char* hwaddr) {
        struct ifreq ifr;
        int rc;

        memset(&ifr, 0, sizeof(ifr));
        rc = do_ioctl(fd, ifname, SIOCGIFHWADDR, &ifr);
        if (rc < 0)
                return -1;

        if (hwaddr)
                memcpy(hwaddr, &ifr.ifr_hwaddr.sa_data, 6);
        return 0;
}

static char ifaces[127][IFNAMSIZ ];

static int
fill_response(mcast_info_response_t* resp, const char* ifname)
{
	size_t resp_len = 0;
        int fd;

	fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
        if (fd >= 0) {
		get_hwaddr(fd, ifname, resp->mac);
       		get_ipaddr(fd, ifname, (struct in_addr*)&(resp->ip));
        }
	close(fd);

	if (resp->ip == 0 || !memcmp(resp->mac, "\00\00\00\00\00\00", sizeof(resp->mac)))
            return resp_len;

	if (fw_version_len == 0) {
		FILE* f = fopen(INFO_VERSION_FILE, "r");
		if (f) {
			if (fgets(fw_version, sizeof(fw_version), f)) {
				fw_version_len = strlen(fw_version);
			}
			fclose(f);
		}
	}

	if (fw_version_len) {
		resp->fw_version_len = htonl(fw_version_len);
		memcpy(resp + 1,
		       fw_version, fw_version_len);
	}

	resp_len = sizeof(mcast_info_response_t) + fw_version_len;

	return resp_len;
}

static int
fill_response_data(mcast_request_new_t* resp, int resp_len, const char* iface, unsigned char type, int data_pos)
{
    int fd;
    request_data_entry_t* entry;

    if (!resp_len || resp_len - sizeof(*resp) <= data_pos)
	return 0;
    entry = (request_data_entry_t*)(resp->data + data_pos);
    switch(type)
    {
    case MAC_IP_ENTRY:
	fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
	if (fd >= 0) {
	    unsigned char mac[6] = {0,0,0,0,0,0};
            unsigned int ip = 0;
            get_hwaddr(fd, iface, mac);
            get_ipaddr(fd, iface, (struct in_addr*)&ip);
            close(fd);
	    if (!memcmp(mac, "\00\00\00\00\00\00", sizeof(mac)) || !ip)
		return 0;
	    entry->len = htons(sizeof(mac) + sizeof(ip));
            entry->type = type;
	    memcpy(entry->data, mac, sizeof(mac));
	    memcpy(entry->data + sizeof(mac), &ip, sizeof(ip));
        return sizeof(mac) + sizeof(ip) + sizeof(*entry);
	}
        break;
    case DEF_MAC_ENTRY:
    case MAC_ENTRY:
	fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
	if (fd >= 0) {
	    unsigned char mac[6] = {0,0,0,0,0,0};
	    get_hwaddr(fd, iface, mac);
	    close(fd);
	    if (!memcmp(mac, "\00\00\00\00\00\00", sizeof(mac)))
		return 0;
	    entry->len = htons(sizeof(mac));
            entry->type = type;
	    memcpy(entry->data, mac, sizeof(mac));
        return sizeof(mac) + sizeof(*entry);
	}
        break;
    case IP_ADDR_ENTRY:
	fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
	if (fd >= 0) {
            unsigned int ip = 0;
	    get_ipaddr(fd, iface, (struct in_addr*)&ip);
	    close(fd);
	    if (ip == 0)
		return 0;
	    entry->len = htons(sizeof(ip));
            entry->type = type;
	    memcpy(entry->data, &ip, sizeof(ip));
        return sizeof(ip) + sizeof(*entry);
	}
        break;
    case FW_VERSION_ENTRY:
	if (fw_version_len == 0) {
		FILE* f = fopen(INFO_VERSION_FILE, "r");
		if (f) {
			if (fgets(fw_version, sizeof(fw_version), f)) {
				fw_version_len = strlen(fw_version);
			}
			fclose(f);
		}
	}

	if (fw_version_len) {
	    entry->len = htons(fw_version_len);
            entry->type = type;
	    memcpy(entry->data, fw_version, fw_version_len);
            return fw_version_len + sizeof(*entry);
	}
        break;
    }
    return 0;
}

static char mcast_buf[64];
static char _response[512];
static int _response_len = 0;
static mcast_info_response_t* response = (mcast_info_response_t*)_response;
static int bcast_sock = -1;
static int mcast_sock = -1;
static unsigned int mcast_magic = 0;

static unsigned int
generate_magic(void)
{
    srandom(mcast_magic + random());
    return random();
}

static int update_ifaces(int sockfd)
{
	struct ifconf ifc;
	struct ifreq *ifr;
        int numreqs = 30;
	int n;

        memset(ifaces, 0, sizeof(ifaces));
	ifc.ifc_buf = NULL;
	for (;;) {
		ifc.ifc_len = sizeof(struct ifreq) * numreqs;
		ifc.ifc_buf = xrealloc(ifc.ifc_buf, ifc.ifc_len);

		if (ioctl(sockfd, SIOCGIFCONF, &ifc) < 0) {
			free(ifc.ifc_buf);
                        return 0;
		}
		if (ifc.ifc_len == sizeof(struct ifreq) * numreqs) {
			/* assume it overflowed and try again */
			numreqs += 10;
			continue;
		}
		break;
	}

	ifr = ifc.ifc_req;
        numreqs = 0;
	for (n = 0; n < ifc.ifc_len && numreqs < sizeof(ifaces) / sizeof(ifaces[0]);
	     n += sizeof(struct ifreq)) {
                memcpy(ifaces[numreqs], ifr->ifr_name, IFNAMSIZ);
		ifr++;
                ++numreqs;
	}
        free(ifc.ifc_buf);
        return numreqs;
}

static int handle_version0(void* buf, size_t buf_len, int sock, struct sockaddr_in addr, socklen_t addr_len)
{
	mcast_request_t* req = (mcast_request_t*)buf;
	int i;
	char* ifname;

	switch (req->cmd) {
	case PROTO_CMD_INFO:
            i = update_ifaces(sock);
	    while ((i > 0) && ((ifname = ifaces[i - 1]) != NULL))
	    {
		memset(response, 0, sizeof(*response));
		response->version = PROTO_VERSION;
		_response_len = fill_response(response, ifname);
		if (_response_len && bcast_sock >= 0) {
		    sendto(bcast_sock, _response, _response_len, MSG_NOSIGNAL,
			   (struct sockaddr*)&addr, addr_len);
		    addr.sin_family = AF_INET;
		    addr.sin_addr.s_addr = htonl(INADDR_BROADCAST);
		    sendto(bcast_sock, _response, _response_len, MSG_NOSIGNAL,
			       (struct sockaddr*)&addr, addr_len);
		}
		--i;
	    }
	    break;
	default:
	    break;
	}
        return 0;
}

static int handle_version1(void* buf, size_t buf_len, int sock, struct sockaddr_in addr, socklen_t addr_len)
{
	mcast_request_new_t* req = (mcast_request_new_t*)buf;
	mcast_request_new_t* resp = (mcast_request_new_t*)_response;
	int i;
	char* ifname;
        int len = 0;

	if (((req->magic != mcast_magic || !mcast_magic) &&
	     req->request.cmd != PROTO_CMD_INFO)
	    ||
	    (buf_len != sizeof(mcast_request_t) &&
	     req->request.cmd == PROTO_CMD_INFO))
            return -1;
	switch (req->request.cmd) {
	case PROTO_CMD_REBOOT:
	    reboot(0);
	    break;
	case PROTO_CMD_INFO:
            i = update_ifaces(sock);
	    memset(resp, 0, sizeof(*resp));
	    resp->request.version = PROTO_VERSION1;
	    resp->request.cmd = req->request.cmd;
	    mcast_magic = htonl(generate_magic());
	    resp->magic = mcast_magic;
	    while ((i > 0) && ((ifname = ifaces[i - 1]) != NULL))
	    {
                len += fill_response_data(resp, sizeof(_response), ifname, MAC_IP_ENTRY, len);
                --i;
	    }
	    len += fill_response_data(resp, sizeof(_response), INFO_IFNAME, DEF_MAC_ENTRY, len);
	    len += fill_response_data(resp, sizeof(_response), NULL, FW_VERSION_ENTRY, len);
	    if (len && bcast_sock >= 0) {
                resp->data_len = htons(len);
                _response_len = sizeof(*resp) + len;
		sendto(bcast_sock, _response, _response_len, MSG_NOSIGNAL,
		       (struct sockaddr*)&addr, addr_len);
		addr.sin_family = AF_INET;
		addr.sin_addr.s_addr = htonl(INADDR_BROADCAST);
		sendto(bcast_sock, _response, _response_len, MSG_NOSIGNAL,
		       (struct sockaddr*)&addr, addr_len);
	    }

            break;
	default:
            mcast_magic = htonl(generate_magic());
	    break;
	}
        return 0;
}

static int
mcast_sock_process(int sock) {
	struct sockaddr_in addr;
	socklen_t addr_len;
	size_t size;
        int ret;
	mcast_request_t* req = (mcast_request_t*)mcast_buf;

	if (sock < 0)
		return -1;

	addr_len = sizeof(addr);
        memset(&addr, 0, addr_len);
	size = recvfrom(sock, mcast_buf, sizeof(mcast_buf), MSG_NOSIGNAL,
			(struct sockaddr*)&addr, &addr_len);
	if (size < sizeof(mcast_request_t))
		return -1;

	if (bcast_sock < 0)
	    if ((bcast_sock = init_bcast_sock()) < 0)
		return -1;

	switch (req->version) {
	case PROTO_VERSION:
            ret = handle_version0(mcast_buf, size, sock, addr, addr_len);
            break;
	case PROTO_VERSION1:
	    ret = handle_version1(mcast_buf, size, sock, addr, addr_len);
            break;
	default: ret = -2;
	}
	close(bcast_sock);
	bcast_sock = -1;

	return ret;
}


static void reinit_mcast(void)
{
    if (mcast_sock != -1)
	close(mcast_sock);
    mcast_sock = -1;
    if (bcast_sock != -1)
	close(bcast_sock);
    bcast_sock = -1;
}

static void check_mcast(void)
{
		fd_set rfds;
		struct timeval tv;
		int retval;
		int last_sock = 0;
		if (mcast_sock < 0)
			mcast_sock = init_mcast_sock();

		FD_ZERO(&rfds);
		if (mcast_sock >= 0) {
			FD_SET(mcast_sock, &rfds);
			last_sock = mcast_sock;
		}
		tv.tv_sec = 1;
		tv.tv_usec = 0;
		retval = select(last_sock + 1, &rfds, NULL, NULL, &tv);
		if (retval > 0) {
			mcast_sock_process(mcast_sock);
		} else if (retval < 0) {
			reinit_mcast();
		}

}
