#!/bin/sh
#
# Initialization script.
# Required parameters:
#
# -l lan devices
# -w wan devices
# -i lan rate
# -o wan rate
#
# Optional parameters:
#
# -j lan burst
# -m wan burst
# -a lan ceil
# -b wan ceil

DEVICESFILE=/tmp/htb_devices.txt
CLASS=1

# these TOS values will place packet to high priority queue
# 0x10 means low latency
# 0x18, 0xb8 - low latency variations, used by voip
HIGH_TOS="0x10 0x18 0xb8"
# 4569 - default voip server port
HIGH_PORTS="4569"

# ports (source or dest) that go to low priority queue
LOW_PORTS="25 22 110 143 943"


usage=\
"
Usage: $0 -l <lan device list> -w <wan device list> -i <lan rate> -o <wan rate> -a <lan ceil rate> -b <wan ceil rate> -j <lan burst> -m <wan burst>\n\n
example: $0 -l 'eth3 ath0' -w 'eth1 eth2' -i 2000Kbit -o 1000Kbit -a 54Mbit -b 100Mbit -j 100Kb -m 100Kb\n
note: default lan/wan ceil rate is 54Mbit\n
"

LANCEIL="54Mbit"
WANCEIL="54Mbit"

while getopts "l:w:i:o:j:m:x:y:a:b:" options; do
    case $options in
	l ) LANDEVICES=$OPTARG;;
	w ) WANDEVICES=$OPTARG;;
	i ) LANRATE=$OPTARG;;
	o ) WANRATE=$OPTARG;;
	j ) LANBURST=$OPTARG;;
	m ) WANBURST=$OPTARG;;
	a ) LANCEIL=$OPTARG;;
	b ) WANCEIL=$OPTARG;;
	* ) 	echo -e $usage
		exit 1;;
    esac
done

if [ -z "$LANDEVICES" ] || [ -z "$LANRATE" ] || [ -z "$WANDEVICES" ] || [ -z "$WANRATE" ]
then
    echo -e $usage
    exit 1
fi      


LG="echo"
# remove stale crap
for dev in $LANDEVICES $WANDEVICES; do
    tc qdisc del dev $dev root > /dev/null 2>&1
done

setup_htb_for_device() {
	CLASS=$1
	dev=$2
	RATE=$3
	BURST=$4
	CEIL=$5
	RATEVAL=`echo ${RATE} | cut -b -$((${#RATE}-4))`

	TC="tc qdisc add dev $dev root handle $CLASS:0 htb default 30"
	$TC || $LG "'${TC}' has failed."

	TC="tc class add dev $dev parent $CLASS:0 classid $CLASS:1 \
		htb rate ${RATE} ${CEIL} burst ${BURST} quantum 1514"
	$TC || $LG "'${TC}' has failed."

	# now add high/norm/low classes
	TC="tc class add dev $dev parent $CLASS:1 classid $CLASS:10 \
		htb rate ${RATE} ${CEIL} burst ${BURST} prio 1 quantum 1514"
	$TC || $LG "'${TC}' has failed."

	TC="tc class add dev $dev parent $CLASS:1 classid $CLASS:20 \
		htb rate ${RATE} ${CEIL} burst ${BURST} prio 2 quantum 1514"
	$TC || $LG "'${TC}' has failed."

	TC="tc class add dev $dev parent $CLASS:1 classid $CLASS:30 \
		htb rate $((9*RATEVAL/10))kbit ${CEIL} burst ${BURST} prio 3 quantum 1514"
	$TC || $LG "'${TC}' has failed."

	TC="tc class add dev $dev parent $CLASS:1 classid $CLASS:40 \
		htb rate $((8*RATEVAL/10))kbit ${CEIL} burst ${BURST} prio 4 quantum 1514"
	$TC || $LG "'${TC}' has failed."

	# all get Stochastic Fairness:
	for id in 10 20 30 40; do
		TC="tc qdisc add dev $dev parent $CLASS:${id} handle ${id}: sfq perturb 10"
		$TC || $LG "'${TC}' has failed."
	done

	#EAPOL highest priority
	TC="tc filter add dev $dev parent $CLASS:0 protocol all prio 5 \
		u32 match u16 0x888e 0xffff at -2 flowid $CLASS:10"
	$TC || $LG "'${TC}' has failed."

	for tos in ${HIGH_TOS}; do
		# TOS Minimum Delay (ssh, NOT scp) in :10
		TC="tc filter add dev $dev parent $CLASS:0 protocol ip prio 10 \
			u32 match ip tos ${tos} 0xff  flowid $CLASS:20"
		$TC || $LG "'${TC}' has failed."
	done

	for port in ${HIGH_PORTS}; do
		TC="tc filter add dev $dev parent $CLASS:0 protocol ip prio 11 \
			u32 match ip dport ${port} 0xffff flowid $CLASS:20"
		$TC || $LG "'${TC}' has failed."
		TC="tc filter add dev $dev parent $CLASS:0 protocol ip prio 11 \
			u32 match ip sport ${port} 0xffff flowid $CLASS:20"
		$TC || $LG "'${TC}' has failed."
	done

	# ICMP (ip protocol 1) in the interactive class :10 so we
	# can do measurements & impress our friends:
	TC="tc filter add dev $dev parent $CLASS:0 protocol ip prio 12 \
		u32 match ip protocol 1 0xff flowid $CLASS:20"
	$TC || $LG "'${TC}' has failed."

	# put ACK packets in the interactive class:
	TC="tc filter add dev $dev parent $CLASS:0 protocol ip prio 15 u32 \
		match ip protocol 6 0xff \
		match u8 0x05 0x0f at 0 \
		match u16 0x0000 0xffc0 at 2 \
		match u8 0x10 0xff at 33 \
		flowid $CLASS:20"
	$TC || $LG "'${TC}' has failed."

	# low priority stuff
	for port in ${LOW_PORTS}; do
		TC="tc filter add dev $dev parent $CLASS:0 protocol ip prio 30 u32 \
			match ip src ${port} flowid $CLASS:40"
		$TC || $LG "'${TC}' has failed."
		TC="tc filter add dev $dev parent $CLASS:0 protocol ip prio 30 u32 \
			match ip dst ${port} flowid $CLASS:40"
		$TC || $LG "'${TC}' has failed."
	done
	#rest will endup in 1:30 class (see root qdisc setting)

} # end of setup_htb_for_device()

## calculate bursts
if [ -z $LANBURST ]; then
	LANBURST="6k"
fi

if [ -z $WANBURST ]; then
	WANBURST="6k"
fi

if [ "${WANCEIL}" != "" ]; then
        WCEIL="ceil ${WANCEIL}"
fi
if [ "${LANCEIL}" != "" ]; then
        LCEIL="ceil ${LANCEIL}"
fi

# init root handles
for dev in $LANDEVICES; do
	setup_htb_for_device $CLASS $dev $LANRATE $LANBURST "$LCEIL"
done

for dev in $WANDEVICES; do
	setup_htb_for_device $CLASS $dev $WANRATE $WANBURST "$WCEIL"
done

# save devices for other scripts
echo $LANDEVICES $WANDEVICES > $DEVICESFILE

