#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
//#include "liblogger/logger.h"
#include "buffer.h"
//#include "uhttpd.h"
//#include "driver.h"


int buffer_empty(struct buffer* buffer)
{ 
	return buffer->size == 0; 
}

/**
 * Checks if there are unread data in bd.
 * @param bd buffer data chunk to check
 * @return return 0, false, 1 - true.
 */
static int buffer_data_empty(struct buffer_data* bd) 
{
	return bd->nread == bd->size ||
	       bd->nread == bd->capacity;
}

/**
 * Checks if there are space left to write data.
 * @param bd buffer data chunk to check
 * @return 0, if there is space left, 1 - if full.
 */
static int buffer_data_full(struct buffer_data* bd) 
{
	return bd->size == bd->capacity;
}

/**
 * @param bd
 * @return next data_buffer chunk.
 */
static struct buffer_data* buffer_data_next(struct buffer_data* bd) 
{
	return list_entry(bd->list.next, struct buffer_data, list);
}

/**
 * @param b buffer
 * @return first data buffer chunk in buffer.
 */
static struct buffer_data* buffer_data_head(struct buffer* b) 
{
	return list_entry(b->bd.next, struct buffer_data, list);	
}

/**
 * Creates buffer chunk of n bytes 'size'.
 * @param size
 * @return  buffer chunk pointer, or NULL on failure.
 */
static struct buffer_data* buffer_data_create(size_t size) 
{
	struct buffer_data* bd;

	bd = malloc(sizeof(*bd));
	if (!bd) return 0;
	memset(bd, 0, sizeof (*bd));
	INIT_LIST_HEAD(&bd->list);

	bd->data = malloc(size);
	if (!bd->data)
	{
		free(bd);
		return 0;
	}
	bd->size = 0;
	bd->nread = 0;
	bd->capacity = size;

	return bd;
}

/**
 * Frees memory taken by bd.
 *
 * @param bd buffer data chunk.
 * @return 0 on success. (never fails though).
 */
static int buffer_data_destroy(struct buffer_data* bd) 
{
	if (!bd) return 0;

	list_del(&bd->list);
	free(bd->data);
	free(bd);	
	
	return 0;
}

/**
 * Expands b by size bytes. 'size' is must be at least 
 * Size should be multiple of BUFFER_CHUNK_SIZE, if not
 * it is adjusted automatically.
 *
 * @param b buffer 
 * @param size bytes count to expand buffer by.
 * @return 0 on success, otherwise failure.
 */
static int buffer_grow(struct buffer* b, size_t size) 
{
	int grow = size + b->size - b->capacity;

	if (grow > 0) {
		struct buffer_data* bd;
		size_t expand = 0;

		expand  = (grow / BUFFER_CHUNK_SIZE);
		expand ++;
		expand *= BUFFER_CHUNK_SIZE;

		bd = buffer_data_create(expand);
		if (!bd) return -1;
		list_add_tail(&bd->list, &b->bd);
		
		b->capacity += expand;
	}

	return 0;
}

/**
 * Initialize buffer. 
 * @todo make chunk size a parameter too.
 *
 * @return 0 on success, otherwise failure.
 */
int buffer_init(struct buffer* b)
{
	if (b == 0) return -1;

	memset(b, 0, sizeof(*b));
	INIT_LIST_HEAD(&b->bd);

	if (buffer_grow(b, BUFFER_INITIAL_CHUNK_SIZE)) {
		return -1;
	}

	b->rbd = buffer_data_head(b);
	b->wbd = buffer_data_head(b);

	return 0;
}


int buffer_create(struct buffer** buffer)
{
	struct buffer* b;

	b = malloc(sizeof(*b));
	if (!b) return -1;

	if (buffer_init(b)) {
		buffer_free(b);
		return -1;
	}

	*buffer = b;
	return 0;
}

int buffer_delete(struct buffer* buffer) 
{
	if (buffer == 0) return 0;

	buffer_free(buffer);
	free(buffer);

	return 0;
}

/**
 * Frees unused chunks as data is read. Chunks are freed and not 
 * reused when all data is already read. 
 *
 * @param b buffer to scan for a unused chunks.
 * @return 0 on sucess, otherwise - failure.
 */
static int buffer_shrink(struct buffer* b) 
{
	struct list_head *safe, *pos;
	struct buffer_data* bd;

	list_for_each_safe(pos,safe, &b->bd) {
		bd = list_entry(pos, struct buffer_data, list);

		if (buffer_data_empty(bd)) {
			b->capacity -= bd->size;
			buffer_data_destroy(bd);	
		} else {
			b->rbd = bd;
			break;	
		}
	}
	
	return 0;
}

/**
 * Store data in buffer.
 * 
 * @param data to store in buffer
 * @param size data length.
 * @return int, bytes written to buffer, or negative on error.
 *    0 bytes written returns zero.
 */
int buffer_write(struct buffer* b, const void* data, size_t size)
{
	struct buffer_data* chunk;
	size_t nwritten = 0;

	if (buffer_grow(b, size)) {
		return -1;
	}

	chunk = b->wbd;
	while(size > nwritten) {
		int left_capacity, left_data, left;
		if (buffer_data_full(chunk)) {
			chunk = buffer_data_next(chunk);
			b->wbd = chunk;
		}

		/* sanity check */
		if (&chunk->list == &b->bd) {
			DBG_WARN( "buffer_write: wbd is HEAD!\n");
			return -1;
		}
	
		left_capacity =  chunk->capacity - chunk->size;
		left_data = size - nwritten;
		left = left_data > left_capacity ? left_capacity : left_data;

		memcpy(chunk->data+chunk->size, data + nwritten, left);
		b->size   += left;
		chunk->size += left;
		nwritten  += left;
	}

	return nwritten;
}

int buffer_printf(struct buffer* b, const char* fmt, ...)
{
    va_list ap;
    char* buff = NULL;
    size_t size = 256; /* initial size */
    int n;

    while(1)
    {
        char* tmp = realloc(buff, size);
        if (0 == tmp)
        {
            free(buff);
            return -1;
        }
        buff = tmp;

        va_start(ap, fmt);
        n = vsnprintf (buff, size, fmt, ap);
        va_end(ap);

        if (n > -1 && n < size)
        {
            break; /* ok */
        }
        if (n > -1) /* exact size */
        {
            size = n + 1;
        }
        else /* twice old */
        {
            size *= 2;
        }
    }
    n = buffer_write(b, buff, n);
    free(buff);
    return n;
}

int buffer_writestr(struct buffer* buffer, const char* str)
{
	return buffer_write(buffer, str, strlen(str));
}

/**
 * Retrieves data from buffer. 
 *
 * @param b buffer that holds data.
 * @param data copy will be made to.
 * @param size up N bytes.
 *
 * @return >0, bytes copied, negative on error.
 *   O returned when b is emtpy.
 */
int buffer_read(struct buffer* b, void* data, size_t size)
{
	struct buffer_data* rbd; 
	int nread = 0;	/* bytes already read */
	int ndata;	/* bytes to read from buffer */

	ndata = size > b->size ? b->size : size; 
	
	rbd = b->rbd;
	while (ndata > nread) {
		size_t left_data, left_capacity, left;
		if (&rbd->list == &b->bd) {
			DBG_WARN("buffer_read: rbd is HEAD!\n");
			return -1;
		}

		if (rbd->nread == rbd->capacity) {
			rbd = buffer_data_next(rbd);
			b->rbd = rbd;
		}

		left_data = rbd->size - rbd->nread;
		left_capacity = ndata-nread;
		left = left_data > left_capacity ? left_capacity : left_data;

		memcpy(data+nread, rbd->data+rbd->nread, left);
		b->size -= left;
		rbd->nread += left;
		nread += left;
	}
	
	if (buffer_shrink(b)) {
		return -1;	
	}

	return nread;
}

/**
 * Frees memory taken by buffer. Note, if b was allocated dynamically,
 * programmer must call free(3) on b himself. 'buffer_free' frees only
 * buffer data chunks. In order to reuse b, call buffer_init() again.
 *
 * @param b buffer to free.
 * @return 0 on success, otherwise failure.
 */
int buffer_free(struct buffer* b)
{
	struct list_head *safe, *pos;
	struct buffer_data* bd;

	if (b == 0) return 0;

	list_for_each_safe(pos,safe, &b->bd) {
		bd = list_entry(pos, struct buffer_data, list);
		buffer_data_destroy(bd);
	}

	return 0;
}

int buffer_seek(struct buffer* buffer, size_t offset, int seek)
{
	return 0;
}
int buffer_peek(struct buffer* buffer, void* data, size_t size, size_t offset)
{
	return 0;
}

int buffer_copy(struct buffer* dst, struct buffer* src) 
{
	char   buff[4096];
	size_t total = 0;
	int    nread, nwrote;

	nread = buffer_read(src, buff, sizeof(buff));
	while (nread > 0) {
		nwrote = buffer_write(dst, buff, nread);
		if (nwrote != nread) 
			return -1;
	
		total += nread;
		nread = buffer_read(src, buff, sizeof(buff));
	}

	return total;
}

