#define _GNU_SOURCE
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>

#include "cfg_parser.h"

static char* strip(char* str, int blank)
{
	int len = strlen(str);
	int j = len - 1;

	while (j >= 0)
	{
		if ((blank && isblank(str[j])) || !isprint(str[j]))
		{
			--j;
			continue;
		}
		str[j + 1] = 0;
		len = j + 1;
		break;
	}
	j = 0;
	while (j < len)
	{
		if ((blank && isblank(str[j])) || !isprint(str[j]))
		{
			++j;
			continue;
		}
		len -= j;
		break;
	}
	memmove(str, str + j, len + 1);
	return str;
}

cfg_t* cfg_add(cfg_t* node, const char* key, const char* value)
{
	cfg_t* new = NULL;

	if (!(new = calloc(1, sizeof(*new))))
		return NULL;

	if (node != NULL) {
		new->next = node->next;
		node->next = new;
	}

	//XXX check for strdup
	new->key = strdup(key);
	new->value = strdup(value);

	strip(new->key, 1);
	strip(new->value, 0);

	return new;
}

cfg_t* parse_stream(FILE* file)
{
	char buf[1025];
	char* pos;
	cfg_t* entry = NULL;
	cfg_t* res = NULL;
	int len;

	if (!file)
		return NULL;

	while (!feof(file))
	{
		memset(buf, 0, sizeof(buf));
		if (!fgets(buf, sizeof(buf), file))
			break;
		if (*buf == '#' || *buf == '\n' || *buf == '\r' || !(pos = strchr(buf, '=')))
			continue;
		len = pos - buf;
		buf[len] = '\0';
		if (!(entry = cfg_add(res, buf, pos + 1)))
			continue;
		if (!res)
			res = entry;
	}
	return res;
}

cfg_t* cfg_parse(const char* file_name)
{
	FILE* f = fopen(file_name, "r");
	cfg_t* cfg = NULL;

	if (!f)
		return NULL;

	cfg = parse_stream(f);
	fclose(f);

	return cfg;
}

void free_cfg(cfg_t* cfg)
{
	cfg_t* n;
	while (cfg)
	{
		n = cfg->next;
		free(cfg->key);
		free(cfg->value);
		free(cfg);
		cfg = n;
	}
}

int get_cfg_key_idx(const char* key, int pos)
{
	int p = 1;
	int idx;

	while (key && (key = strchr(key, '.')))
	{
		errno = 0;
		idx = strtol(key + 1, NULL, 0);
		if (errno)
			continue;
		if (p == pos)
			return idx;
		++p;
	}
	return -1;
}

static const char* vget_value(const cfg_t* cfg, const char* default_val, const char* key_frm, va_list ap)
{
	//XXX use dynamic allocation to accept any length
	char key[256] = "";

	vsnprintf(key, sizeof(key), key_frm, ap);
	while (cfg)
	{
		if (!strcmp(cfg->key, key))
		{
			return cfg->value;
		}
		cfg = cfg->next;
	}
	return default_val;
}

const char* cfg_get_value(const cfg_t* cfg, const char* default_val, const char* key_frm, ...)
{
	va_list ap;
	const char* ret;

	va_start(ap, key_frm);
	ret = vget_value(cfg, default_val, key_frm, ap);
	va_end(ap);

	return ret;
}
