#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include "php.h"
#include "parse.h"

typedef struct dict_entry {
        char* key;
        char* word;
} dict_entry_t;

static dict_entry_t* dict_map = NULL;
static int dict_size = 0;
static int reserved_size = 0;
static int hilite_missing = 0;

static const char *find_word(const char* key, size_t key_len)
{
	int i;
	for (i = 0; i < dict_size; ++i)
	{
		if (strlen(dict_map[i].key) == key_len && 
			strncmp(dict_map[i].key, key, key_len) == 0)
		{
			return dict_map[i].word;
		}		
	}
	return NULL;
}

static void add_entry(const char* key, const char* word)
{
	void *tmp;
	size_t tmp_len;
	if (reserved_size <= dict_size)
	{
		tmp_len = sizeof(dict_entry_t) * (reserved_size + 100);
		tmp = realloc(dict_map, tmp_len);
		if (!tmp)
		{
			return;
		}			
		dict_map = (dict_entry_t*)tmp;
		reserved_size += 100;			
	}
	dict_map[dict_size].key = strdup(key); 
	dict_map[dict_size].word = strdup(word); 
	dict_size++;
}

#define LINE_BUF 4096

static int load_dictionary_file(const char* file)
{
	FILE *f;
	char *b, *key, *value;
	char buf[LINE_BUF];
	int count = 0;

	if ((f = fopen(file, "r")) == NULL)
	{
		return -1;
	}

	while (fgets(buf, sizeof(buf), f) != NULL)
	{
		b = buf;
		if (*b == '#')
			continue; /* skip comments */

		while (isspace(*b))
		{
			b++; /* skip leading spaces */
		}
		
		if (!*b)
		{
			continue;
		}
		
		key = b;
		if ((b = strstr(key, "=>")) == NULL || b == key)
		{
			continue; /* separator not found or key is empty string */
		}

		value = b + 2;
		if (!*value)
		{
			continue;
		}
		
		for (b--; isspace(*b); b--)
		; /* Note empty for */
		b[1] = '\0'; /* remove trailing spaces */
		
		while (isspace(*value))
		{
			value++; /* skip leading spaces */
		}

		if (!*value)
		{
			continue;
		}

		for (b = value + strlen(value) - 1; isspace(*b); b--);
		; /* Note empty for */
		b[1] = '\0'; /* remove trailing spaces */
		
		add_entry(key, value);
		++count;
	}

	fclose(f);
	return count;
}

void TranslateCleanDictionary(void)
{
	int i;
	for (i = 0; i < dict_size; ++i)
	{
		free(dict_map[i].key);
		free(dict_map[i].word);
	}
	free(dict_map);
	dict_size = 0;
	reserved_size = 0;
}

void Translate(void)
{
	Stack *s;
	char *def, *p;
	const char *word;

	s = Pop();
	if (!s)
	{
		Error("Stack Error in Translate");
		return;
	}
	
	def = s->strval;
	if (((p = strchr(s->strval, '|')) != NULL)
		|| ((p = strchr(s->strval, '#')) != NULL))
	{
		def = p + 1;
	}
	
	word = find_word(s->strval, (p ? p - s->strval: strlen(s->strval)));
	if (word)
	{		
		Push((char*)word, STRING);
	}
	else
	{
		if (hilite_missing)
		{
			static char missing[] = "<strong>#####MISSING#####</strong> ";
			p = emalloc(1, sizeof(missing) + strlen(def));
			strcpy(p, def);
			strcat(p, missing);
			def = p;
		}
		Push(def, STRING);
	}
}

void TranslateHiliteMissing(void)
{
	char temp[8];
	Stack *s;
	s = Pop();
	if (!s)
	{
		Error("Stack Error in TranslateHiliteMissing");
		return;
	}
	snprintf(temp, sizeof(temp), "%d", hilite_missing);
	hilite_missing = (s->intval ? 1 : 0);
	Push(temp, LNUMBER);
}

void TranslateLoadDictionary(void)
{
	char temp[8];
	Stack *s;
	s = Pop();
	if (!s)
	{
		Error("Stack Error in TranslateLoadDictionary");
		return;
	}
	snprintf(temp, sizeof(temp), "%d", load_dictionary_file(s->strval));
	Push(temp, LNUMBER);
}
