#define _GNU_SOURCE
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

// #include "header.h"
#include "buffer.h"
#include "uploadbuffer.h"
//#include "liblogger/logger.h"
#include "utils.h"

#define CONTENT_DISPHDR "Content-Disposition"
#define CONTENT_TYPEHDR "Content-Type"

static int parameter_add_file(struct list_head* head,
			      char* name, char* value,
			      int is_file, size_t size)
{
	struct header* h = (struct header*) calloc(1, sizeof(*h));
	if (!h) return -1;

	INIT_LIST_HEAD(&h->list);
	h->name = name;
	h->value = value;
	h->is_file = is_file;
	h->filesize = size;
	list_add_tail(&h->list, head);

	return 0;
}

static int parameter_add(struct list_head* head,
			 char* name, char* value)
{
	return parameter_add_file(head, name, value, 0, 0);
}

int parameter_remove_all(struct list_head* head)
{
	struct list_head *pos, *safe;
	struct header* h;

	if (list_empty(head)) return 0;

	list_for_each_safe(pos, safe, head) {
		h = list_entry(pos, struct header, list);
		list_del(&h->list);

		if (h->is_file && access(h->value, W_OK) == 0) {
			unlink(h->value);
		}

		free(h->name);
		free(h->value);
		free(h);
	}

	return 0;
}

static int ub_normalize_filename(char* filename)
{

    char *fwd_slash;
    char *back_slash;
    size_t size;
    /**
     *  normalize uploaded file name. In particular windows
     *  browsers tend to give a full file path in C:\dir\filename.txt
     *  format ... leave only filename.txt. in case any unix browser
     *  does similar thing  - strip path, leaving only uploaded file
     *  name.
     * CORRECTION:
     * strip only windows path part, leaving our local filesystem path
     * strip everything between (last '/') and (last '\'), but leaving everything else the same
     * Example:
     * /tmp/uhttpd/upload/C:\dir\filename.txt normalizes to /tmp/uhttpd/upload/filename.txt
     */
    if (filename == 0)
    {
        return -1;
    }
    size = strlen(filename);
    fwd_slash = strrchr(filename, '/');
    back_slash = strrchr(filename, '\\');
    if ((fwd_slash == 0) ||
	(back_slash == 0))
    {
	    /* No slashes or backslashes */
        return 0;
    }
    if ((back_slash - filename == size - 1) ||
	(fwd_slash - filename == size - 1))
    {
        return -1; /* 'Empty filename ???' */
    }
    /* Move file part from last backslash to last forwardslash,
       Copy including ending string \0 */
    memmove(fwd_slash + 1, back_slash + 1, size - (back_slash - filename));
    return 0;
}

static char* trim(char* ptr)
{
	char *begin, *end;

	if (!ptr) return 0;
	if (strlen(ptr) == 0) return ptr;

	begin = ptr;
	end = ptr + strlen(ptr);
	while (*begin == ' ' && begin < end) begin ++;
	while (*end   == ' ' && begin < ptr) end --;
	*end = 0;

	return begin;
}

static int ub_init_writeout(struct upload_buffer* ub)
{
	ub->ignore = 0;

	switch (ub->state) {
	case ST_PARAMETER:
		if (ub->content) DBG_WARN("ub_init_writeout: buffer was already created ?\n");

		if (buffer_create(&ub->content))
			return -1;
		break;
	case ST_FILE:
		ub_normalize_filename(ub->filename);
		ub->fd = open(ub->filename, O_CREAT|O_TRUNC|O_WRONLY, 0600);
		if (ub->fd < 0) {
			// error_errno("can't open file '%s'\n", ub->filename);
			ub->ignore = 1;
			// return -1;
		}
		break;
	default:
		DBG_ERROR("ub_init_writeout: Invalid state %d\n on writeout init.\n", ub->state);
		return -1;
	};
	return 0;
}

static int ub_do_writeout(struct upload_buffer* ub, const void* ptr, size_t size)
{
	int nwrote = -1;

	switch (ub->state) {
	case ST_PARAMETER:
		return buffer_write(ub->content, ptr, size);
	case ST_FILE:
		if (ub->ignore) {
			ub->filesize += nwrote;
			return size;
		} else {
			nwrote = write(ub->fd, ptr, size);
			if (nwrote > 0) ub->filesize += nwrote;
			return nwrote;
		}

	default:
		DBG_ERROR("ub_do_writeout: Invalid state %d\n on writeout.\n", ub->state);
	};
	return 0;
}

static int ub_finish_writeout(struct upload_buffer* ub)
{
	char* content;
	size_t size;

	switch (ub->state)
	{
	case ST_PARAMETER:
		size = ub->content->size;
		if (size > 0) {
			content = malloc(size+1);
			if (!content) return -1;
			if (buffer_read(ub->content, content, size) != size)
				return -1;
			content[size] = 0;
		} else {
			content = strdup("");
		}

		if (parameter_add(&ub->parameters, ub->parameter, content))
			return -1;

		break;
	case ST_FILE:
		if (ub->ignore) {
			free(ub->filename);
			free(ub->parameter);
			break;
		}

		if (parameter_add_file(&ub->parameters,
				  strdup(ub->parameter),
				  strdup(ub->filename), 1, ub->filesize))
			return -1;

		close(ub->fd);
		ub->fd = -1;
		break;
	default:
		DBG_ERROR("ub_finish_writeout: Invalid state %d\n on writeout finish.\n", ub->state);
	};

	buffer_delete(ub->content);
	ub->content = 0;
	ub->filesize = 0;
	ub->filename = 0;
	ub->parameter = 0;
	ub->fd = -1;
	ub->state = ST_UNKNOWN;

	return 0;
}

static int ub_parse_disposition(const struct upload_buffer* ub,
				char *begin, char* end,
				char** name, char** filename)
{
	char* bq, *eq;	/* begin and end of quotes. */
	char* ptr = begin;
	char* token;
	size_t size;

	while ((token = strsep(&ptr, ";"))) {
		token = trim(token);

		if (strncmp(token, CONTENT_DISPHDR,
			    sizeof(CONTENT_DISPHDR)-1) == 0) {
			if (strncmp(token + strlen(token) - 9, "form-data", 9) != 0)
				DBG_WARN("ub_parse_disposition: ingoring '%s'\n", token);

		} else if (strncmp(token, "filename", 8) == 0) {
			bq = strchr(token+8, '"');
			eq = strrchr(token+8, '"');

			if (bq == 0 ||
			    eq == 0 ||
			    bq == eq) {
				DBG_ERROR("ub_parse_disposition: token: %s, missing quotes in filename=\"...\"\n",
				      token);
				return -1;
			}
			*eq=0;	/* terminate string on '"' */
			++ bq;  /* skip '"' too ... */

			if (eq - bq == 0) {
				*filename = 0;
				continue;
			}
			size = 10;
			size += strlen(ub->upload_path);
			size += strlen(bq);

			*filename = malloc(size+1);
			if (!*filename) return -1;

			strcpy(*filename, ub->upload_path);
			strcat(*filename, "/");
			strcat(*filename, bq);
		} else if (strncmp(token, "name", 4) == 0) {
			bq = strchr(token + 4, '"');
			eq = strrchr(token + 4, '"');

			if (bq == 0 ||
			    eq == 0 ||
			    bq == eq) {
				DBG_ERROR("ub_parse_disposition: token: %s, missing quotes in name=\"...\"\n",
				      token);
				return -1;
			}

			*eq=0;
			++ bq ;

			if (eq - bq == 0) {
				*filename = 0;
				continue;
			}

			*name = strdup(bq);
		} else {
			DBG_WARN("ub_parse_disposition: ignoring unknown token '%s'\n", token);
		}
	}

	return 0;
}

static int ub_parse_type(char* begin, char* end)
{
	/* leave this function empty for now. we don't care about
	 * content type, as we can't do much about it now. */
	return 0;
}

/**
 * Parses content headers, to find out content type.
 * If it is a POST parameter, ST_PARAMETER state is set,
 * and parameter content is stored in 'content' buffer.
 *
 * If 'content' is a file - ST_FILE is set and content
 * is written out to a specified config->upload_directory
 * under the same file name as specified in content headers.
 *
 * @return 0, success, otherwise failure.
 */
static int ub_parse_headers(const struct upload_buffer* ub,
			    char* line, size_t size,
			    char** name, char** filename)
{
	size_t length;
	char* end;
	char* ptr = line;

	end = (char*)memmem(ptr, size, "\r\n", 2);
	if (!end) return -1;

	length = end - ptr;
	if (length > 0) {
		*end = 0;
		if (ub_parse_disposition(ub, ptr, end, name, filename)) {
			DBG_ERROR("ub_parse_headers: Invalid header: '%s'\n", ptr);
			return -1;
		}
	}

	ptr = end + 2;
	size -= length + 2;

	end = (char*)memmem(ptr, size, "\r\n", 2);
	if (!end) {
		DBG_WARN("ub_parse_headers: no content type present ?\n");
	}

	length = end - ptr;
	if (length > 0) {
		*end = 0;
		if (ub_parse_type(ptr, end)) {
			// debug("Invalid content type header: '%s'\n", ptr);
			return -1;
		}

		ptr = end + 2;
		size -= length + 2;
	}

	if (size != 2      ||
	    ptr[0] != '\r' ||
	    ptr[1] != '\n')
		DBG_WARN("ub_parse_headers: unknown bytes %zu left, content: '%s'\n", size, ptr);

	return 0;
}


int ub_create(struct upload_buffer** ub)
{
	struct upload_buffer *buf;

	buf = malloc(sizeof(*buf));
	if (!buf) return -1;

	memset(buf, 0, sizeof(*buf));
	/* prevent fd number '0' to be closed in ub_free() */
	buf->fd = -1;
	*ub = buf;
	return 0;
}

/*
 * TODO: there some members in structure never free(3) ...
 */
void ub_free(struct upload_buffer* ub)
{
	if (!ub) return;

	free(ub->buffer);
	free(ub->boundary);
	free(ub->upload_path);
	buffer_delete(ub->content);

	parameter_remove_all(&ub->parameters);
	if (ub->fd >= 0) {
		close(ub->fd);
		ub->fd = -1;
	}

	if (ub->filename) {
		unlink(ub->filename);
	}

	free(ub->parameter);
	free(ub->filename);
	free(ub);
}


int ub_init(struct upload_buffer* ub,
	    size_t expected,
	    const char* boundary,
	    const char* upload)
{
	size_t size;

	memset(ub, 0, sizeof(*ub));
	/* make sure fd is not zero, prevents /dev/null closing by accident */
	ub->fd = -1;

	ub->buffer = malloc(MAX_UPLOADBUFFER);
	if (!ub->buffer) return -1;

	ub->size = MAX_UPLOADBUFFER;

	if (boundary) {
		size = strlen(boundary) + 2;
		ub->boundary = malloc(size+1);
		if (!ub->boundary) return -1;
		strcpy(ub->boundary, "--");
		strcat(ub->boundary, boundary);
		ub->length = size;
	}
	ub->expected = expected;
	ub->state = ST_UNKNOWN;
	ub->upload_path = strdup(upload);
	{
		struct stat st;
		if (stat(ub->upload_path, &st) < 0) {
			mkdir(ub->upload_path, 0755);
		} else if (!S_ISDIR(st.st_mode)) {
			unlink(ub->upload_path);
			mkdir(ub->upload_path, 0755);
		}
	}
	INIT_LIST_HEAD(&ub->parameters);

	return 0;
}

int ub_process_content(struct upload_buffer* ub)
{
	char  line[512];
	size_t length;
	size_t offset = 0;

	char  *eol;
	char  *ptr = ub->buffer;
	char  *end = ub->buffer + ub->data;

	size_t size = ub->data;

	while (ptr < end) {
		if (size < ub->length)
			break;

		if ((~ub->got) & GOT_BOUNDARY) {
			eol = (char*)memmem(ptr, size, "\r\n", 2);
			if (!eol) break;

			length = eol - ptr + 2;
			memcpy(line, ptr, length);
			line[length] = 0;

			if (strncmp(ub->boundary, line, ub->length) == 0) {
				ub->got |= GOT_BOUNDARY;

				/* check if boundary has trailing '--' -
				 * which is the end of POST request.
				 */
				if (length > ub->length &&
				    line[ub->length+0] == '-' &&
				    line[ub->length+1] == '-') {
					ub->got |= GOT_ENDBOUNDARY;
					return 0;
				}
			} else  {
				return -1;
			}

			ptr = eol + 2;
			size -= length;
			offset += length;

			continue;
		}

		if ((~ub->got) & GOT_HEADER) {
			eol = (char*)memmem(ptr, size, "\r\n\r\n", 4);
			if (!eol) break;

			length = eol - ptr + 4;
			memcpy(line, ptr, length);
			line[length] = 0;

			if (ub_parse_headers(ub, line, length, &ub->parameter, &ub->filename))
				return -1;

			if (ub->parameter && strlen(ub->parameter)> 0) {
				ub->state = ST_PARAMETER;
			} else {
				/* name can't not be empty ! */
				return -1;
			}

			if (ub->filename && strlen(ub->filename) > 0) {
				ub->state = ST_FILE;
			}

			if (ub_init_writeout(ub))
				return -1;

			ub->got |= GOT_HEADER;

			ptr = eol + 4;
			size -= length;
			offset += length;
			continue;
		}

		if ((ub->got & (GOT_HEADER|GOT_BOUNDARY)) == (GOT_HEADER|GOT_BOUNDARY))
		{
			// if (size < ub->length) return 0;
			eol = (char*)memmem(ptr, size, ub->boundary, ub->length);
			if (!eol) {
				size_t left = end - ptr;

				if (left > ub-> length) {
					left -= ub->length;
				}

				if (ub_do_writeout(ub, ptr, left) != left)
					return -1;

				// if (buffer_write(ub->content, ptr, left) != left)
				//	return -1;

				ptr = ptr + left;
				break;
			}

			length = eol - ptr;

			/* strip extra '\r\n' */
			if (ub_do_writeout(ub, ptr, length -2) != length - 2)
				return -1;


			if (ub_finish_writeout(ub))
				return -1;

			// if (buffer_write(ub->content, ptr, length) != length)
			//	return -1;

			offset += length;
			size   -= length;
			ub->got = 0;
			ptr = eol;
			continue;
		}
	}

	if (ptr != ub->buffer) {
		size = end - ptr;
		ub->data = size;
		memmove(ub->buffer, ptr, ub->data);
	}
	return 0;
}

static int ub_write_post(struct upload_buffer* ub, const void* buffer, size_t size)
{
	size_t uleft, bleft, nleft;

	uleft = ub->size - ub->data;	/* ub capacity left */
	bleft = size;			/* bytes left in buffer */
	nleft = (uleft > bleft) ? bleft : uleft;

	memcpy(ub->buffer+ub->data, buffer, nleft);

	ub->data += nleft;
	ub->total += nleft;

	if (ub_process_content(ub))
		return -1;

	return nleft;
}

static int ub_parse_urlencoded(struct list_head* parameters, char* query, size_t size)
{
	char *token, *value;
	char *ptr = query;

	while ((token=strsep(&ptr, "&"))) {
		value = strchr(token, '=');
		if (!value) return -1;
		*value = 0;
		++ value;

		decode_urlencoded(token, &token);
		decode_urlencoded(value, &value);

		if (parameter_add(parameters, token, value))
			return -1;
	}

	return 0;
}

static int ub_write_urlencoded(struct upload_buffer* ub, const void* buffer, size_t size)
{
	/* make sure that ub->content is created, it there we store
	 * url-encoded-form-data */
	if (!ub->boundary && !ub->content) {
		if (buffer_create(&ub->content))
			return -1;
	}

	if (buffer_write(ub->content, buffer, size) != size) {
		// debug("error writing bytes ...\n");
		return -1;
	}

	ub->total += size;

	if (ub->total > ub->expected)
		DBG_WARN("ub_write_urlencoded: too much data got.\n");

	if (ub->total >= ub->expected) {
		size_t urlsize = ub->content->size;
		char* urlencoded = malloc(urlsize+1);

		if (!urlencoded) return -1;
		buffer_read(ub->content, urlencoded, ub->content->size);
		urlencoded[urlsize] = 0;

		if (ub_parse_urlencoded(&ub->parameters,
					urlencoded, urlsize))
			DBG_WARN("ub_write_urlencoded: invalid URL encoded parameters passed.\n");
		free(urlencoded);
	}

	return size;
}

int ub_write(struct upload_buffer* ub, const void* buffer, size_t size)
{
	size_t nwritten = 0;
	int nwrote;

	/* check parameters before procceeding. */
	if (!size) {
		// debug("ub_write: zero bytes write requested.\n");
		return 0;
	}

	do {
		if (ub->boundary) {
			nwrote = ub_write_post(ub,
					       buffer + nwritten,
					       size   - nwritten);
			if (nwrote < 0) {
				// debug("error writing POST content bytes ... \n");
				return -1;
			}

			nwritten += nwrote;
	 	} else {
			nwrote = ub_write_urlencoded(ub,
						     buffer + nwritten,
						     size   - nwritten);

			if (nwrote != size) {
				// debug("error writing bytes ...\n");
				return -1;
			}

			// debug("wrote %d bytes to content ...\n", size);
			nwritten  += nwrote;
		}

	} while(nwritten < size);

	return nwritten;
}

#if 0
int main(int argc, char* argv[])
{
	char buffer[4096*2];
	struct upload_buffer* ub;
	int fd, nread, nwrote;


	fd = open("content.dump", O_RDONLY);
	if (fd < 0)
		return -1;

#define BOUNDARY "---------------------------2519993291436235669941269728"
	if (ub_create(&ub))
		return -1;

	if (ub_init(ub, BOUNDARY, "/tmp/upload"))
		return -1;

	nread = read(fd, buffer, sizeof(buffer));
	while (nread > 0) {
		nwrote = ub_write(ub, buffer, nread);
		if (nwrote != nread) {
			DBG_ERROR("Error : nwrote: %d, nread: %d\n", nwrote, nread);
			break;
		}
		nread = read(fd, buffer, sizeof(buffer));
	}


	close(fd);

	ub_free(ub);
	return 0;
}
#endif
