#include <stdio.h>
#include <string.h>

#define MAX_LINE 255

#define MAC_LEN		18
#define ESSID_LEN	33
#define MODE_LEN	10
#define FREQ_LEN	10
#define CHAN_LEN	4
#define QUAL_LEN	6
#define LEVEL_LEN	9
#define ENC_LEN		5

/* this fragment helps to determine where the new entry starts */
static const char* const entry_start = "          Cell ";
static const char* const essid_start = "                    ESSID:";
static const char* const mode_start  = "                    Mode:";
static const char* const freq_start  = "                    Frequency:";
static const char* const chan_start  = "(Channel ";
static const char* const qual_start  = "                    Quality=";
static const char* const encr_start  = "                    Encryption key:";
static const char* const wpa_start   = "                    Extra:wpa_ie=";
static const char* const wpa2_start  = "                    Extra:rsn_ie=";

static const char* const freq_pattern = "                    Frequency:%s GHz (Channel %s)";
static const char* const qual_pattern = "                    Quality=%s  Signal level=%s dBm  Noise level=%s dBm";
static const char* const encr_pattern = "                    Encryption key:%s";

static const char separator = '#';

typedef struct scan_info {
	char	mac[MAC_LEN];
	char	essid[ESSID_LEN];
	char	mode[MODE_LEN];
	char	frequency[FREQ_LEN];
	char	channel[CHAN_LEN];
	char	quality[QUAL_LEN];
	char	signal_level[LEVEL_LEN];
	char	noise_level[LEVEL_LEN];
	char	encryption[ENC_LEN]; /* none, wep, wpa, wpa2 */
} scan_info_t;

/* TODO: description */
typedef int (*line_parser)(const char* const line, scan_info_t* sinfo);

/* forward declarations */
static int mac_parser        (const char* const line, scan_info_t* sinfo);
static int essid_parser      (const char* const line, scan_info_t* sinfo);
static int mode_parser       (const char* const line, scan_info_t* sinfo);
static int frequency_parser  (const char* const line, scan_info_t* sinfo);
static int quality_parser    (const char* const line, scan_info_t* sinfo);
static int encryption_parser (const char* const line, scan_info_t* sinfo);
static int wpa_parser        (const char* const line, scan_info_t* sinfo);
static int wpa2_parser       (const char* const line, scan_info_t* sinfo);

line_parser parser[] = {
	mac_parser,
	essid_parser,
	mode_parser,
	frequency_parser,
	quality_parser,
	encryption_parser,
	wpa_parser,
	wpa2_parser,
	0
};

/* returns 1 if line begins with given pattern, 0 otherwise */
static int begins_with(const char* const line, const char* const pattern)
{
	char* pos = strstr(line, pattern);
	if (pos == line)
		return 1;

	return 0;
}

static void print_entry(scan_info_t* sinfo)
{
	printf("%s", sinfo->mac);
	printf("%c%s", separator, sinfo->mode);
	printf("%c%s", separator, sinfo->frequency);
	printf("%c%s", separator, sinfo->channel);
	printf("%c%s", separator, sinfo->quality);
	printf("%c%s", separator, sinfo->signal_level);
	printf("%c%s", separator, sinfo->noise_level);
	printf("%c%s", separator, sinfo->encryption);
	printf("%c\"%s\"", separator, sinfo->essid);
	printf("\n");
}

static int parse_scan(void)
{
	FILE* scan_file = stdin;
	char line[MAX_LINE];
	scan_info_t sinfo;
	memset(&sinfo, 0, sizeof(sinfo));

	while (fgets(line, MAX_LINE, scan_file)) {
		int i = 0, processed = 0;

		/* process line now */
		if (begins_with(line, entry_start) != 0) {
			/* new item. save old one (if any) and reset */
			if (strlen(sinfo.mode) > 0) {
				print_entry(&sinfo);
			}

			/* cleanup now */
			memset(&sinfo, 0, sizeof(sinfo));
		}

		while (parser[i] != 0 && !processed) {
			processed = parser[i++](line, &sinfo);
		}
	}

	/* last one */
	if (strlen(sinfo.mode) > 0) {
		print_entry(&sinfo);
	}

	return 0;
}

int main (int argc, char *argv[])
{
	if (parse_scan()) {
		return -1;
	}

	return 0;
}

/*----------------------------------------------------------------------------*/
static int mac_parser(const char* const line, scan_info_t* sinfo)
{
	if (!line || !sinfo ||
	    begins_with(line, entry_start) == 0)
		return 0;

	strncpy(sinfo->mac, line + 29, sizeof(sinfo->mac) - 1);

	return 1;
}

static int essid_parser(const char* const line, scan_info_t* sinfo)
{
	int line_length = 0;

	if (!line || !sinfo ||
	    begins_with(line, essid_start) == 0)
		return 0;

	line_length = strlen(line);
	if (line_length > strlen(essid_start) + 2) {
		int j = 0, idx = strlen(essid_start) + 1;
		while (idx < line_length && line[idx] != '"') {
			sinfo->essid[j++] = line[idx];
			idx++;
		}
	}

	return 1;
}

static int mode_parser(const char* const line, scan_info_t* sinfo)
{
	int line_length = 0, pattern_length = 0;

	if (!line || !sinfo ||
	    begins_with(line, mode_start) == 0)
		return 0;

	line_length = strlen(line);
	pattern_length = strlen(mode_start);
	if (line_length > pattern_length) {
		strncpy(sinfo->mode,
			line + pattern_length,
			line_length - pattern_length - 1);
	}

	return 1;
}

static int frequency_parser(const char* const line, scan_info_t* sinfo)
{
	char* pos = 0;
	int i = 0;

	if (!line || !sinfo ||
	    begins_with(line, freq_start) == 0)
		return 0;

	/* get frequency */
	pos = strstr(line, freq_start);
	pos += strlen(freq_start);
	while ((pos[i] != ' ') && (i < FREQ_LEN)) {
		sinfo->frequency[i] = pos[i];
		++i;
	}

	/* channel */
	pos = strstr(line, chan_start);
	if (pos) {
		pos += strlen(chan_start);
		i = 0;
		while ((i < CHAN_LEN) && (pos[i] != ')') && (pos[i] != '\0')) {
			sinfo->channel[i] = pos[i];
			++i;
		}
	} else {
	    	/* XXX: workaround for 13/14 channel if supported channel list from driver is > IW_MAX_FREQUENCIES (aka > 32)
                   Actualy we have that issue on LS2 in Complianse Test country code.
		   actualy we have to do something like that for all frequencies that are out of IW_MAX_FREQUENCIES */
	    	if (!strncmp(sinfo->frequency, "2.484", 5))
                	strcpy(sinfo->channel, "14");
	    	else if (!strncmp(sinfo->frequency, "2.472", 5))
                	strcpy(sinfo->channel, "13");
	}
	return 1;
}

static int quality_parser(const char* const line, scan_info_t* sinfo)
{
	if (!line || !sinfo ||
	    begins_with(line, qual_start) == 0)
		return 0;

	sscanf(line, qual_pattern, sinfo->quality, sinfo->signal_level, sinfo->noise_level);

	return 1;
}

static int encryption_parser(const char* const line, scan_info_t* sinfo)
{
	char enc_status[4];

	if (!line || !sinfo ||
	    begins_with(line, encr_start) == 0)
		return 0;

	sscanf(line, encr_pattern, enc_status);
	if (strcmp(enc_status, "on") == 0) {
		strcpy(sinfo->encryption, "wep");
	}
	else {
		strcpy(sinfo->encryption, "none");
	}


	return 1;
}

static int wpa_parser(const char* const line, scan_info_t* sinfo)
{
	if (!line || !sinfo ||
	    begins_with(line, wpa_start) == 0)
		return 0;

	strcpy(sinfo->encryption, "wpa");

	return 1;
}

static int wpa2_parser(const char* const line, scan_info_t* sinfo)
{
	if (!line || !sinfo ||
	    begins_with(line, wpa2_start) == 0)
		return 0;

	strcpy(sinfo->encryption, "wpa2");

	return 1;
}

