
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stddef.h>
#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>

#include <ber/ber.h>
#include <debug/memory.h>
#include <debug/log.h>
#include <abz/typedefs.h>
#include <abz/error.h>

#include <tinysnmp/tinysnmp.h>
#include <tinysnmp/agent/module.h>
#include <tinysnmp/agent/odb.h>
#include <tinysnmp/agent/ifcache.h>

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>		/* For AF_INET & struct sockaddr */
#include <linux/if.h>			/* for IFNAMSIZ and co... */
//USE local copy of wireless.h
#include "wireless.h"

#ifndef __packed
#define	__packed	__attribute__((__packed__))
#endif

//#include "net80211/ieee80211.h"
//#include "net80211/ieee80211_crypto.h"
#include "net80211/ieee80211_ioctl.h"

static int
update(struct odb **odb, const uint32_t *oid, uint8_t type, void *data)
{
    snmp_value_t value;

    value.type = type;

    switch (type) {
    case BER_INTEGER:
		value.data.INTEGER = *(int32_t *) data;
		break;
    case BER_Counter32:
		value.data.Counter32 = *(uint32_t *) data;
		break;
    case BER_Gauge32:
		value.data.Gauge32 = *(uint32_t *) data;
		break;
    case BER_TimeTicks:
		value.data.TimeTicks = *(uint32_t *) data;
		break;
    case BER_Counter64:
		value.data.Counter64 = *(uint64_t *) data;
		break;
    case BER_OID:
		value.data.OID = (uint32_t *) data;
		break;
    case BER_OCTET_STRING:
		value.data.OCTET_STRING = *(octet_string_t *) data;
		break;
    default:
		abz_set_error ("invalid type (0x%02x) specified",type);
		return (-1);
    }

    return (odb_add(odb, oid, &value));
}

/*
 * Open a socket (taken from iwlib.c)
 * Depending on the protocol present, open the right socket. The socket
 * will allow us to talk to the driver.
 */
static int
open_socket(void)
{
    static const int families[] = {
		AF_INET, AF_IPX, AF_AX25, AF_APPLETALK
    };
    unsigned int  i;
    int   sock;

    /*
     * Now pick any (exisiting) useful socket family for generic queries
     * Note : don't open all the socket, only returns when one matches,
     * all protocols might not be valid.
     * Workaround by Jim Kaba <jkaba@sarnoff.com>
     * Note : in 99% of the case, we will just open the inet_sock.
     * The remaining 1% case are not fully correct...
     */

    /* Try all families we support */
    for(i = 0; i < sizeof(families)/sizeof(int); ++i) {
		/* Try to open the socket, if success returns it */
		sock = socket(families[i], SOCK_DGRAM, 0);
		if(sock >= 0)
		    return sock;
    }

    return -1;
}


#define	STAINFO_BUF_SIZE	24 * 1024

static void
mtik_wireless_station_update(struct odb **odb, int skfd, char *ifname, int ifidx)
{
    struct iwreq	iwr;                       // ioctl request structure
    uint8_t			bfr[34];
	struct iw_statistics	iwstats;
	uint32_t 		oid[14] = {13, 43, 6, 1, 4, 1, 14988, 1, 1, 1, 1, 1, 1, 0};
	octet_string_t	str;

	int len;
	struct ieee80211req_sta_info *si;
	u_int8_t *cp, *buf;

	oid[13] = ifidx;
	strncpy(iwr.ifr_name, ifname, IFNAMSIZ);

	// At first check if am working in STA mode
    if ((ioctl(skfd, SIOCGIWMODE, &iwr) < 0) || (iwr.u.mode != IW_MODE_INFRA))
		return;


    // Get ESSID
    iwr.u.essid.pointer = (caddr_t) &bfr;
	memset(&bfr, 0, sizeof(bfr));
    iwr.u.essid.length = IW_ESSID_MAX_SIZE + 1;
    iwr.u.essid.flags = 0;
    if (ioctl(skfd, SIOCGIWESSID, &iwr) >= 0 ) {
		oid[12] = 5;
		str.len = iwr.u.essid.length;
		str.buf = bfr;
		update(odb, oid, BER_OCTET_STRING, &str);
	}

    // Get frequency/channel
    if ( ioctl ( skfd, SIOCGIWFREQ, &iwr ) >= 0 ) {
		uint32_t freq = iwr.u.freq.m;
		while (freq > 9999)
	    	freq /= 10;
		oid[12] = 7;
		update(odb, oid, BER_INTEGER, &freq);
    }

	// Get BSSID address
    if ( ioctl ( skfd, SIOCGIWAP, &iwr ) >= 0 ) {
		int found = 0;
		oid[12] = 6;
		str.len = IFHWADDRLEN;
		str.buf = bfr;
		memcpy(bfr, iwr.u.ap_addr.sa_data, IFHWADDRLEN);
		update(odb, oid, BER_OCTET_STRING, &str);

		// I'll try to get all the rest info via single ioctl
		buf = malloc(STAINFO_BUF_SIZE);
		iwr.u.data.pointer = (void*)buf;
		iwr.u.data.length = STAINFO_BUF_SIZE;
		if (ioctl(skfd, IEEE80211_IOCTL_STA_INFO, &iwr) >= 0) {
			len = iwr.u.data.length;
			if (len >= sizeof(struct ieee80211req_sta_info)) {
				cp = buf;
				do {
					si = (struct ieee80211req_sta_info *) cp;
					// find maching ap
					if (memcmp(bfr, si->isi_macaddr, IFHWADDRLEN) == 0) {
						int tmp = si->isi_noisefloor + si->isi_rssi;
						oid[12] = 4;
						update(odb, oid, BER_INTEGER, &tmp);

						oid[12] = 2;
						tmp = (si->isi_rates[si->isi_txrate] & 0x7F) / 2 * 1000000;
						update(odb, oid, BER_Gauge32, &tmp);

						oid[12] = 3;
						tmp = (si->isi_rates[si->isi_rxrate] & 0x7F) / 2 * 1000000;
						update(odb, oid, BER_Gauge32, &tmp);

						found = 1;
						break;
					}
					cp += si->isi_len, len -= si->isi_len;
				} while (len >= sizeof(struct ieee80211req_sta_info));
			}
		}
		free(buf);
		if (found)
			return;
    }


	// failed to get via Atheros IOCTL. Try continue in traditional way
	// Get TX Rate
	if ( ioctl ( skfd, SIOCGIWRATE, &iwr ) >= 0 ) {
		oid[12] = 2;
		update(odb, oid, BER_Gauge32, &iwr.u.bitrate.value);
	}
		
	// get signal strength
	iwr.u.data.pointer = ( caddr_t ) &iwstats;
	iwr.u.data.length = 0;
	iwr.u.data.flags = 1;   // Clear updated flag
	if ( ioctl ( skfd, SIOCGIWSTATS, &iwr ) >= 0 ) {
		int strength = iwstats.qual.level - 0x100;
		oid[12] = 4;
		update(odb, oid, BER_INTEGER, &strength);
	}
}





static void
mtik_node_update(struct odb **odb, int skfd, char *ifname, 
		int ifidx, struct ieee80211req_sta_info *si)
{
	struct iwreq iwr;
	struct ieee80211req_sta_stats stats;
	int tmp;

	uint32_t oid[20] = { 19, 43, 6, 1 ,4, 1, 14988, 1, 1, 1, 2, 1, 1, 0, 0, 0, 0, 0, 0, 0 };

	oid[13] = si->isi_macaddr[0];
	oid[14] = si->isi_macaddr[1];
	oid[15] = si->isi_macaddr[2];
	oid[16] = si->isi_macaddr[3];
	oid[17] = si->isi_macaddr[4];
	oid[18] = si->isi_macaddr[5];

	oid[19] = ifidx;

	oid[12] = 3;
	tmp = si->isi_noisefloor + si->isi_rssi;
	update(odb, oid, BER_INTEGER, &tmp);


	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, ifname, sizeof(iwr.ifr_name));
	iwr.u.data.pointer = (void *) &stats;
	iwr.u.data.length = sizeof(stats);
	memcpy(stats.is_u.macaddr, si->isi_macaddr, sizeof(stats.is_u.macaddr));
	if (ioctl(skfd, IEEE80211_IOCTL_STA_STATS, &iwr) < 0) {
		log_printf(LOG_WARNING,
				"unable to get node information on %s\n", ifname);
		return;
	}

	oid[12] = 4;
	tmp = stats.is_stats.ns_tx_bytes;
	update(odb, oid, BER_Counter32, &tmp);
	oid[12] = 5;
	tmp = stats.is_stats.ns_rx_bytes;
	update(odb, oid, BER_Counter32, &tmp);
	oid[12] = 6;
	tmp = stats.is_stats.ns_tx_data;
	update(odb, oid, BER_Counter32, &tmp);
	oid[12] = 7;
	tmp = stats.is_stats.ns_rx_data;
	update(odb, oid, BER_Counter32, &tmp);

	oid[12] = 8;
	tmp = (si->isi_rates[si->isi_txrate] & 0x7F) / 2 * 1000000;
	update(odb, oid, BER_Gauge32, &tmp);
	oid[12] = 9;
	tmp = (si->isi_rates[si->isi_rxrate] & 0x7F) / 2 * 1000000;
	update(odb, oid, BER_Gauge32, &tmp);
}



static int
mtik_registration_table_update(struct odb **odb, int skfd, char *ifname, int ifidx)
{
	int len;
	struct iwreq iwr;
	struct ieee80211req_sta_info *si;
	u_int8_t *cp;
	u_int8_t *buf = malloc(STAINFO_BUF_SIZE);

	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, ifname, sizeof(iwr.ifr_name));
	iwr.u.data.pointer = (void*)buf;
	iwr.u.data.length = STAINFO_BUF_SIZE;

	if (ioctl(skfd, IEEE80211_IOCTL_STA_INFO, &iwr) < 0) {
		log_printf(LOG_WARNING, "unable to get station information on %s\n", ifname);
		free(buf);
		return -1;
	}

	len = iwr.u.data.length;
	if (len >= sizeof(struct ieee80211req_sta_info)) {
		cp = buf;
		do {
			si = (struct ieee80211req_sta_info *) cp;
			mtik_node_update(odb, skfd, ifname, ifidx, si);
			cp += si->isi_len, len -= si->isi_len;
		} while (len >= sizeof(struct ieee80211req_sta_info));
	}

	free(buf);
	return 0;
}


static int
mtik_wireless_update (struct odb **odb)
{
    struct iwreq iwr;                       // ioctl request structure
    FILE *fp;
    char  bfr[1024], ifName[IFNAMSIZ+1];
    char *s, *t;
	int ifIndex = 0;
	int skfd = open_socket();

    if (skfd < 0) {
		log_printf (LOG_ERROR, "SNMP mikrotik.%s(): socket open failure\n", __func__ );
		return -1;
    }

	odb_destroy (odb);

    // find interfaces in /proc/net/dev and find the wireless interfaces
    fp = fopen("/proc/net/dev", "r");
    if (fp) {
		while (fgets (bfr, sizeof (bfr), fp)) {
			if (strstr(bfr, ":")) {
				s = bfr; t = ifName;
				while (isspace (*s)) // discard white space
					*s++;
				while (*s != ':')     // get interface name
					*t++ = *s++;
				*t = '\0';

				ifIndex = ifcache_get_ifindex(ifName);
				if (!ifIndex) {
				    log_printf (LOG_WARNING, "SNMP mikrotik.%s()  `%s` has no ifIndex\n",
							__func__, ifName);
				    continue;
				}

				// verify as a wireless device
				strncpy(iwr.ifr_name, ifName, IFNAMSIZ);
				if ( ioctl ( skfd, SIOCGIWNAME, &iwr ) >= 0 ) {
					mtik_wireless_station_update(odb, skfd, ifName, ifIndex);
					mtik_registration_table_update(odb, skfd, ifName, ifIndex);
				}
			}
		}
		fclose(fp);
    }

    close(skfd);
	return 0;
}


static const uint32_t mtik_oid[] = { 6, 43, 6, 1, 4, 1, 14988 };

struct module module =
{
   .name	= "mikrotik",
   .descr	= "The Mikrotik experimental wireless MIB module ",
   .mod_oid	= mtik_oid,
   .con_oid	= mtik_oid,
   .parse	= NULL,
   .open	= NULL,
   .update	= mtik_wireless_update,
   .close	= NULL
};

