/* dhcpd.c
 *
 * udhcp Server
 * Copyright (C) 1999 Matthew Ramsay <matthewr@moreton.com.au>
 *			Chris Trew <ctrew@moreton.com.au>
 *
 * Rewrite by Russ Dill <Russ.Dill@asu.edu> July 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <time.h>
#include <sys/time.h>

#include "debug.h"
#include "dhcpd.h"
#include "arpping.h"
#include "socket.h"
#include "options.h"
#include "files.h"
#include "leases.h"
#include "packet.h"
#include "serverpacket.h"
#include "pidfile.h"


#define ARP_TIMEOUT_SEC 2
#define MAX_ARP_PER_ROUND 20
/* globals */
struct dhcpOfferedAddr *leases;
struct server_config_t server_config;
static int signal_pipe[2];

/* Exit and cleanup */
static void exit_server(int retval)
{
        arp_shutdown();
	pidfile_delete(server_config.pidfile);
        CLOSE_LOG();
	exit(retval);
}


/* Signal handler */
static void signal_handler(int sig)
{
	if (send(signal_pipe[1], &sig, sizeof(sig), MSG_DONTWAIT) < 0) {
		LOG(LOG_ERR, "Could not send signal: %s",
			strerror(errno));
	}
}

/**
 * normalizes timeval values after addition or subtraction operations
 * normalizing result of more than one sequent addition or subtraction operation
 * can lead to undefined results because of tv_usec field overflow.
 */
static struct timeval* time_normalize(struct timeval* a)
{
    if ((a->tv_sec > 0 && a->tv_usec < 0) || (a->tv_usec <= -1000000 ))
    {
        /* detected negative usec value, while tv_sec is positive or
           tv_usec contains overflowed negative value */
        --a->tv_sec;
        a->tv_usec += 1000000;
    }
    else if ((a->tv_sec < 0 && a->tv_usec > 0) || (a->tv_usec >= 1000000))
    {
        /* detected positive usec value, while tv_sec is negative or
           tv_usec contains overflowed positive value */
	++a->tv_sec;
	a->tv_usec -= 1000000;
    }
    return a;
}

/**
 * returns normalized value of a - b result.
 * Method can handle positive and negative values
 * Precondition: a and b should be normalized timval values.
 */
static struct timeval time_sub(const struct timeval* a, const struct timeval* b)
{
    struct timeval result;
    result.tv_sec = a->tv_sec - b->tv_sec;
    result.tv_usec = a->tv_usec - b->tv_usec;
    time_normalize(&result);
    return result;
}

/**
 * returns normalized value of a + b result.
 * Method can handle positive and negative values
 * Precondition: a and b should be normalized timval values.
 */
static struct timeval time_add(const struct timeval* a, const struct timeval* b)
{
    struct timeval result;
    result.tv_sec = a->tv_sec + b->tv_sec;
    result.tv_usec = a->tv_usec + b->tv_usec;
    time_normalize(&result);
    return result;
}

/**
 * compares 2 timeval values and returns result of a < b
 */
static int time_lt(const struct timeval* a, const struct timeval* b)
{
    return a->tv_sec == b->tv_sec ? a->tv_usec < b->tv_usec : a->tv_sec < b->tv_sec;
}

/* @return 0 when timout is available or nonzero value otherwise */
static int process_timeouts(struct timeval *server_timeout, struct timeval *tv)
{
    static const struct timeval zero = {0,0};
    static const struct timeval arp_timeout = {ARP_TIMEOUT_SEC, 0};
    struct timeval now;
    struct timeval timeout_left;
    struct timeval arp_compare_timestamp;
    unsigned int i;
    int rv = -1;

    tv->tv_sec = INT_MAX;
    tv->tv_usec = 0;
    gettimeofday(&now, 0);
    arp_compare_timestamp = time_sub(&now, &arp_timeout);
    /* TODO: procees pending arp timeouts */
    for (i = 0; i < server_config.max_leases; i++)
    {
        if(leases[i].discovery)
        {
            if (time_lt(&now, &leases[i].discovery->timestamp))
            {
                DEBUG(LOG_INFO,"Time skew detected, re-timestamping.");
                leases[i].discovery->timestamp = now;
            }
            else if (time_lt(&leases[i].discovery->timestamp, &arp_compare_timestamp))
            {
                DEBUG(LOG_INFO,"ARP timeout: address is not used.");
                sendOffer( &leases[i] );
            }
            else
            {
                timeout_left = time_sub(&leases[i].discovery->timestamp, &arp_compare_timestamp);
                if(time_lt(&timeout_left, tv))
                {
                    *tv = timeout_left;
                    rv = 0;
                }
            }
        }
    }

    if (server_config.auto_time)
    {
        timeout_left = time_sub(server_timeout, &now);
        if (time_lt(&timeout_left, &zero ))
        {
            write_leases(); /* it may take some time ... */
            timeout_left.tv_sec = server_config.auto_time;
            timeout_left.tv_usec = 0;
            *server_timeout = time_add(&now, &timeout_left);
        }
        if (time_lt(&timeout_left, tv))
        {
            *tv = timeout_left;
            rv = 0;
        }
    }
    return rv;
}

static void dhcpdiscover( const struct dhcpMessage *packet )
{
    DEBUG(LOG_INFO,"received DISCOVER");
    if (handleDiscover(packet) < 0)
    {
        LOG(LOG_ERR, "handle DISCOVER failed");
    }
}

static void dhcprequest( const struct dhcpMessage *packet )
{
    const unsigned char *server_id, *requested;
    u_int32_t server_id_align, requested_align;
    struct dhcpOfferedAddr *lease;

    DEBUG(LOG_INFO, "received REQUEST");

    /* ADDME: look for a static lease */
    lease = find_lease_by_chaddr(packet->chaddr);

    requested = get_option(packet, DHCP_REQUESTED_IP);
    server_id = get_option(packet, DHCP_SERVER_ID);

    if (requested) memcpy(&requested_align, requested, 4);
    if (server_id) memcpy(&server_id_align, server_id, 4);

    if (lease) /*ADDME: or static lease */
    {
        if (server_id)
        {
            /* SELECTING State */
            DEBUG(LOG_INFO, "server_id = %08x", ntohl(server_id_align));
            if (server_id_align == server_config.server && requested &&
                requested_align == lease->yiaddr)
            {
                sendACK(packet, lease->yiaddr);
            }
        }
        else if (requested)
        {
            /* INIT-REBOOT State */
            if (lease->yiaddr == requested_align)
            {
                sendACK(packet, lease->yiaddr);
            }
            else
            {
                sendNAK(packet, requested_align);
            }
        }
        else
        {
            /* RENEWING or REBINDING State */
            if (lease->yiaddr == packet->ciaddr)
            {
                sendACK(packet, lease->yiaddr);
            }
            else
            {
                /* don't know what to do!!!! */
                sendNAK(packet, packet->ciaddr);
            }
        }

        /* what to do if we have no record of the client */
    }
    else if (server_id)
    {
        /* SELECTING State */
    }
    else if (requested)
    {
        /* INIT-REBOOT State */
        lease = find_lease_by_yiaddr(requested_align);
        if (lease && lease_expired(lease))
        {
            /* probably best if we drop this lease */
            memset(lease->chaddr, 0, 16);
            /* make some contention for this address */
        }
        else
        {
            /* always send NAK, if lease not found or it is not expired */
            sendNAK(packet, requested_align);
        }

    }
    else
    {
        /* RENEWING or REBINDING State */
    }
}

static void dhcpdecline( const struct dhcpMessage *packet )
{
    struct dhcpOfferedAddr *lease;

    DEBUG(LOG_INFO,"received DECLINE");

    /* ADDME: look for a static lease */
    lease = find_lease_by_chaddr(packet->chaddr);
    if (lease)
    {
        memset(lease->chaddr, 0, 16);
        lease->expires = time(0) + server_config.decline_time;
    }
}

static void dhcprelease( const struct dhcpMessage *packet )
{
    struct dhcpOfferedAddr *lease;

    /* ADDME: look for a static lease */
    lease = find_lease_by_chaddr(packet->chaddr);
    DEBUG(LOG_INFO,"received RELEASE");
    if (lease)
    {
        lease->expires = time(0);
    }
}

static void dhcpinform( const struct dhcpMessage *packet )
{
    DEBUG(LOG_INFO,"received INFORM");
    send_inform( packet );
}

static int handle_server_socket(int server_socket)
{
    int bytes;
    struct dhcpMessage packet;
    const unsigned char *state;

    do
    {
        bytes = get_packet(&packet, server_socket);
    }
    while (bytes < 0 && errno == EINTR);

    if (bytes < 0)
    {
        DEBUG(LOG_INFO, "error on read, %s, closing socket", strerror(errno));
        return -1;
    }

    if ((state = get_option(&packet, DHCP_MESSAGE_TYPE)) == NULL)
    {
        DEBUG(LOG_ERR, "couldn't get option from packet, ignoring");
        return 0;
    }

    switch (state[0])
    {
    case DHCPDISCOVER:
        dhcpdiscover( &packet );
        break;
    case DHCPREQUEST:
        dhcprequest( &packet );
        break;
    case DHCPDECLINE:
        dhcpdecline( &packet );
        break;
    case DHCPRELEASE:
        dhcprelease( &packet );
        break;
    case DHCPINFORM:
        dhcpinform( &packet );
        break;
    default:
        LOG(LOG_WARNING, "unsupported DHCP message (%02x) -- ignoring", state[0]);
    }
    return 0;
}

void handle_arp_response(void)
{
    int result;
    int i;
    in_addr_t ip;
    unsigned char mac[ETH_ALEN];
    struct dhcpOfferedAddr* lease;
    struct dhcpDiscoverData* d_data;

    for (i = 0; i < MAX_ARP_PER_ROUND; ++i)
    {
        result = arp_handle(&ip, mac);
        if (result < 0)
        {
            break;
        }
        if (result == 0)
        {
            continue;
        }
        lease = find_lease_by_yiaddr(ip);
        if (lease && lease->discovery)
        {
            if (memcmp(mac, lease->chaddr, sizeof(mac)))
            {
                /* response from another mac - mark lesa as used and continue searching for leases. */
                d_data = lease->discovery;
                lease->discovery = 0;
                add_lease(blank_chaddr, ip, server_config.conflict_time);
                handleDiscover(&d_data->packet);
                free(d_data);
            }
            else
            {
                sendOffer(lease);
            }
        }
    }
}

#ifdef COMBINED_BINARY
int udhcpd_main(int argc, char *argv[])
#else
int main(int argc, char *argv[])
#endif
{
    fd_set rfds;
    struct timeval tv;
    struct timeval timeout_end;
    struct timeval* p_timeout;
    int server_socket = -1;
    int retval;
    int pid_fd;
    int max_sock;
    int sig;
    int arp_socket;
    struct option_set *option;
    int foreground = 0;

    OPEN_LOG("udhcpd");
    LOG(LOG_INFO, "udhcp server (v%s) started", VERSION);

    memset(&server_config, 0, sizeof(struct server_config_t));

    if (argc < 2)
        read_config(DHCPD_CONF_FILE);
    else
    {
	if (strcasecmp(argv[1], "-f"))
	{
	    read_config(argv[1]);
	}
	else
	{
	    foreground = 1;
            if (argc < 3)
		read_config(DHCPD_CONF_FILE);
            else
		read_config(argv[2]);
	}
    }
    pid_fd = pidfile_acquire(server_config.pidfile);
    pidfile_write_release(pid_fd);

    if ((option = find_option(server_config.options, DHCP_LEASE_TIME))) {
        memcpy(&server_config.lease, option->data + 2, 4);
        server_config.lease = ntohl(server_config.lease);
    }
    else server_config.lease = LEASE_TIME;
    if ((option = find_option(server_config.options, DHCP_SUBNET))) {
        memcpy(&server_config.subnet, option->data + 2, 4);
    }
    else
    {
        LOG(LOG_WARNING, "No subnet option configured.");
        server_config.subnet = htonl(SUBNET_MASK);
    }
    
    if ( (server_config.start & server_config.subnet) != (server_config.end & server_config.subnet))
    {
        LOG(LOG_WARNING, "Start and end is in the different subnets. Adjusting end value.");
        server_config.end = htonl(ntohl(server_config.start | ~server_config.subnet) - 1);
    }
    /* do not alow using first network address */
    if ( ntohl(server_config.start) == ntohl(server_config.start & server_config.subnet) )
    {
        server_config.start = htonl(ntohl(server_config.start) + 1);
    }
    /* do not alow using last network address */
    if ( ntohl(server_config.end) == ntohl(server_config.start | ~server_config.subnet) )
    {
        server_config.end = htonl(ntohl(server_config.end) - 1);
    }

    if (read_interface(server_config.interface, &server_config.ifindex,
                       &server_config.server, server_config.arp) < 0)
        exit_server(1);

    leases = malloc(sizeof(struct dhcpOfferedAddr) * server_config.max_leases);
    memset(leases, 0, sizeof(struct dhcpOfferedAddr) * server_config.max_leases);
    read_leases(server_config.lease_file);

#ifndef DEBUGGING
    pid_fd = pidfile_acquire(server_config.pidfile); /* hold lock during fork. */
#ifndef NODAEMON
    if (!foreground && daemon(0, 0) == -1) {
        perror("fork");
        exit_server(1);
    }
#endif
    pidfile_write_release(pid_fd);
#endif

    arp_socket = arp_init(server_config.server, server_config.arp, server_config.ifindex);
    if(arp_socket < 0)
    {
        LOG(LOG_ERR, "FATAL: couldn't create arp socket, %s", strerror(errno));
        exit_server(-1);
    }

    socketpair(AF_UNIX, SOCK_STREAM, 0, signal_pipe);
    signal(SIGUSR1, signal_handler);
    signal(SIGTERM, signal_handler);

    gettimeofday(&timeout_end, NULL);
    timeout_end.tv_sec += server_config.auto_time;
    while(1) /* loop until universe collapses */
    {

        if (server_socket < 0)
        {
            if ((server_socket = listen_socket(INADDR_ANY, SERVER_PORT, server_config.interface)) < 0)
            {
                LOG(LOG_ERR, "FATAL: couldn't create server socket, %s", strerror(errno));
                exit_server(-1);
            }
        }

        FD_ZERO(&rfds);
        FD_SET(server_socket, &rfds);
        FD_SET(arp_socket, &rfds);
        FD_SET(signal_pipe[0], &rfds);

        max_sock = server_socket;
        if( max_sock < arp_socket )
        {
            max_sock = arp_socket;
        }
        if( max_sock < signal_pipe[0] )
        {
            max_sock = signal_pipe[0];
        }

        if (process_timeouts(&timeout_end, &tv))
        {
            p_timeout = 0;
        }
        else
        {
            p_timeout = &tv;
        }

        retval = select(max_sock + 1, &rfds, NULL, NULL, p_timeout);
        if (retval < 0)
        {
            DEBUG(LOG_INFO, "error on select");
            continue;
        }

        if (FD_ISSET(signal_pipe[0], &rfds))
        {
            if (read(signal_pipe[0], &sig, sizeof(sig)) > 0)
            {
                switch (sig)
                {
                case SIGUSR1:
                    LOG(LOG_INFO, "Received a SIGUSR1");
                    write_leases();
                    /* why not just reset the timeout, eh */
                    gettimeofday(&timeout_end, 0);
                    timeout_end.tv_sec += server_config.auto_time;
                    break;
                case SIGTERM:
                    LOG(LOG_INFO, "Received a SIGTERM");
                    write_leases();
                    exit_server(0);
                }
            }
            /* else probably just EINTR */
        }

        if (FD_ISSET(arp_socket, &rfds))
        {
            handle_arp_response();
        }

        if (FD_ISSET(server_socket, &rfds))
        {
            if(handle_server_socket(server_socket))
            {
                close(server_socket);
                server_socket = -1;
            }
        }
    }
    return 0;
}
