/*
 * leases.c -- tools to manage DHCP leases
 * Russ Dill <Russ.Dill@asu.edu> July 2001
 */

#include <time.h>
#include <string.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "debug.h"
#include "dhcpd.h"
#include "files.h"
#include "options.h"
#include "leases.h"
#include "arpping.h"

const unsigned char blank_chaddr[] = {[0 ... 15] = 0};

/* clear every lease out that chaddr OR yiaddr matches and is nonzero */
void clear_lease(const u_int8_t *chaddr, u_int32_t yiaddr)
{
    unsigned int i;
    int blank;
#ifdef DEBUGGING
    struct in_addr addr;
#endif

    blank = memcmp(blank_chaddr, chaddr, 16);

    for (i = 0; i < server_config.max_leases; ++i)
    {
        if (
            (blank && !memcmp(leases[i].chaddr, chaddr, 16)) ||
            (yiaddr && leases[i].yiaddr == yiaddr)
           )
        {
            if (leases[i].discovery)
            {
#ifdef DEBUGGING
                addr.s_addr = leases[i].yiaddr;
                DEBUG(
                    LOG_DEBUG,"Destroing discovery data: %s of %02X:%02X:%02X:%02X:%02X:%02X",
                    inet_ntoa(addr),
                    leases[i].chaddr[0], leases[i].chaddr[1], leases[i].chaddr[2],
                    leases[i].chaddr[3], leases[i].chaddr[4], leases[i].chaddr[5]
                   );
#endif
                free (leases[i].discovery);
                leases[i].discovery = 0;
            }
            memset(&(leases[i]), 0, sizeof(struct dhcpOfferedAddr));
        }
    }
}

/* add a lease into the table, clearing out any old ones */
struct dhcpOfferedAddr *add_lease(const u_int8_t *chaddr, u_int32_t yiaddr, unsigned long lease)
{
	struct dhcpOfferedAddr *oldest;

	/* clean out any old ones */
	clear_lease(chaddr, yiaddr);

	oldest = oldest_expired_lease();

	if (oldest) {
		memcpy(oldest->chaddr, chaddr, 16);
		oldest->yiaddr = yiaddr;
		oldest->expires = time(0) + lease;
	}

	return oldest;
}


/* true if a lease has expired */
int lease_expired(const struct dhcpOfferedAddr *lease)
{
	return (lease->expires < (unsigned long) time(0));
}


/* Find the oldest expired lease, NULL if there are no expired leases */
struct dhcpOfferedAddr *oldest_expired_lease(void)
{
	struct dhcpOfferedAddr *oldest = NULL;
	unsigned long oldest_lease = time(0);
	unsigned int i;


	for (i = 0; i < server_config.max_leases; i++)
		if (oldest_lease > leases[i].expires) {
			oldest_lease = leases[i].expires;
			oldest = &(leases[i]);
		}
	return oldest;

}


/* Find the first lease that matches chaddr, NULL if no match */
struct dhcpOfferedAddr *find_lease_by_chaddr(const u_int8_t *chaddr)
{
	unsigned int i;

        for (i = 0; i < server_config.max_leases; i++)
        {
            if (!memcmp(leases[i].chaddr, chaddr, 16))
            {
                return &(leases[i]);
            }
        }
	return NULL;
}


/* Find the first lease that matches yiaddr, NULL is no match */
struct dhcpOfferedAddr *find_lease_by_yiaddr(u_int32_t yiaddr)
{
    unsigned int i;

    for (i = 0; i < server_config.max_leases; i++)
    {
        if (leases[i].yiaddr == yiaddr)
        {
            return &(leases[i]);
        }
    }
    return NULL;
}


/* find an assignable address, it check_expired is true, we check all the expired leases as well.
 * Maybe this should try expired leases by age... */
u_int32_t find_address(int check_expired)
{
    u_int32_t addr, begin, end, ret;
    struct dhcpOfferedAddr *lease = NULL;

    /* server_config in network order */
    begin = ntohl(server_config.start);
    end = ntohl(server_config.end);

    for (addr = begin; addr <= end; ++addr)
    {
        ret = htonl(addr);
        /* check if it not Our IP */
        if (ret == server_config.server)
        {
            continue;
        }

        if (
            /* lease is not taken */
            !(lease = find_lease_by_yiaddr(ret)) ||

            /* or it expired and we are checking for expired leases */
            (check_expired  && lease_expired(lease))
           )
        {
            return ret;
        }
    }
    return 0;
}
