/* serverpacket.c
 *
 * Constuct and send DHCP server packets
 *
 * Russ Dill <Russ.Dill@asu.edu> July 2001
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <errno.h>

#include "packet.h"
#include "debug.h"
#include "dhcpd.h"
#include "options.h"
#include "leases.h"
#include "arpping.h"

/* send a packet to giaddr using the kernel ip stack */
static int send_packet_to_relay(const struct dhcpMessage *payload)
{
	DEBUG(LOG_INFO, "Forwarding packet to relay");

	return kernel_packet(payload, server_config.server, SERVER_PORT,
			payload->giaddr, SERVER_PORT);
}


/* send a packet to a specific arp address and ip address by creating our own ip packet */
static int send_packet_to_client(const struct dhcpMessage *payload, int force_broadcast)
{
	const unsigned char *chaddr;
	u_int32_t ciaddr;

	if (force_broadcast) {
		DEBUG(LOG_INFO, "broadcasting packet to client (NAK)");
		ciaddr = INADDR_BROADCAST;
		chaddr = MAC_BCAST_ADDR;
	} else if (payload->ciaddr) {
		DEBUG(LOG_INFO, "unicasting packet to client ciaddr");
		ciaddr = payload->ciaddr;
		chaddr = payload->chaddr;
	} else if (ntohs(payload->flags) & BROADCAST_FLAG) {
		DEBUG(LOG_INFO, "broadcasting packet to client (requested)");
		ciaddr = INADDR_BROADCAST;
		chaddr = MAC_BCAST_ADDR;
	} else {
		DEBUG(LOG_INFO, "unicasting packet to client yiaddr");
		ciaddr = payload->yiaddr;
		chaddr = payload->chaddr;
	}
	return raw_packet(payload, server_config.server, SERVER_PORT,
			ciaddr, CLIENT_PORT, chaddr, server_config.ifindex);
}


/* send a dhcp packet, if force broadcast is set, the packet will be broadcast to the client */
static int send_packet(const struct dhcpMessage *payload, int force_broadcast)
{
	int ret;

	if (payload->giaddr)
		ret = send_packet_to_relay(payload);
	else ret = send_packet_to_client(payload, force_broadcast);
	return ret;
}


static void init_packet(struct dhcpMessage *packet, const struct dhcpMessage *oldpacket, char type)
{
	init_header(packet, type);
	packet->xid = oldpacket->xid;
	memcpy(packet->chaddr, oldpacket->chaddr, 16);
	packet->flags = oldpacket->flags;
        packet->giaddr = oldpacket->giaddr;
	packet->ciaddr = oldpacket->ciaddr;
	add_simple_option(packet->options, DHCP_SERVER_ID, server_config.server);
}


/* add in the bootp options */
static void add_bootp_options(struct dhcpMessage *packet)
{
	packet->siaddr = server_config.siaddr;
	if (server_config.sname)
		strncpy(packet->sname, server_config.sname, sizeof(packet->sname) - 1);
	if (server_config.boot_file)
		strncpy(packet->file, server_config.boot_file, sizeof(packet->file) - 1);
}

int handleDiscover( const struct dhcpMessage *oldpacket )
{
    /**
     * try:
     * 1. find lease by chaddr (client MAC);
     * 2. find lease by yaddr (REQUESTED IP);
     * 3. find free lease;
     * 4. find expired lease;
     */
    struct dhcpOfferedAddr *lease = 0;
    u_int32_t ip = 0;
    const unsigned char *req;
    struct dhcpDiscoverData* d_data;

    DEBUG(
        LOG_INFO, "handling DISCOVER from  %02X:%02X:%02X:%02X:%02X:%02X",
        oldpacket->chaddr[0], oldpacket->chaddr[1], oldpacket->chaddr[2],
        oldpacket->chaddr[3], oldpacket->chaddr[4], oldpacket->chaddr[5]
       );

    /* 1. */
    lease = find_lease_by_chaddr(oldpacket->chaddr);
    if (!lease)
    {
        req = get_option(oldpacket, DHCP_REQUESTED_IP);
        if (req)
        {
            /* 2. */
            memcpy(&ip, req, sizeof(ip));
            if (
                ntohl(ip) < ntohl(server_config.start) ||
                ntohl(ip) > ntohl(server_config.end) ||
                ip == server_config.server
               )
            {
                /* ip is invalid - reset it */
                ip = 0;
            }
            else
            {
                lease = find_lease_by_yiaddr(ip);
                if(lease && !lease_expired(lease))
                {
                    /* can't give, lease not expired yet - reset it */
                    ip = 0;
                }
            }
        }
        if (!ip)
        {
            /* 3. */
            ip = find_address(0);
        }
        if (!ip)
        {
            /* 4. */
            /* try for an expired lease */
            ip = find_address(1);
        }
        if (!ip)
        {
            LOG(LOG_WARNING, "no IP addresses to give -- OFFER abandoned");
            return -1;
        }
    }
    else
    {
        ip = lease->yiaddr;
    }

    lease = add_lease(oldpacket->chaddr, ip, server_config.offer_time);
    if (!lease)
    {
        LOG(LOG_WARNING, "lease pool is full -- OFFER abandoned");
        return -1;
    }

    if (arp_ping(ip) <= 0)
    {
        /* Forget this lease */
        memset(lease, 0, sizeof(*lease));
        return -1;
    }

    d_data = (struct dhcpDiscoverData*)malloc(sizeof(struct dhcpDiscoverData));
    if (!d_data)
    {
        LOG(LOG_ERR, "Error allocating memory for DHCP Discover data.");
        return -1;
    }
    memcpy(&d_data->packet, oldpacket, sizeof(*oldpacket));
    gettimeofday(&d_data->timestamp, 0);
    lease->discovery = d_data;

    return 0;
}

/* send a DHCP OFFER to a DHCP DISCOVER */
int sendOffer( struct dhcpOfferedAddr *lease )
{
    struct dhcpMessage *oldpacket;
    struct dhcpMessage packet;
    u_int32_t lease_time_align = server_config.lease;
    const unsigned char *lease_time;
    struct option_set *curr;
    struct in_addr addr;

    if (!lease)
    {
        LOG(LOG_ERR, "BUG: Lease is NULL.");
        return -1;
    }
    if (!lease->discovery)
    {
        LOG(LOG_ERR, "BUG: Lease has no DHCP DISCOVER data.");
        return -1;
    }
    oldpacket = &(lease->discovery->packet);

    init_packet(&packet, oldpacket, DHCPOFFER);
    packet.yiaddr = lease->yiaddr;
    if ((lease_time = get_option(oldpacket, DHCP_LEASE_TIME)))
    {
        memcpy(&lease_time_align, lease_time, 4);
        lease_time_align = ntohl(lease_time_align);
        if (lease_time_align > server_config.lease)
        {
            lease_time_align = server_config.lease;
        }
    }

    /* Make sure we aren't just using the lease time from the previous offer */
    if (lease_time_align < server_config.min_lease)
    {
        lease_time_align = server_config.lease;
    }
    add_simple_option(packet.options, DHCP_LEASE_TIME, htonl(lease_time_align));

    curr = server_config.options;
    while (curr)
    {
        if (curr->data[OPT_CODE] != DHCP_LEASE_TIME)
        {
            add_option_string(packet.options, curr->data);
        }
        curr = curr->next;
    }
    add_bootp_options(&packet);
    addr.s_addr = packet.yiaddr;
    LOG(LOG_INFO, "sending OFFER of %s to %02X:%02X:%02X:%02X:%02X:%02X",
        inet_ntoa(addr),
        packet.chaddr[0], packet.chaddr[1], packet.chaddr[2],
        packet.chaddr[3], packet.chaddr[4], packet.chaddr[5]
       );

    free(lease->discovery);
    lease->discovery = 0;

    return send_packet(&packet, 0);
}

int sendNAK(const struct dhcpMessage *oldpacket, u_int32_t yiaddr)
{
    struct dhcpMessage packet;
    struct in_addr addr;

    init_packet(&packet, oldpacket, DHCPNAK);

    addr.s_addr = yiaddr;
    LOG(
        LOG_INFO, "sending NAK of %s to %02X:%02X:%02X:%02X:%02X:%02X",
        inet_ntoa(addr),
        packet.chaddr[0], packet.chaddr[1], packet.chaddr[2],
        packet.chaddr[3], packet.chaddr[4], packet.chaddr[5]
       );
    return send_packet(&packet, 1);
}

void notify_addr(char *cmd, char *iface, char *ip_addr, char *mac_addr)
{
	int pid, status;

	if (cmd == 0)
		return;

	pid = fork();

	if (pid == -1)
	{
		LOG(LOG_INFO, "Unable to fork \"%s\"", cmd);
		return;
	}

	if (pid == 0) {
	   	char *argv[5];

	        argv[0] = cmd;
	        argv[1] = iface;
		argv[2] = ip_addr;
                argv[3] = mac_addr;
		argv[4] = 0;

		if (execv(cmd,argv))
			LOG(LOG_INFO, "Error executing command \"%s\"", cmd);

		exit(127);
	}
	do {
	        if (waitpid(pid, &status, 0) == -1) {
			if (errno != EINTR)
			{
				// error
				LOG(LOG_INFO, "Error executing command \"%s\"", cmd);
				return;
			}
		} else
		{
			// success
			LOG(LOG_INFO, "Command \"%s\" successful.", cmd);
			return;
		}
	} while(1);
}

int sendACK(const struct dhcpMessage *oldpacket, u_int32_t yiaddr)
{
	struct dhcpMessage packet;
	struct option_set *curr;
	const unsigned char *lease_time;
	u_int32_t lease_time_align = server_config.lease;
	struct in_addr addr;
	char ip_buf[255], mac_buf[255];
        int i = 0;

	init_packet(&packet, oldpacket, DHCPACK);
	packet.yiaddr = yiaddr;

	if ((lease_time = get_option(oldpacket, DHCP_LEASE_TIME))) {
		memcpy(&lease_time_align, lease_time, 4);
		lease_time_align = ntohl(lease_time_align);
		if (lease_time_align > server_config.lease)
			lease_time_align = server_config.lease;
		else if (lease_time_align < server_config.min_lease)
			lease_time_align = server_config.lease;
	}

	add_simple_option(packet.options, DHCP_LEASE_TIME, htonl(lease_time_align));

	curr = server_config.options;
	while (curr) {
		if (curr->data[OPT_CODE] != DHCP_LEASE_TIME)
			add_option_string(packet.options, curr->data);
		curr = curr->next;
	}

	add_bootp_options(&packet);

	addr.s_addr = packet.yiaddr;
        LOG(LOG_INFO, "sending ACK of %s to %02X:%02X:%02X:%02X:%02X:%02X",
            inet_ntoa(addr),
            packet.chaddr[0], packet.chaddr[1], packet.chaddr[2],
            packet.chaddr[3], packet.chaddr[4], packet.chaddr[5]
           );

	if (send_packet(&packet, 0) < 0)
		return -1;

	add_lease(packet.chaddr, packet.yiaddr, lease_time_align);

	if (server_config.notify_addr) {
	    sprintf(ip_buf, "%s", inet_ntoa(addr));
	    memset(mac_buf, 0, 1);
            while (i < 5)
		sprintf(mac_buf, "%s%.2X:", mac_buf, packet.chaddr[i++]);
	    sprintf(mac_buf, "%s%.2X", mac_buf, packet.chaddr[i]);
	    notify_addr(server_config.notify_addr, server_config.interface, ip_buf, mac_buf);
	}

	return 0;
}


int send_inform(const struct dhcpMessage *oldpacket)
{
	struct dhcpMessage packet;
	struct option_set *curr;

	init_packet(&packet, oldpacket, DHCPACK);

	curr = server_config.options;
	while (curr) {
		if (curr->data[OPT_CODE] != DHCP_LEASE_TIME)
			add_option_string(packet.options, curr->data);
		curr = curr->next;
	}

	add_bootp_options(&packet);

	return send_packet(&packet, 0);
}
