#include <stdio.h>
#include <unistd.h>
#include <stdlib.h> 
#include <string.h>
#include <netinet/if_ether.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <linux/if.h>
#include <errno.h>

#define FILE_PROC_NET_ARP "/proc/net/arp"
#define DEFAULT_IPP_SIZE 64 

static int checksum(u_int16_t *buf, size_t size)
{
	int nleft = size;
	int sum = 0;
	u_int16_t *p = buf;
	u_int16_t result = 0;

	while (nleft > 1)
	{
		sum += *p++;
		nleft -= 2;
	}

	if (nleft == 1)
	{
		*(u_int8_t *)(&result) = *(u_int8_t *)p;
		sum += result;
	}

	sum = (sum >> 16) + (sum & 0xFFFF);
	sum += (sum >> 16);
	result = ~sum;
	return result;
}

typedef struct
{
	u_int32_t ip;
	u_int32_t ignored;
}
ip_info_t;

typedef struct
{
	ip_info_t* ips;
	size_t count;
	size_t size; 
} 
ip_pool_t;

int ip_pool_grow(ip_pool_t* ipp)
{
	size_t new_size = ipp->size * 2;
	ip_info_t *new_pool;
	
	if (!ipp->ips)
	{
		new_size = DEFAULT_IPP_SIZE;
	}
	else
	{
		new_size = ipp->size * 2;
	}
	
	new_pool = (ip_info_t *)realloc(ipp->ips, new_size * sizeof(*ipp->ips));
	if (new_pool)
	{
		ipp->size = new_size;
		ipp->ips = new_pool;
		return 0;
	}
	return -1;
}

void ip_pool_free(ip_pool_t *ipp)
{
	free(ipp->ips);
	memset(ipp, 0, sizeof(*ipp));
}

int add_ip(ip_pool_t *ipp, u_int32_t ip, int ignored)
{
	int i = 0;
	
	if (ipp->count == ipp->size)
	{
		if (ip_pool_grow(ipp))
		{
			return -1;
		}		
	}
	for (;i < ipp->count; ++i)
	{
		if (ntohl(ipp->ips[i].ip) >= ntohl(ip))
		{
			if (ipp->ips[i].ip != ip)
			{
				memmove((char*)&ipp->ips[i+1], (char*)&ipp->ips[i],
					(ipp->count - i) * sizeof(*ipp->ips));
			}
			else
			{
				if (ipp->ips[i].ignored)
				{
					// IP is ignored
					return 0;
				}
				ipp->count--; /* hack, because we do not need duplicate ip's */				
			}				
			break;
		}
	}
	ipp->count++;
	ipp->ips[i].ip = ip;
	ipp->ips[i].ignored = ignored;
	return 0;	
}

void html_output(ip_pool_t *ipp)
{
	int i;
	printf("Content-Type: text/html\r\n\r\n");
	for (i = 0; i < ipp->count; ++i)
	{
		if (ipp->ips[i].ignored)
		{
			continue;
		}
		printf("%1$s\n", 
			inet_ntoa(*(struct in_addr*)(&ipp->ips[i].ip)));		
	}
}

int get_arps(ip_pool_t *ipp)
{
	char line[256];
	char ip[16];
	int flags;
	int result;
	struct in_addr ip_addr;
	FILE* fp = fopen(FILE_PROC_NET_ARP, "r");
	
	if (!fp)
	{
		return -1;
	}
	
	if (fgets(line, sizeof(line), fp) != NULL)
	{
		for(;(result = fscanf(fp, "%15s 0x%*x 0x%x %*s %*s %*s\n", ip, &flags)) == 2;)
		{
			if ((flags & ATF_COM) && inet_aton(ip, &ip_addr))
			{
				add_ip(ipp, ip_addr.s_addr, 0);
			}
		}
	}	
	fclose(fp);
	return 0;
}

static int send_interfaces_broadcast(int s, const void *buf, size_t len,
		ip_pool_t *ipp)
{
	int result = 0;
	int numreqs = 30;
	int nsent = 0;
	struct ifconf ifc;
	struct ifreq *ifr;
	int i;
	int fd;
	struct sockaddr_in addr;
	void *tmp;
	size_t tmp_len;

	memset(&ifc, 0, sizeof(ifc));

	memset(&addr, 0, sizeof(addr));
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = htonl(INADDR_BROADCAST);

	ifc.ifc_buf = NULL;
	fd = socket(AF_INET, SOCK_DGRAM, 0);
	if (fd < 0)
	{
		/* Just try broadcast on failure... */
		return sendto(s, buf, len, 0, (struct sockaddr *)&addr, sizeof(addr));
	}

	for (;;)
	{
		tmp_len = sizeof(struct ifreq) * numreqs;
		tmp = realloc(ifc.ifc_buf, tmp_len);
		if(!tmp)
		{
			break;
		}		
		ifc.ifc_len = tmp_len;
		ifc.ifc_buf = tmp;
		
		if (ioctl(fd, SIOCGIFCONF, &ifc) < 0)
		{
			break;
		}
		if (ifc.ifc_len == sizeof(struct ifreq) * numreqs)
		{
			/* assume it overflowed and try again */
			numreqs += 10;
			continue;
		}
		break;
	}

	ifr = ifc.ifc_req;
	for (i = 0; i < ifc.ifc_len; i += sizeof(struct ifreq), ifr++)
	{
		if (ioctl(fd, SIOCGIFFLAGS, ifr) < 0)
		{
			continue;
		}
		if (!((ifr->ifr_flags & IFF_UP) && (ifr->ifr_flags & IFF_BROADCAST)))
		{
			continue;
		}
		if (ioctl(fd, SIOCGIFBRDADDR, ifr) >= 0)
		{
			addr.sin_addr.s_addr =
				((struct sockaddr_in*)&ifr->ifr_broadaddr)->sin_addr.s_addr;
			result = sendto(s, buf, len, 0, (struct sockaddr *)&addr,
				sizeof(addr));
			if (result == len)
			{
				nsent++;
			}
		}		 
		if (ioctl(fd, SIOCGIFADDR, ifr) >= 0)
		{
			add_ip(ipp, ((struct sockaddr_in*)&ifr->ifr_addr)->sin_addr.s_addr,
					1);
		}
	}

	if (nsent == 0)
	{
		/* Just try broadcast on failure... */
		result = sendto(s, buf, len, 0, (struct sockaddr *)&addr, sizeof(addr));
	}
	
	close(fd);
	free(ifc.ifc_buf);
	return result;	
}

#define ICMP_DATA 56 

int get_bicmp(ip_pool_t *ipp)
{
	char buf[1500];
	char *icmp_packet = buf;
	struct iphdr* ip_hdr = (struct iphdr*)icmp_packet;
	struct icmphdr* icmp_hdr = (struct icmphdr*)icmp_packet;
	struct sockaddr_in addr;
	int broadcast = 1;
	int s = socket(PF_INET, SOCK_RAW, IPPROTO_ICMP);
	int result;
	int ip_size;
	struct timeval timeout = {0, 200000};
	struct timeval now, end_time;
	fd_set read_fds;
	
	if (s < 0)
	{
		return -1;		
	}
	
	if (setsockopt(s, SOL_SOCKET, SO_BROADCAST, &broadcast, sizeof(broadcast))
		 < 0)
	{
		return -2;
	}
		
	memset(&addr, 0, sizeof(addr));
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = htonl(INADDR_BROADCAST);

	memset (icmp_packet, 0, sizeof(addr));
	icmp_hdr->type = ICMP_ECHO;
	icmp_hdr->checksum = checksum((u_int16_t *) icmp_packet, 
		sizeof(*icmp_hdr) + ICMP_DATA);
	
	result = send_interfaces_broadcast(s, 
		icmp_packet, sizeof(*icmp_hdr) + ICMP_DATA, ipp);		
	if (result != sizeof(*icmp_hdr) + ICMP_DATA)
	{
		return -3;
	}
	
	gettimeofday(&now, NULL);
	timeradd(&now, &timeout, &end_time);
	
	FD_ZERO(&read_fds);
	FD_SET(s, &read_fds);
	while(1)
	{
		gettimeofday(&now, NULL);
		if (timercmp(&now, &end_time, >))
		{
			break;
		}
		timersub(&end_time, &now, &timeout);
		result = select(s+1, &read_fds, NULL, NULL, &timeout);
		if (result <= 0)
		{
			break;
		}
		while(1)
		{
			result = recv(s, buf, sizeof(buf), MSG_DONTWAIT);
			if (result < 0 && errno == EAGAIN)
			{
				result = 0;
				break;
			}
			if (result <= 0)
			{
				result = -1;
				break;
			}
			
			if (result < sizeof(*ip_hdr))
			{
				continue;
			}
			ip_size = ip_hdr->ihl << 2;
			icmp_hdr = (struct icmphdr *)(buf + ip_size);
			if (result == ip_size + sizeof(*icmp_hdr) + ICMP_DATA
				&& icmp_hdr->type == ICMP_ECHOREPLY)
			{
				add_ip(ipp, ip_hdr->saddr, 0);
			}
		}
		if (result < 0)
		{
			break;
		}
	}
	close(s);
	return 0;
}

#ifdef IPSCAN
int main(int argc, const char* argv[])
#else
int ipscan_main(void)
#endif
{
	ip_pool_t ipp;
	memset(&ipp, 0, sizeof(ipp));
	get_bicmp(&ipp);
	get_arps(&ipp);
	html_output(&ipp);
	ip_pool_free(&ipp);
	return 0;
}
