#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <arpa/inet.h>
#include <netinet/ip_icmp.h>

#define DEFAULT_PACKET_SIZE 56

static int nreceived;

static int usage(const char* progname, ...)
{
	va_list ap;
	char *fmt;
	int rv = 0;
	
	va_start(ap, progname);
	fmt = va_arg(ap, char *);
	if (fmt)
	{
		vprintf(fmt, ap);
		rv = -1;
	}
	va_end(ap);
	printf("Web ping utility v0.1 (c) Ubiquiti Networks\n");
	printf("Usage: %s [options] <ip_address>\n"
		"\t-s <bytes>\t\tnumber of data bytes to be sent; default: %d\n"
		"\t-h\t\t\tdisplay this help and exit.\n",				
		progname, DEFAULT_PACKET_SIZE);
	return rv;		
}

static void alarm_handler(int signum)
{
	exit(nreceived ? 0: -4);
}

static int checksum(u_int16_t *buf, size_t size)
{
	int nleft = size;
	int sum = 0;
	u_int16_t *p = buf;
	u_int16_t result = 0;

	while (nleft > 1)
	{
		sum += *p++;
		nleft -= 2;
	}

	if (nleft == 1)
	{
		*(u_int8_t *)(&result) = *(u_int8_t *)p;
		sum += result;
	}

	sum = (sum >> 16) + (sum & 0xFFFF);
	sum += (sum >> 16);
	result = ~sum;
	return result;
}

static void randomfill(u_int8_t *buf, size_t size)
{
	struct timeval now;
	
	gettimeofday(&now, NULL);
	srandom((unsigned int)(now.tv_sec ^ now.tv_usec));
	while (size > 0)
	{
		*buf++ = (u_int8_t)(random() & 0xff);
		size--;
	}	
}

static void init_icmp(struct icmphdr* packet, size_t size)
{
	/* it will randomly initialize id and sequence also */
	randomfill((u_int8_t*)packet, size);
	
	packet->type = ICMP_ECHO;
	packet->code = 0;
}

static void prepare_icmp(struct icmphdr* packet, size_t size)
{
	packet->un.echo.sequence++;
	/* Timestamp packet */
	gettimeofday((struct timeval*)(packet+1), NULL);
	/* set to 0 before calculating actual checksum */
	packet->checksum = 0;
	packet->checksum = checksum((u_int16_t *) packet, size);
}

void send_icmp(int sock, struct icmphdr* packet,
	size_t packet_size, const struct sockaddr_in *dst)
{
	int result;

	prepare_icmp(packet, packet_size);
	result = sendto(sock, packet, packet_size, 0,
			   (struct sockaddr *)dst, sizeof(*dst));
	if (result <= 0)
	{
		exit(-3);
	}
	return;
}

void recv_icmp(int sock, const struct icmphdr* packet, size_t packet_size)
{
	int result;
	char buf[packet_size + 60];
	struct iphdr *ip_hdr = (struct iphdr*)buf;
	struct icmphdr *icmp_hdr;
	int ip_size;
	struct timeval travel_time;

	for(;;)
	{
		result = recv(sock, buf, sizeof(buf), 0);
		gettimeofday(&travel_time, NULL);
		if (result <= 0)
		{
			continue;		
		}
		if (result < sizeof(*ip_hdr))
		{
			continue;
		}
		ip_size = ip_hdr->ihl << 2;
		if (result != ip_size + packet_size)
		{
			continue;
		}		
		icmp_hdr = (struct icmphdr *)(buf + ip_size);
		if (icmp_hdr->type == ICMP_ECHOREPLY
			&& icmp_hdr->un.echo.id == packet->un.echo.id
			&& icmp_hdr->un.echo.sequence == packet->un.echo.sequence)
		{				
			timersub(&travel_time, (struct timeval*)(packet+1), &travel_time);				
			printf("%hhu.%hhu.%hhu.%hhu|%ld.%02ld|%d\n", 
				((u_int8_t *)(&ip_hdr->saddr))[0],
				((u_int8_t *)(&ip_hdr->saddr))[1],
				((u_int8_t *)(&ip_hdr->saddr))[2],
				((u_int8_t *)(&ip_hdr->saddr))[3],
				travel_time.tv_sec*1000 + travel_time.tv_usec/1000,
				(travel_time.tv_usec % 1000)/10,
				ip_hdr->ttl);
			++nreceived;				
		}		
	}
	return;		
}

static int do_ping(const struct sockaddr_in *dst_addr, size_t size)
{
	char buf[size];
	struct icmphdr *packet = (struct icmphdr *)buf;
	struct itimerval timer = {{0,0},{0,0}};
	timer.it_value.tv_usec = 800000;
	int value_1 = 1;
	
	int sock = socket(PF_INET, SOCK_RAW, IPPROTO_ICMP);	
	setsockopt(sock, SOL_SOCKET, SO_BROADCAST, &value_1, sizeof(value_1));

	if (sock < 0)
	{
		return -2;
	}	
	signal(SIGALRM, alarm_handler);	
	init_icmp(packet, sizeof(buf));
	send_icmp(sock, packet,	sizeof(buf), dst_addr);
	setitimer(ITIMER_REAL, &timer, NULL);
	recv_icmp(sock, packet, sizeof(buf));
	return 0;
}

#ifdef WEBPING
int main(int argc, char* argv[])
#else
int webping_main(int argc, char* argv[])
#endif
{
	int data_size = DEFAULT_PACKET_SIZE;

	int int_value;		
	struct sockaddr_in dst_addr;
	
	memset(&dst_addr, 0, sizeof(dst_addr));
	dst_addr.sin_family = AF_INET;
	
	while (1)
	{
	    int c = getopt(argc, argv, ":s:h");
	    if (c == -1)
	    {
	    	/* finished options */
	    	break;
	    }
	    switch(c)
	    {
	    	case 's':
	    		int_value = atoi(optarg);
	    		if (int_value < 28 || int_value > 65507)
	    		{
	    			return usage(argv[0], "Invalid packet size. Allowed values [28-65507].\n");
	    		}
	    		data_size = int_value;
	    		break;
	    		break;
	    	case '?':
	    		return usage(argv[0], "Unknown option: %c\n", optopt); 
	    		break;
	    	case ':':
	    		return usage(argv[0], "Option '%c' requires an argument.\n", optopt); 
    			break;
    		case 'h':
    			return usage(argv[0], NULL);
	    }
	}
	if (optind >= argc)
	{
		return usage(argv[0], "No IP address specified.\n"); 
	}
	if (inet_aton(argv[optind], &dst_addr.sin_addr) == 0)
	{
		return usage(argv[0], "Invalid IP address specified.\n"); 	
	}
	return do_ping(&dst_addr, data_size);
}
