#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <linux/if.h>
#include <netinet/if_ether.h>
#include <linux/types.h>
//USE local copy of wireless.h
#include "wireless.h"
#include <include/compat.h>
#include <net80211/ieee80211_ioctl.h>

/* Problematic including into userspace apps: */
/* #include <net80211/ieee80211_node.h> */
/* Use single required define instead. */
#ifndef IEEE80211_NODE_STATIC
#define IEEE80211_NODE_STATIC	0x8000	/* staticaly added node (UBNT) */
#endif /* IEEE80211_NODE_STATIC */

#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include "common.h"


#define __MAC2STR(a) (a)[0], (a)[1], (a)[2], (a)[3], (a)[4], (a)[5]
#define __MACSTR "%02X:%02X:%02X:%02X:%02X:%02X"
#define ARRAYSIZE(x) (sizeof(x) / sizeof((x)[0]))

#define PROC_ANTENNA_DIVERSITY "/proc/sys/dev/wifi0/diversity"
#define PROC_TX_ANTENNA "/proc/sys/dev/wifi0/txantenna"
#define PROC_RX_ANTENNA "/proc/sys/dev/wifi0/rxantenna"
#define PROC_EXT_ANTENNA "/proc/sys/dev/wifi0/extantenna"
#define PROC_COUNTRYCODE "/proc/sys/dev/wifi0/countrycode"

static int
read_int(const char* file, int* value) {
	FILE* f;
	char line[255];

	f = fopen(file, "r");
	if (!f) {
		return -1;
	}

	if (!fgets(line, sizeof(line), f)) {
		fclose(f);
		return -1;
	}
	if (sscanf(line, "%d", value) != 1) {
		fclose(f);
		return -1;
	}
	fclose(f);
	return 0;
}

static inline int
fill_rx_rate(wlan_info_t* status, int turbo) {
	const char RXRATE[] = "/proc/sys/dev/wifi0/rx_rate";
	int rate;
	int rc = read_int(RXRATE, &rate);
	if (rc == 0) {
		status->rx_rate = (double)rate / (turbo ? 1 : 2);
	}
	return rc;
}

static inline int
fill_acktimeout(wlan_info_t* status) {
	const char ACKTIME[] = "/proc/sys/dev/wifi0/acktimeout";
	int ack;
	int rc = read_int(ACKTIME, &ack);
	if (rc == 0) {
		status->acktimeout = ack;
	}
	return rc;
}

static inline int
fill_countrycode(wlan_info_t* status) {
	int ccode;
	int rc = read_int(PROC_COUNTRYCODE, &ccode);
	if (rc == 0) {
		status->countrycode = ccode;
	}
	return rc;
}


static inline int
get_wpa_mode(int fd, const char* ifname) {
	struct iwreq wrq;
	int rc;

	memset(&wrq, 0, sizeof(wrq));
        wrq.u.mode = IEEE80211_PARAM_WPA;
        strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
	rc = ioctl(fd, IEEE80211_IOCTL_GETPARAM, &wrq);
	if (rc < 0) {
		return -1;
	}

	return wrq.u.mode;
}

static int get_bandwidth(int fd, const char* ifname)
{
	struct iwreq wrq;
	int rc;
	char name[IFNAMSIZ + 1];

	memset(&wrq, 0, sizeof(wrq));
    strncpy(wrq.ifr_name, ifname, IFNAMSIZ);	
	rc = ioctl(fd, SIOCGIWNAME, &wrq);
	if (rc < 0)
	{
		return -1;
	}
    strncpy(name, wrq.u.name, IFNAMSIZ);
    name[IFNAMSIZ] = '\0';
	if (strcmp(name, "IEEE 802.11STa") == 0)
	{
		/* Turbo Static A */
		return 40;
	}
    wrq.u.mode = IEEE80211_PARAM_CHANBW;
	rc = ioctl(fd, IEEE80211_IOCTL_GETPARAM, &wrq);
	if (rc < 0)
	{
		return -1;
	}
	switch (wrq.u.mode)
	{
	case 2:
		return 5;
	case 1:
		return 10;
	}
	return 20;	
}

static char* format_rate(int rate, int bandwidth, char* str)
{
	if (bandwidth == 40)
	{
		sprintf(str, "%u", rate * 2);
	}
	else if (bandwidth == 10)
	{
		if (rate % 2)
		{
			sprintf(str, "%u.5", rate / 2);			
		}
		else
		{		
			sprintf(str, "%u", rate / 2);			
		}
	}
	else if (bandwidth == 5)
	{
		if (rate % 4 == 0)
		{
			sprintf(str, "%u", rate / 4);			
		}
		else if (rate % 4 % 2)
		{
			sprintf(str, "%u.%u", rate / 4, 25 * (rate % 4));						
		}
		else
		{
			sprintf(str, "%u.5", rate / 4);
		}
	}
	else
	{
		sprintf(str, "%u", rate);		
	}
	return str;
}

static int
get_ccq(int fd, const char* ifname, const char* hwaddr) {
	struct iwreq wrq;
	int len, ccq = 0;
	unsigned char* buf;
	unsigned char* p;

#define STA_BUF_SIZE 16 * 1024

	buf = (unsigned char*)calloc(1, STA_BUF_SIZE);
	if (!buf) {
		return 0;
	}
	memset(&wrq, 0, sizeof(wrq));
	strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
	wrq.u.data.pointer = (void*)buf;
	wrq.u.data.length = STA_BUF_SIZE;

	if (ioctl(fd, IEEE80211_IOCTL_STA_INFO, &wrq) < 0) {
		free(buf);
		return 0;
	}

	len = wrq.u.data.length;
	if (len < sizeof(struct ieee80211req_sta_info)) {
		free(buf);
		return 0;
	}
	p = buf;
	do {
		struct ieee80211req_sta_info* si = (struct ieee80211req_sta_info *)p;
		if (memcmp(hwaddr, si->isi_macaddr, 6) == 0) {
			ccq = si->isi_ccq;
			break;
		}
		p += si->isi_len;
		len -= si->isi_len;
	} while (len >= sizeof(struct ieee80211req_sta_info));

	free(buf);
	return ccq;
#undef STA_BUF_SIZE
}


typedef struct ant_info {
    int id;
    char name[64];
} ant_info_t;

static ant_info_t ant_info[10];

static int init_ant(void) {
	FILE* f;
	char line[255];
	char key[32];
	int idx = 1;
        int i = 0;
        char* s;

	f = fopen("/etc/board.info", "r");
	if (!f) {
		return -1;
	}

	while (!feof(f) && fgets(line, sizeof(line), f) != NULL) {
	    sprintf(key, "radio.1.antenna.%d.id=", idx);
	    s = strstr(line, key);
	    if (!s) {
		sprintf(key, "radio.1.antenna.%d.name=", idx);
		s = strstr(line, key);
		if (!s) continue;
		strcpy(ant_info[i].name, s + strlen(key));
		ant_info[i].name[strlen(ant_info[i].name) - 1] = 0;
		++i;
                ++idx;
                continue;
	    }
	    ant_info[i].id = atoi(s + strlen(key));
	}
	fclose(f);
	return 0;
}

static int fix_ant(char* name, int id) {
        int i = 0;

	while (ant_info[i].id || *(ant_info[i].name))
	{
	    if (ant_info[i].id == id) {
		strcpy(name, ant_info[i].name);
                break;
	    }
	    ++i;
	}
	return 0;
}


static inline int
fill_antenna(antenna_t* antenna) {
        FILE* fd;
	int rc = -1;

	init_ant();

	if (!antenna || !(fd = fopen(PROC_ANTENNA_DIVERSITY, "r")))
	    return -1;

	if (fscanf(fd, "%d", &rc) < 1 || rc < 0)
	{
            fclose(fd);
	    return -1;
	}
        antenna->diversity = rc;
        fclose(fd);

	if (!(fd = fopen(PROC_TX_ANTENNA, "r")))
	    return -1;

	if (fscanf(fd, "%d", &rc) < 1 || rc < 0)
	{
            fclose(fd);
	    return -1;
	}
        antenna->tx = rc;
        fclose(fd);

	if (!(fd = fopen(PROC_RX_ANTENNA, "r")))
	    return -1;

	if (fscanf(fd, "%d", &rc) < 1 || rc < 0)
	{
            fclose(fd);
	    return -1;
	}
        antenna->rx = rc;
        fclose(fd);
	if (!(fd = fopen(PROC_EXT_ANTENNA, "r")))
	    return -1;

	if (fscanf(fd, "%d", &rc) < 1 || rc < 0)
	{
		fclose(fd);
		return -1;
	}
	antenna->ext = rc;
	fclose(fd);

	strcpy(antenna->name, "Unknown");
	if (antenna->diversity) {
	    strcpy(antenna->name, "Adaptive");
            fix_ant(antenna->name, 4);
	}
	else if (antenna->ext)  {
	    strcpy(antenna->name, "External");
	    fix_ant(antenna->name, 5);
	}
	else if (antenna->tx == antenna->rx) {
		fix_ant(antenna->name, antenna->tx);
		if (0 == strcmp("Unknown",antenna->name)) {
			switch (antenna->tx) {
			case 1:
				strcpy(antenna->name, "Horizontal");
				break;
			case 2:
				strcpy(antenna->name, "Vertical");
				break;
			default:
				// Leave unknown
				break;
			}
		}
	}

	return 0;
}


static inline int
get_privacy(int fd, const char* ifname) {
	struct iwreq wrq;
	int rc;

	memset(&wrq, 0, sizeof(wrq));
        wrq.u.mode = IEEE80211_PARAM_PRIVACY;
        strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
	rc = ioctl(fd, IEEE80211_IOCTL_GETPARAM, &wrq);
	if (rc < 0) {
		return -1;
	}

	return wrq.u.mode;
}

typedef struct wmm_level {
	char name[32];
	int value;
} wmm_level_t;

static wmm_level_t wmm_levels[] = {
	{ "No QoS", -1 },
	{ "Auto Priority", 0 },
	{ "Video Priority", 4 },
	{ "Voice Priority", 8 },
};

static inline int
get_wmm_levelname(int fd, const char* ifname, char* levelname, size_t levelname_len) {
	struct iwreq wrq;
	int rc, i;

	strncpy(levelname, wmm_levels[0].name, levelname_len);

	memset(&wrq, 0, sizeof(wrq));
        wrq.u.mode = IEEE80211_PARAM_WMM;
        strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
	rc = ioctl(fd, IEEE80211_IOCTL_GETPARAM, &wrq);
	if (rc < 0) {
		return -1;
	}

	/* quit if WMM is disabled */
	if (wrq.u.mode == 0) {
		return 0;
	}

	/* if WMM is enabled, we have to figure out the level */
	memset(&wrq, 0, sizeof(wrq));
        wrq.u.mode = IEEE80211_PARAM_WMM_LEVEL;
        strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
	rc = ioctl(fd, IEEE80211_IOCTL_GETPARAM, &wrq);
	if (rc < 0) {
		return -1;
	}

	for (i = 1; i < sizeof(wmm_levels) / sizeof(wmm_level_t); ++i) {
		if (wmm_levels[i].value == wrq.u.mode) {
			strncpy(levelname, wmm_levels[i].name, levelname_len);
			break;
		}
	}
	return 0;
}

int
fill_link_stats(const char* ifname, wlan_info_t* status) {
	struct iwreq wrq;
	struct iw_statistics stats;
	int fd, rc;

        fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
        if (fd < 0) {
                return -1;
        }
	memset(&wrq, 0, sizeof(wrq));
	wrq.u.data.pointer = (void *)&stats;
        wrq.u.data.length = sizeof(struct iw_statistics);
        wrq.u.data.flags = 1;
        strncpy(wrq.ifr_name, ifname, IFNAMSIZ);

	rc = ioctl(fd, SIOCGIWSTATS, &wrq);
	if (rc >= 0) {
#define IEEE80211_S_RUN		4
		status->rstatus = stats.status;
		status->noisef = stats.qual.noise - 256;
		if (stats.status == IEEE80211_S_RUN) {
			status->signal = stats.qual.level - 256;
		} else {
			status->signal = status->noisef;
			status->rx_rate = 0;
		}
		status->rx_nwids = stats.discard.nwid;
		status->rx_crypts = stats.discard.code;
		status->rx_frags = stats.discard.fragment;
		status->tx_retries = stats.discard.retries;
		status->missed_beacons = stats.miss.beacon;
		status->other_errors = stats.discard.misc;
	}

	close(fd);
	return 0;
}

int
fill_wireless_info(const char* ifname, wlan_info_t* status) {
	struct iwreq wrq;
	char range_buf[sizeof(struct iw_range)];
	struct iw_range* range = 0;
	int rc;
	int fd;
	int bw;

	fill_antenna(&status->antenna);
	fill_countrycode(status);
	
	fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
	if (fd < 0) {
		return 0;
	}
	bw = get_bandwidth(fd, ifname);
	fill_acktimeout(status);

	/* ESSID */
	memset(&wrq, 0, sizeof(wrq));
	wrq.u.essid.pointer = (caddr_t)status->essid;
	wrq.u.essid.length = sizeof(status->essid);
	wrq.u.essid.flags = 0;
        strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
	rc = ioctl(fd, SIOCGIWESSID, &wrq);
	if (rc < 0) {
		status->essid[0] = '\0';
	}
	/* OPERATION MODE */
	memset(&wrq, 0, sizeof(wrq));
        strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
	rc = ioctl(fd, SIOCGIWMODE, &wrq);
	if (rc >= 0) {
		if (wrq.u.mode == 2)
			status->mode = 1;
		else if (wrq.u.mode == 3)
			status->mode = 2;
	}

	/* AP MAC */
	memset(&wrq, 0, sizeof(wrq));
        strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
	rc = ioctl(fd, SIOCGIWAP, &wrq);
	if (rc >= 0) {
		memcpy(status->apmac, wrq.u.ap_addr.sa_data, 6);
	}

	/* FREQUENCY RANGE INFO */
	memset(&wrq, 0, sizeof(wrq));
	wrq.u.data.pointer = (caddr_t)range_buf;
	wrq.u.data.length = sizeof(range_buf);
	wrq.u.data.flags = 0;
        strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
	rc = ioctl(fd, SIOCGIWRANGE, &wrq);
	if (rc >= 0) {
		range = (struct iw_range*)range_buf;
	}

	/* FREQUENCY */
	memset(&wrq, 0, sizeof(wrq));
        strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
	rc = ioctl(fd, SIOCGIWFREQ, &wrq);
	if (rc >= 0) {
		int i;
		double freq = wrq.u.freq.m;
		for (i = 0; i < wrq.u.freq.e; ++i)
			freq *= 10;
		status->frequency = freq / 1e6;
		if (range) {
			for (i = 0; i < range->num_frequency; ++i) {
				if (wrq.u.freq.m == range->freq[i].m &&
				    wrq.u.freq.e == range->freq[i].e)
					status->channel = range->freq[i].i;
			}
		}

		if (status->countrycode == 1003) {
			status->frequency -= 2000;
		}
	}

	fill_rx_rate(status, bw == 40);
	/* WIRELESS STATISTICS */
	fill_link_stats(ifname, status);
	/* TX RATE */
	memset(&wrq, 0, sizeof(wrq));
        strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
	rc = ioctl(fd, SIOCGIWRATE, &wrq);
	if (rc >= 0) {
		status->tx_rate = (double)wrq.u.bitrate.value / (bw == 40 ? 5e5 : 1e6);
	}

	/* CCQ for STA only now */
	if (status->mode == 1) {
		status->ccq = get_ccq(fd, ifname, status->apmac);
	}

	/* SECURITY: WEP/WPA/WPA2/none */
	if ((rc = get_wpa_mode(fd, ifname)) > 0) {
		strncpy(status->security, rc == 1 ? "WPA" : "WPA2", sizeof(status->security));
	} else if (get_privacy(fd, ifname) > 0) {
		strncpy(status->security, "WEP", sizeof(status->security));
	} else {
		strncpy(status->security, "none", sizeof(status->security));
	}

	/* WMM LEVEL */
	get_wmm_levelname(fd, ifname, status->wmm_level, sizeof(status->wmm_level));

	close(fd);
	return 0;
}

static void put_uptime(u_int32_t uptime)
{
	printf("%u", uptime);
}

static int put_nodestats(int s, const char* ifname,
	const u_int8_t macaddr[IEEE80211_ADDR_LEN])
{
	struct iwreq iwr;
	struct ieee80211req_sta_stats stats;
	const struct ieee80211_nodestats *ns = &stats.is_stats;

	(void) memset(&iwr, 0, sizeof(iwr));
	(void) strncpy(iwr.ifr_name, ifname, sizeof(iwr.ifr_name));
	iwr.u.data.pointer = (void *) &stats;
	iwr.u.data.length = sizeof(stats);
	memcpy(stats.is_u.macaddr, macaddr, IEEE80211_ADDR_LEN);
	if (ioctl(s, IEEE80211_IOCTL_STA_STATS, &iwr) < 0)
	{
		return -1;
	}
	printf("  Stats: %u|%llu|%u|%llu\n", ns->ns_rx_data, ns->ns_rx_bytes,
		ns->ns_tx_data, ns->ns_tx_bytes);
	return 0;
}

int
print_wstalist(const char* ifname, const char* mac) {
	uint8_t* buf;
	struct iwreq iwr;
	uint8_t *cp;
	int s, len;
	int i;
	struct ieee80211req_sta_info *si;
	int bw = 0;

	buf = (uint8_t*)calloc(16, 1024);
	if (!buf)
		return -1;

	s = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
	if (s < 0) {
		free(buf);
		return -1;
	}

	(void) memset(&iwr, 0, sizeof(iwr));
	(void) strncpy(iwr.ifr_name, ifname, sizeof(iwr.ifr_name));
	iwr.u.data.pointer = (void *) buf;
	iwr.u.data.length = 16 * 1024;
	if (ioctl(s, IEEE80211_IOCTL_STA_INFO, &iwr) < 0) {
		free(buf);
		close(s);
		return -2;
	}

	bw = get_bandwidth(s, ifname);
	
	puts("Station Address"
		" AID"
		" TxRate"
		" RxRate"
		" Signal"
		" CCQ"
		" Idle"
		" Uptime"
		" Static"
		" NoiseFloor"
		" TxPower");
	cp = buf;
	for (len = iwr.u.data.length;
			len >= sizeof(struct ieee80211req_sta_info);
			cp += si->isi_len, len -= si->isi_len)
	{
		uint8_t *vp;
		uint tx = 0, rx = 0;
		uint8_t tx_str[10], rx_str[10];

		si = (struct ieee80211req_sta_info *) cp;
		if (mac && memcmp(mac, si->isi_macaddr, sizeof(si->isi_macaddr)))
		{
			continue;
		}
		vp = (u_int8_t *)(si+1);
		if (si->isi_nrates > si->isi_txrate) {
			tx = (si->isi_rates[si->isi_txrate] & IEEE80211_RATE_VAL)/2;
		} else if (si->isi_txrate & 0x8000) {
			tx = si->isi_txrate & 0x7fff;
		}

		if (si->isi_nrates > si->isi_rxrate) {
			rx = (si->isi_rates[si->isi_rxrate] & IEEE80211_RATE_VAL)/2;
		} else if (si->isi_rxrate & 0x8000) {
			rx = si->isi_rxrate & 0x7fff;
		}

		printf(__MACSTR"|%hu|%s|%s|%hd|%hu|%hu|",
				__MAC2STR(si->isi_macaddr),
			       	IEEE80211_AID(si->isi_associd),
				format_rate(tx, bw, tx_str),
				format_rate(rx, bw, rx_str),
				si->isi_noisefloor + si->isi_rssi,
				si->isi_ccq,
				si->isi_inact
		);
		put_uptime(si->isi_uptime);
		putchar('|');
		printf((si->isi_state & IEEE80211_NODE_STATIC ? "Yes" : "No"));
		printf("|%hd|%hu",si->isi_noisefloor, si->isi_txpower);
		printf("\n  Rates:");
		for (i = 0; i < si->isi_nrates; ++i)
		{
		    rx = (si->isi_rates[i] & IEEE80211_RATE_VAL)/2;
			format_rate(rx, bw, rx_str),
		    printf(" %s", rx_str);
		}
		printf("\n  Signals:");
		for (i = 0; i < si->isi_nrates; ++i)
		{
			printf(" %hhu", si->isi_ratesrssi[i]);
		}
		putchar('\n');
		put_nodestats(s, ifname, si->isi_macaddr);
	}

	free(buf);
	close(s);
	return 0;
}
