/*
 * AR531X GPIO control driver
 * Provides proc interface to reset button and LEDs control through GPIO
 * /proc/gpio/reset_btn
 *	read/write values are [0|1]
 * /proc/gpio/system_led
 *	read values from AR531X GPIO Control and
 *	AR531X GPIO Direction Output registers.
 *       write values are: bit_no dir val [[bit_no dir val] ...]
 *
 * Copyright (C) 2007 Ubiquiti Networks, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/init.h>
#include <asm/gdb-stub.h>
#include <linux/interrupt.h>

#include <ar531xlnx.h>

#ifndef CONFIG_PROC_FS
#error This module requires PROCFS
#endif

#if CONFIG_AR5315
#undef AR531X_APBBASE
#undef AR531X_GPIO
#undef AR531X_GPIO_DI
#undef AR531X_GPIO_DO
#undef AR531X_GPIO_CR

#define AR531X_APBBASE      AR531XPLUS_DSLBASE
#define AR531X_GPIO         AR531X_APBBASE
#define AR531X_GPIO_DO      AR531XPLUS_GPIO_DO        /* output register */
#define AR531X_GPIO_DI      AR531XPLUS_GPIO_DI        /* intput register */
#define AR531X_GPIO_CR      AR531XPLUS_GPIO_CR        /* control register */
#endif

void gpio_line_get(unsigned char line, u32 * data)
{
    *data = (sysRegRead(AR531X_GPIO_DI) >> line) & 1;
}

u32 gpio_line_set(unsigned char line, u32 val)
{
    u32 reg;

    reg = sysRegRead(AR531X_GPIO_DO);
    reg &= ~(1 << line);
    reg |= (val&1) << line;
    sysRegWrite(AR531X_GPIO_DO, reg);
    return sysRegRead(AR531X_GPIO_DO); /* flush write to hardware */
}

u32 gpio_leds_set(u8* leds)
{
    u32 reg_dir;
    u32 reg_out;
    u8 c, n, d, o, i;

    if (!leds || !(c = leds[0]))
        return 0;

    reg_dir = sysRegRead(AR531X_GPIO_CR);
    reg_out = sysRegRead(AR531X_GPIO_DO);

#ifdef DEBUG
    printk("COUNT: %d\n", (int)c);
    printk("AR531X_GPIO_CR: %x\n", AR531X_GPIO_CR);
    printk("Read %x from AR531X_GPIO_CR\n", reg_dir);
    printk("Read %x from AR531X_GPIO_DO\n", reg_out);
#endif
    
    for (i = 1; i <= c; ++i)
    {
	n = leds[i] >> 2;
	d = (leds[i] >> 1) & 0x1;
	o = leds[i] & 0x1;
#ifdef DEBUG
        printk("GPIOCR[%d] = %d, GPIODO[%d] = %d (0x%02X)\n", n, d, n, o, leds[i]);
#endif
	if (d) /* configure for OUTPUT */
	    reg_dir |= (1 << n);
	else /* configure for INPUT */
	    reg_dir &= ~(1 << n);
	if (o) /* write OUTPUT */
	    reg_out |= (1 << n);
	else
	    reg_out &= ~(1 << n);
    }
#ifdef DEBUG
    printk("Write %x to AR531X_GPIO_CR\n", reg_dir);
#endif
    sysRegWrite(AR531X_GPIO_CR, reg_dir);
    reg_dir = sysRegRead(AR531X_GPIO_CR); /* flush */
#ifdef DEBUG
    printk("Read %x from AR531X_GPIO_CR\n", reg_dir);
    printk("Write %x from AR531X_GPIO_DO\n", reg_out);
#endif
    sysRegWrite(AR531X_GPIO_DO, reg_out);
    reg_out = sysRegRead(AR531X_GPIO_DO); /* flush write to hardware */
#ifdef DEBUG
    printk("Read %x from AR531X_GPIO_DO\n", reg_out);
#endif
    return reg_out;
}

u32 gpio_line_config(unsigned char line, unsigned char state)
{
    u32 reg;

    reg = sysRegRead(AR531X_GPIO_CR);
    if (state) /* configure for OUTPUT */
	sysRegWrite(AR531X_GPIO_CR, reg | (state << line));
    else /* configure for INPUT */
	sysRegWrite(AR531X_GPIO_CR, reg & ~(!state << line));
    return sysRegRead(AR531X_GPIO_CR); /* flush */
}

#define GPIO_CHAR_DEV 1

#ifdef GPIO_CHAR_DEV

#include <asm/uaccess.h>
#include <linux/miscdevice.h>
#include "gpio.h"

#define DEVICE_NAME "gpio"

#define GPIO_IS_OPEN             0x01    /* means /dev/gpio is in use     */

static unsigned char gpio_status;        /* bitmapped status byte.       */

static int gpio_ioctl(struct inode *inode, struct file *file,
		      unsigned int cmd, unsigned long arg)
{
    struct gpio_bit bit;
    u32 val;
    u8 leds[33];

    if (cmd == GPIO_SET_LEDS)
    {
	    u8 len;
	    if (copy_from_user(&len, (u8*)arg, sizeof(len)))
		    return -EFAULT;
	    if (copy_from_user(leds, (u8*)arg, len > 33 ? 33 : len + 1))
		    return -EFAULT;
    }
    else 
    { 
	    if (copy_from_user(&bit, (struct gpio_bit *)arg, sizeof(bit)))
		    return -EFAULT;
    }

    switch (cmd) {

    case GPIO_GET_BIT:
	gpio_line_get(bit.bit, &val);
	bit.state = val;
	return copy_to_user((void *)arg, &bit, sizeof(bit)) ? -EFAULT : 0;
    case GPIO_SET_BIT:
	val = bit.state;
	gpio_line_set(bit.bit, val);
	return 0;
    case GPIO_GET_CONFIG:
	gpio_line_config(bit.bit, bit.state);
	return copy_to_user((void *)arg, &bit, sizeof(bit)) ? -EFAULT : 0;
    case GPIO_SET_CONFIG:
	val = bit.state;
	gpio_line_config(bit.bit, bit.state);
	return 0;
    case GPIO_SET_LEDS:
	gpio_leds_set(leds);
	return 0;
    }
    return -EINVAL;
}


static int gpio_open(struct inode *inode, struct file *file)
{
    if (gpio_status & GPIO_IS_OPEN)
	return -EBUSY;

    MOD_INC_USE_COUNT;

    gpio_status |= GPIO_IS_OPEN;

    return 0;
}


static int gpio_release(struct inode *inode, struct file *file)
{
    /*
     * Turn off all interrupts once the device is no longer
     * in use and clear the data.
     */

    gpio_status &= ~GPIO_IS_OPEN;
    MOD_DEC_USE_COUNT;

    return 0;
}


/*
 *      The various file operations we support.
 */

static struct file_operations gpio_fops = {
    .owner          = THIS_MODULE,
    .ioctl          = gpio_ioctl,
    .open           = gpio_open,
    .release        = gpio_release,
};

static struct miscdevice gpio_dev =
{
    .minor          = 0,
    .name           = "gpio",
    .fops           = &gpio_fops,
};

#endif /* GPIO_CHAR_DEV */

#define GP_PROC_NAME		"gpio"
#define GP_RESET_BTN		"reset_btn"
#define GP_SYS_LED			"system_led"

/* we should get these values from board configuration page on the FLASH */
#define AR531X_RESET_GPIO	6

/* These GPIO should be configured for output */
#define AR531X_LED_GPIO	7

/* proc filesystem functions */
static int ar531x_reset_read(char *page, char **start, off_t off,
			     int count, int *eof, void *data)
{
    u32 gpioDataIn;

    /* get button state */
    gpioDataIn = (sysRegRead(AR531X_GPIO_DI) >> AR531X_RESET_GPIO) & 1;
    return snprintf(page, count, "%d\n", gpioDataIn ? 0 : 1 );
}

static int ar531x_reset_write(struct file *file, const char *buf,
			      unsigned long count, void *data)
{
    return count;
}

static int ar531x_led_read(char *page, char **start, off_t off,
			   int count, int *eof, void *data)
{
    return snprintf(page, count, "LEDS: %x %x\n", sysRegRead(AR531X_GPIO_CR), sysRegRead(AR531X_GPIO_DO));
}

static int ar531x_led_write(struct file *file, const char *buf,
			    unsigned long count, void *data)
{
    u32 i = 0, c = 0;
    u32 nr, dir, out;
    const char* p = buf;
    u8 leds[33];

    if (count < 5 || !p)
	return -EIO;

    leds[0] = 0;
#ifdef DEBUG
    printk("ARG: %s\n", buf);
#endif
    while (p && *p && (leds[0] < 32))
    {
        ++p; ++i;
	if (!*p || *p == ' ')
	{
            ++c;
	    if (c != 3)
		continue;
            c = 0;
	}
	else
	    continue;
#ifdef DEBUG
	printk("PARSE: _%s_\n", (p - i));
#endif
	c = sscanf((p - i), "%u %u %u", &nr, &dir, &out);
#ifdef DEBUG
	printk("COUNT: %lu [%u %u %u]\n", count, nr, dir, out);
#endif
        i = 0;
	if (c != 3)
	{
            c = 0;
	    continue;
	}
        c = 0;
	++leds[0];
	leds[leds[0]] = (nr << 2) | ((dir ? 1 : 0) << 1) | (out ? 1 : 0);
    }
    gpio_leds_set(leds);
    return count;
}

/* procfs */
struct proc_dir_entry *ar531x_gpio_entry=NULL;
struct proc_dir_entry *ar531x_reset_entry;
struct proc_dir_entry *ar531x_led_entry;

int __init ar531x_gpio_setup(void)
{
#ifdef GPIO_CHAR_DEV
    int retval;

    /* register /dev/gpio file ops */
    retval = register_chrdev(GPIO_MAJOR, DEVICE_NAME, &gpio_fops);
    if(retval < 0)
	return retval;
#endif

    /* register /proc/gpio */
    ar531x_gpio_entry = proc_mkdir(GP_PROC_NAME, NULL);

    if (!ar531x_gpio_entry) {
	printk("Failed to create /proc/%s\n", GP_PROC_NAME);
	return -ENOENT;
    }

    /* create /proc/gpio/reset_btn */
    ar531x_reset_entry = create_proc_entry(GP_RESET_BTN, 0644,
					   ar531x_gpio_entry);
    if (ar531x_reset_entry) {
	ar531x_reset_entry->write_proc = ar531x_reset_write;
	ar531x_reset_entry->read_proc = ar531x_reset_read;
    }
    /* create /proc/gpio/system_led */
    ar531x_led_entry = create_proc_entry(GP_SYS_LED, 0644,
					 ar531x_gpio_entry);
    if (ar531x_led_entry) {
	ar531x_led_entry->write_proc = ar531x_led_write;
	ar531x_led_entry->read_proc = ar531x_led_read;
    }
    return 0;
}

module_init(ar531x_gpio_setup);

static void __exit
ar531x_gpio_unload(void)
{
    remove_proc_entry(GP_RESET_BTN, ar531x_gpio_entry);
    remove_proc_entry(GP_SYS_LED, ar531x_led_entry);
    remove_proc_entry(GP_PROC_NAME, NULL);
#ifdef GPIO_CHAR_DEV
    misc_deregister(&gpio_dev);
#endif
}


module_exit(ar531x_gpio_unload);
MODULE_AUTHOR("Unknown");
MODULE_DESCRIPTION("AR531X GPIO control");
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif

