/***************************************************************************
* eeprom read/write support without HAL
* Used to write regdomain
***************************************************************************/
#include <linux/delay.h>

// some registers, stolen from HAL
#define	AR_EEPROM_ADDR	0x6000	/* EEPROM address register (10 bit) */
#define	AR_EEPROM_DATA	0x6004	/* EEPROM data register (16 bit) */
#define	AR_EEPROM_CMD	0x6008	/* EEPROM command register */
#define	AR_EEPROM_STS	0x600c	/* EEPROM status register */

/* EEPROM Registers in the MAC */
#define	AR_EEPROM_CMD_READ	0x00000001
#define	AR_EEPROM_CMD_WRITE	0x00000002
#define	AR_EEPROM_CMD_RESET	0x00000004

#define	AR_EEPROM_STS_READ_ERROR	0x00000001
#define	AR_EEPROM_STS_READ_COMPLETE	0x00000002
#define	AR_EEPROM_STS_WRITE_ERROR	0x00000004
#define	AR_EEPROM_STS_WRITE_COMPLETE 0x00000008

#define	AR_EEPROM_REG_DOMAIN	0xbf	/* current regulatory domain */

/*
 * Poll the register looking for a specific value.
 */
static inline HAL_BOOL
ath_hal_wait(struct ath_hal *ah, u_int reg, u_int32_t mask, u_int32_t val)
{
	int i;
	for (i = 0; i < 1000; i++) {
		if ((OS_REG_READ(ah, reg) & mask) == val)
			return AH_TRUE;
		udelay(10);
	}
	printk("%s: timeout on reg 0x%x: 0x%08x & 0x%08x != 0x%08x\n",
		__func__, reg, OS_REG_READ(ah, reg), mask, val);
	return AH_FALSE;
}	//	ath_hal_wait

static inline HAL_BOOL
ar5212EepromRead(struct ath_hal *ah, u_int off, u_int16_t *data)
{
	OS_REG_WRITE(ah, AR_EEPROM_ADDR, off);
	OS_REG_WRITE(ah, AR_EEPROM_CMD, AR_EEPROM_CMD_READ);

	if (!ath_hal_wait(ah, AR_EEPROM_STS,
	    AR_EEPROM_STS_READ_COMPLETE | AR_EEPROM_STS_READ_ERROR,
	    AR_EEPROM_STS_READ_COMPLETE)) {
		printk("%s: read failed for entry 0x%x\n", __func__, off);
		return AH_FALSE;
	}
	*data = OS_REG_READ(ah, AR_EEPROM_DATA) & 0xffff;
	return AH_TRUE;
}	//	ar5212EepromRead
/*
 * Write 16 bits of data from data to the specified EEPROM offset.
 */
static inline HAL_BOOL
ar5212EepromWrite(struct ath_hal *ah, u_int off, u_int16_t data)
{
	OS_REG_WRITE(ah, AR_EEPROM_ADDR, off);
	OS_REG_WRITE(ah, AR_EEPROM_DATA, data);
	OS_REG_WRITE(ah, AR_EEPROM_CMD, AR_EEPROM_CMD_WRITE);

	
	if (!ath_hal_wait(ah, AR_EEPROM_STS,
	    AR_EEPROM_STS_WRITE_COMPLETE | AR_EEPROM_STS_WRITE_ERROR,
	    AR_EEPROM_STS_WRITE_COMPLETE)) {
		printk("%s: write failed for entry 0x%x, data 0x%x\n",
			__func__, off, data);
		return AH_FALSE;
	}
	return AH_TRUE;
}

// setting regulatory domain
#define	ath_hal_setregdomain(ah, v, status) \
	*status = ar5212EepromWrite(ah, AR_EEPROM_REG_DOMAIN, v)
