/*-
 * Copyright (c) 2007 UBT
 * All rights reserved. */

#ifndef EXPORT_SYMTAB
#define EXPORT_SYMTAB
#endif

#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/if_vlan.h>

#include "if_llc.h"
#include "if_ethersubr.h"
#include "if_media.h"
#include <net80211/ieee80211_var.h>
#include <net80211/if_athproto.h>
#include <ath/if_athvar.h>

#error BUILDING ME ???
#define	SPER_DBG_LEVEL1       0x0001 /* print level 1 information*/
#define	SPER_DBG_LEVEL2       0x0002 /* print level 2 information*/
#define	SPER_DBG_LEVEL3       0x0004 /* print level 3 information*/
#define	SPER_DBG_LEVEL4       0x0008 /* print level 4 information*/
#define	SPER_DBG_LEVEL5       0x0010 /* print level 5 information*/
#define	SPER_DBG_LEVEL6       0x0020 /* print level 6 information*/

#define	SPER_NO_SPER          0	/* No SPER features */
#define	SPER_REP_ONLY         1	/* SPER station name and link status reporting only */
#define	SPER_REP_SPER         2	/* REP + SPER features */
#define	SPER_REP_SPER_POLL    3	/* REP + SPER + POLLING features */

static void sper_txFr_timeout(unsigned long arg);
static void sper_rxAck_timeout(unsigned long arg);
static void sper_start_txFr_timer(void);
static void sper_start_rxAck_timer(struct ether_header *peh);
static void sper_get_plm( sper_phdr phdr);
static void sper_handle_plm( struct sper_plm_info *pinfo);
static void sper_ack_eth_hdr(struct net_device *dev, struct ether_header *peh);
static int send_frame(sper_frame_kind fk, u_int16_t arg);

static int ath_hal_sperdbg = 0;//(SPER_DBG_LEVEL1 | SPER_DBG_LEVEL2 | SPER_DBG_LEVEL3 | SPER_DBG_LEVEL4 | SPER_DBG_LEVEL5 | SPER_DBG_LEVEL6);

extern int ieee80211_classify(struct ieee80211_node *ni, struct sk_buff *skb);


/* Tx /Rx frame numbers */
sper_seq_nr out_ack_expected;       /* lower edge of sender's window */
sper_seq_nr out_next_tx_frame; /* upper edge of sender's window + 1 */
sper_seq_nr in_frame_expected;     /* lower edge of receiver's window */
sper_seq_nr in_max_rx_frame;            /* upper edge of receiver's window + 1 */

/* out/in packet buffers and their status. */
sper_buf sper_out_buf[SPER_TXBUF];   /* buffers for the outbound stream */
sper_buf sper_in_buf[SPER_RXBUF];    /* buffers for the inbound stream */
sper_buf sper_ack_buf; /* used to eliminate multiple ACK packets in POLL queue. */

/* timers to wait for ack or retransmission. */
sper_timer sper_ack_timer; /* ack timer */
sper_timer sper_retx_timer; /* retransmission timer */

/* Misc info.*/
sper_boolean sper_first_packet;
sper_boolean sper_no_packet_rcv;
int sper_ack_disabled = 0;
int sper_plm_rstamp = 0;
struct sper_plm_info sper_rx_plm_info;  /* information for Rx PLM handling. */
struct sper_plm_info sper_retx_info;  /* information for Retx a single packet. */

static sper_boolean sper_between(sper_seq_nr a, sper_seq_nr b, sper_seq_nr c);

static void sper_print_pkt(u_int8_t* buf, int len, int max)
{
    int i;
    if (len > 0) {
	if (len > max)
	    len = max;
	for (i = 0; i < len; i++)
	{
	    if ((i%0x10 == 0)&&(i!=0)) printk("\n");
	    printk("%02x:", buf[i]);
	}
	printk("\n");
    }
}

/* Swap the two bytes within a unsigned short. */
static inline u_int16_t sper_bswap16(u_int16_t _x)
{
#ifdef __BIG_ENDIAN
	/* Note, that m-tik is using litle endian order in packets */
	return ((((_x) >> 8) & 0xff) | (((_x) & 0xff) << 8));

#endif
    return _x;
}


static sper_boolean sper_between(sper_seq_nr a, sper_seq_nr b, sper_seq_nr c)
{
    return ((a <= b) && (b < c)) || ((c < a) && (a <= b)) || ((b < c) && (c < a));
}

static inline u_int32_t sper_bswap32(u_int32_t _x)
{
#ifdef __BIG_ENDIAN
	/* Note, that m-tik is using litle endian order in packets */
	return ((((_x) & 0xff000000) >> 24) | (((_x) & 0x00ff0000) >>  8) |
	  (((_x) & 0x0000ff00) <<  8) | (((_x) & 0x000000ff) << 24));

#endif
    return _x;
}

int sper_do_ack(int enable, struct net_device *dev)
{
    struct ath_softc *sc = dev->priv;
    struct ath_hal *ah = sc->sc_ah;
    u_int32_t off;
    off = 0x8048;

    if (ath_hal_sperdbg & SPER_DBG_LEVEL1)
	printk("%s ACK: register 0x%x\n", enable?"Enable":"Disable", off);
    if (enable) {
	if (sper_ack_disabled) {
	    sper_ack_disabled = 0;
	    OS_REG_WRITE(ah, off, OS_REG_READ(ah, off) & 0xFFFFFFFD);
	}
    } else {
	if (!sper_ack_disabled) {
	    sper_ack_disabled = 1;
	    OS_REG_WRITE(ah, off, OS_REG_READ(ah, off) | 0x2);
	}
    }
    return 0;
}

static int sper_hardstart_chk(struct sk_buff *skb, struct net_device *dev)
{
    struct ieee80211vap *vap = dev->priv;
    struct ieee80211com *ic = vap->iv_ic;
    struct net_device *parent = ic->ic_dev;
    struct ieee80211_node *ni = NULL;
    struct ieee80211_cb *cb;
    struct ether_header *eh;
    int rtn = SPER_WLL_LINK_ERR;


    /* NB: parent must be up and running */
    if ((parent->flags & (IFF_RUNNING|IFF_UP)) != (IFF_RUNNING|IFF_UP))
    {
	goto bad;
    }

    /* No data frames go out unless we're running. */
    if (vap->iv_state != IEEE80211_S_RUN) {
	IEEE80211_DPRINTF(vap, IEEE80211_MSG_OUTPUT,
			  "%s: ignore data packet, state %u\n",
			  __func__, vap->iv_state);
#if 0
	vap->iv_stats.ist_tx_discard++;
#endif
	goto bad;
    }

    /* cancel bg scan */
    if (ic->ic_flags & IEEE80211_F_SCAN)
	ieee80211_cancel_scan(vap);

    /* Find the node for the destination so we can do
     things like power save. */
    eh = (struct ether_header *)skb->data;
    ni = ieee80211_find_txnode(vap, eh->ether_dhost);
    if (ni == NULL){
	/* NB: ieee80211_find_txnode does stat+msg */
	IEEE80211_NOTE(vap, IEEE80211_MSG_OUTPUT, ni,
		       "%s: discard, find txnode failure", __func__);
	goto bad;
    }

    /* Check if the node is associated. */
    if (ni->ni_associd == 0 && ni != vap->iv_bss) {
	/* the node hasn't been associated */
	if (skb != NULL)	dev_kfree_skb(skb);
	IEEE80211_NOTE(vap, IEEE80211_MSG_OUTPUT, ni,
		       "%s: discard, node is not associated", __func__);
	return SPER_WLL_LINK_ERR;
    }

    /* calculate priority so drivers can find the tx queue */
    if (ieee80211_classify(ni, skb)) {
	IEEE80211_NOTE(vap, IEEE80211_MSG_OUTPUT, ni,
		       "%s: discard, classification failure", __func__);
	goto bad;
    }

    cb = (struct ieee80211_cb *) skb->cb;
    cb->ni = ni;
    cb->next = NULL;

    /* power-save checks */
    if (WME_UAPSD_AC_CAN_TRIGGER(skb->priority, ni)) {
	/* U-APSD power save queue */
	/* XXXAPSD: assuming triggerable means deliverable */
	M_FLAG_SET(skb, M_UAPSD);
    }
    else if ((ni->ni_flags & IEEE80211_NODE_PWR_MGT) && !M_PWR_SAV_GET(skb)) {
	/*Station in power save mode; stick the frame on the sta's power save queue
	 and continue. We'll get the frame back when the time is right. */
	ieee80211_pwrsave(ni, skb);
	rtn = SPER_WLL_PWR_SAVE;
	IEEE80211_NOTE(vap, IEEE80211_MSG_OUTPUT, ni,
		       "%s: reclaim, node in power save mode", __func__);
	goto reclaim;
    }
    M_FLAG_KEEP_ONLY(skb, M_UAPSD | M_PWR_SAV);

    vap->iv_devstats.tx_packets++;
    vap->iv_devstats.tx_bytes += skb->len;
    ic->ic_lastdata = jiffies;

    skb->dev = parent;

    IEEE80211_DPRINTF(vap, IEEE80211_MSG_OUTPUT,
		      "%s: SUCCESS\n",
		      __func__);
    return SPER_SUCCESS;

bad:
    if (skb != NULL)
	dev_kfree_skb(skb);
reclaim:
    if (ni != NULL)
	ieee80211_free_node(ni);
    return rtn;
}

int sper_deinit (struct net_device *dev)
{
    /* delete the ACK timer */
    del_timer(&sper_ack_timer.timer);
    sper_ack_timer.status = sper_timer_init;

    /* Initialize the Rx window variables. */
    in_frame_expected = 0;
    in_max_rx_frame = SPER_RXBUF - 1;
    sper_no_packet_rcv = sper_true;

    /* Delete the re-transmission timer. */
    del_timer(&sper_retx_timer.timer);
    sper_retx_timer.status = sper_timer_init;

    /* Initialize the Tx window variables. */
    out_ack_expected = 0;   /* next ack expected on the inbound stream */
    out_next_tx_frame = 0; /* number of next outgoing frame */
    sper_first_packet = sper_true;
    return 0;
}

int sper_init (struct net_device *dev)
{
    u_int32_t i;

    out_ack_expected = 0;   /* next ack expected on the inbound stream */
    out_next_tx_frame = 0; /* number of next outgoing frame */
    in_frame_expected = 0;
    in_max_rx_frame = SPER_RXBUF -1;
    sper_first_packet = sper_true;
    sper_no_packet_rcv = sper_true;
    sper_rx_plm_info.time = 0;
    sper_rx_plm_info.handle = 0;

    /* initialization Tx buffers and outbound ACKed bit map. */
    for ( i = 0; i < SPER_TXBUF; i++)
    {
	sper_out_buf[i].skb = NULL;
	sper_out_buf[i].sst = SPER_NO_DATA;
	sper_out_buf[i].pai = sper_true;
	sper_out_buf[i].cnt = 0;
	sper_out_buf[i].time = 0;
    }

    /* initialization Rx buffers and inbound ACKed bit map. */
    for (i=0; i<SPER_RXBUF; i++)
    {
	sper_in_buf[i].skb = NULL;
	sper_in_buf[i].pai =sper_false;
	sper_in_buf[i].cnt = 0;
	sper_in_buf[i].time = 0;
    }

    sper_ack_buf.skb = NULL;
    sper_ack_buf.sst = SPER_NO_DATA;

    /* Initialize the retransmit and ack timers. */
    sper_ack_timer.status = sper_timer_init;
    sper_ack_timer.timer.function = sper_rxAck_timeout;
    //sper_ack_timer.timer.data = (u_int32_t)dev;
    init_timer(&sper_ack_timer.timer);
    sper_retx_info.time = 0;
    sper_retx_info.handle = 0;
    sper_retx_timer.status = sper_timer_init;
    sper_retx_timer.timer.function = sper_txFr_timeout;
    //sper_retx_timer.timer.data = (u_int32_t)dev;
    init_timer(&sper_retx_timer.timer);

    return 0;
}

/* Send the data frame again, after send, arm the timer again. */
static void sper_txFr_timeout(unsigned long arg)
{
	if (out_ack_expected != out_next_tx_frame)
	{
	    u_int16_t j;
	    j = out_ack_expected % SPER_MAX_BUF;

	    /* We can retransmit just the out_ack_expected, or all the
	     missing packets based on out_arrived[]. */
	    //      sper_out_buf[j].time = 0;
	    sper_out_buf[j].cnt += 1;
	    if (send_frame(sper_retry, j) == SPER_SUCCESS && sper_out_buf[j].cnt < SPER_RETRY_MAX) {
		if (ath_hal_sperdbg & SPER_DBG_LEVEL3)
		    printk("%s: mod_timer: out_ack_expected: %d, out_next_tx_frame: %d\n", __func__, out_ack_expected, out_next_tx_frame);
		/* Arm the timer for the next period. */
		mod_timer(&sper_retx_timer.timer, jiffies + SPER_TXRETRY_TIME);
	    }
	}
	/* No more packet waiting for ACK, don't re-arm the timer. */
	else
	{
	    if (ath_hal_sperdbg & SPER_DBG_LEVEL3)
		printk("%s: del_timer2: out_ack_expected: %d, out_next_tx_frame: %d\n", __func__, out_ack_expected, out_next_tx_frame);
	    del_timer(&sper_retx_timer.timer);
	    sper_retx_timer.status = sper_timer_init;
	}
}

/* Send a ACK packet, and delete the ACK timer. */
static void sper_rxAck_timeout(unsigned long arg)
{
	u_int16_t bn, fn;
	/* Get the received frame number. */
	if (in_frame_expected == 0)
	    fn = SPER_MAX_SEQ1;
	else
	    fn = in_frame_expected -1;
	bn = fn % SPER_MAX_BUF;
	if (send_frame(sper_ack, bn) == SPER_SUCCESS) {
	    del_timer(&sper_ack_timer.timer);
	    sper_ack_timer.status = sper_timer_init;
	}
}

static void  sper_start_txFr_timer(void)
{
    if (sper_retx_timer.status == sper_timer_running)
	return;
    else if (sper_retx_timer.status == sper_timer_added)
    {
	mod_timer(&sper_retx_timer.timer, jiffies + SPER_TXRETRY_TIME);
	sper_retx_timer.status =sper_timer_running;
    }
    else /*sper_retx_timer.status == sper_timer_init */
    {
	sper_retx_timer.timer.expires = jiffies + SPER_TXRETRY_TIME;
	add_timer(&sper_retx_timer.timer);
	sper_retx_timer.status =sper_timer_running;
    }
}

static void  sper_start_rxAck_timer(struct ether_header *peh)
{
    if (sper_ack_timer.status == sper_timer_running)
	return;
    else if (sper_ack_timer.status == sper_timer_added)
    {
	mod_timer(&sper_ack_timer.timer, jiffies + SPER_ACK_TIME_OUT);
	sper_ack_timer.status = sper_timer_running;

	/* Save the Ethernet header for possible ACK packet*/
	memcpy(&(sper_ack_timer.eh), peh, sizeof(struct ether_header));
    }
    else /* sper_ack_timer.status == sper_timer_init */
    {
	sper_ack_timer.timer.expires = jiffies + SPER_ACK_TIME_OUT;
	add_timer(&sper_ack_timer.timer);
	sper_ack_timer.status = sper_timer_running;

	/* Save the Ethernet header for possible ACK packet*/
	memcpy(&(sper_ack_timer.eh), peh, sizeof(struct ether_header));
    }
}

/* Packet Loss Map (PLM) is used to tell the sender which packets are
 received, and which are lost. The PLM starts from in_frame_expected.
 The meaning of each bit is as the following.
 0 -- the corresponding packet is received.
 1 -- the corresponding packet is lost.*/

/* sper_get_plm() travels through the Rx buffers to construct
 the packet loss map for receiving side. */
static void sper_get_plm( sper_phdr phdr)
{
    u_int16_t i, j;
    u_int32_t bit, plm;
    sper_boolean send_plm = sper_false;

    /* If no packet has received yet, don't send any
     plm and rsn information. */
    if(sper_no_packet_rcv == sper_true)
    {
	phdr->plm = 0x0;
	phdr->rsn = 0x0;
	phdr->cmd &= ~(SPER_CMD_PLM_BIT|SPER_CMD_ACK_BIT);
	return;
    }
    else /* Otherwise, always carry rsn. */
    {
	phdr->cmd |= SPER_CMD_ACK_BIT;

	/* received frame number = in_frame_expected -1 */
	if (in_frame_expected == 0)
	    phdr->rsn = SPER_MAX_SEQ1;
	else
	    phdr->rsn = in_frame_expected -1;

	/* No need for a separate ACK packet. */
	if (sper_ack_timer.status == sper_timer_running)
	{
	    del_timer(&sper_ack_timer.timer);
	    sper_ack_timer.status = sper_timer_init;
	}
    }

    /* Obtain the packet loss map */
    plm = 0xFFFFFFFF;
    bit = 1;
    j = in_frame_expected % SPER_MAX_BUF;
    for (i=0; i<SPER_MAX_BUF; i++)
    {
	if (sper_in_buf[j].pai == sper_true)
	{
	    plm = plm & (~bit);
	    send_plm = sper_true;
	}
	bit = bit << 1;
	j = (j+1)%SPER_MAX_BUF;
    }
    plm = sper_bswap32(plm);

    /* Set plm and corresponding command bit in the header accordingly. */
    if(send_plm == sper_true)
    {
	phdr->plm = plm;
	phdr->cmd |= SPER_CMD_PLM_BIT;
    }
    else
    {
	phdr->plm = 0x0;
	phdr->cmd &= ~(SPER_CMD_PLM_BIT);
    }
}

/* To overcome too many retry packets, we only process the PLM at
 the end of each Rx cycle.*/
void sper_rx_loop_end( void)
{
    /* Do re-transmission. */
    if(sper_rx_plm_info.handle)
	sper_handle_plm(&sper_rx_plm_info);
    /* Clear the requests. */
    sper_retx_info.handle = 0;
    sper_rx_plm_info.handle = 0;
}
EXPORT_SYMBOL(sper_rx_loop_end);

/* sper_handle_plm() do packet retransmission based on the plm
 sent from the receiver.*/
static void sper_handle_plm( struct sper_plm_info *pinfo)
{
    u_int16_t i,j,k;
    u_int32_t plm, gap, bit;
    struct sper_header *pfr = &(pinfo->sper);

    /* do nothing if all packets sent are received.*/
    if (out_ack_expected == out_next_tx_frame)
	return;

    /* do nothing if the plm is old. */
    j = (pfr->rsn +1)&(0xFFFF);
    if (j != out_ack_expected)
	return;

    /* retransmit the loss packets based on the PLM. We only transmit
     the packets in the gaps.*/
    bit = 1;
    plm = sper_bswap32(pfr->plm);
    gap = plm >> 1;
    j = j % SPER_MAX_BUF;
    k = out_next_tx_frame % SPER_MAX_BUF;
    if (j<k)
    {
	for (i=j; i<k; i++)
	{
	    if (plm & bit)
	    {
		if (sper_out_buf[i].cnt == 0)
		{
		    sper_out_buf[i].cnt += 1;
		    send_frame(sper_retry, i);
		}
		else if (sper_out_buf[i].time<pinfo->time)
		{
		    sper_out_buf[i].cnt += 1;
		    send_frame(sper_retry, i);
		}
	    }
	    /* Don't want to Re_Tx the packets on their way. */
	    if ( (gap & 0xff ) == 0xff) break;
	    gap = gap >> 1;
	    bit = bit << 1;
	}
    }
    else
    {
	for (i=j; i<SPER_MAX_BUF; i++)
	{
	    if (plm & bit)
	    {
		if (sper_out_buf[i].cnt == 0)
		{
		    sper_out_buf[i].cnt += 1;
		    send_frame(sper_retry, i);
		}
		else if (sper_out_buf[i].time<pinfo->time)
		{
		    sper_out_buf[i].cnt += 1;
		    send_frame(sper_retry, i);
		}
	    }
	    /* Don't want to Re_Tx the packets on their way. */
	    if ( (gap & 0xff ) == 0xff) break;
	    gap = gap >> 1;
	    bit = bit << 1;
	}
	if( (gap&0xff)!=0xff )
	{
	    for (i=0; i<k; i++)
	    {
		if (plm & bit)
		{
		    if (sper_out_buf[i].cnt == 0)
		    {
			sper_out_buf[i].cnt += 1;
			send_frame(sper_retry, i);
		    }
		    else if (sper_out_buf[i].time<pinfo->time)
		    {
			sper_out_buf[i].cnt += 1;
			send_frame(sper_retry, i);
		    }
		}
		/* Don't want to Re_Tx the packets on their way. */
		if ( (gap & 0xff ) == 0xff) break;
		gap = gap >> 1;
		bit = bit << 1;
	    }
	}
    }

    /* Arm the re-Tx time again. */
    sper_start_txFr_timer();
}

/* Prepare the ethernet header for ACK packets. */
static void sper_ack_eth_hdr(struct net_device *dev, struct ether_header *peh)
{
    struct ieee80211vap *vap=dev->priv;
    struct ieee80211_node *ni=vap->iv_bss;
    unsigned char brcst[]={0xff, 0xff, 0xff, 0xff, 0xff, 0xff};

    IEEE80211_ADDR_COPY(peh->ether_shost, sper_ack_timer.eh.ether_dhost);
    IEEE80211_ADDR_COPY(peh->ether_dhost, sper_ack_timer.eh.ether_shost);
    peh->ether_type = htons(ETHERTYPE_ACK);
    if (IEEE80211_ADDR_EQ(peh->ether_shost, brcst))
    {
	IEEE80211_ADDR_COPY(peh->ether_shost, vap->iv_myaddr);
	if (ath_hal_sperdbg & SPER_DBG_LEVEL2)
	    printk("A broadcast MAC address replaced with. \n");
    }
    IEEE80211_ADDR_COPY(peh->ether_shost, vap->iv_myaddr);
    IEEE80211_ADDR_COPY(peh->ether_dhost, ni->ni_macaddr);
}

static int sper_hardstart_xmit(struct sk_buff *skb, struct net_device *dev, void *pbuf)
{
    struct ieee80211vap *vap = dev->priv;
    struct ieee80211_cb *cb;
    struct ieee80211_node *ni;
    struct ether_header *eh;


    cb = (struct ieee80211_cb *) skb->cb;
    ni = cb->ni;
    eh = (struct ether_header *)skb->data;

#ifdef ATH_SUPERG_XR
    /* broadcast/multicast packets need to be sent on XR vap in addition to
     normal vap. */
    if(vap->iv_xrvap && ni == vap->iv_bss && vap->iv_xrvap->iv_sta_assoc) {
	struct sk_buff *skb1;
	ni = ieee80211_find_txnode(vap->iv_xrvap, eh->ether_dhost);
	skb1 = skb_copy(skb,GFP_ATOMIC);
	if(skb1) {
	    cb = (struct ieee80211_cb *) skb1->cb;
	    cb->ni = ni;
	    cb->flags = 0;
	    cb->next = NULL;
	    (void) dev_queue_xmit(skb1);
	}
    }
#endif
    IEEE80211_DPRINTF(vap, IEEE80211_MSG_OUTPUT,
		      "%s: xmit data packet\n", __func__);

    (void) dev_queue_xmit(skb);

    return SPER_SUCCESS;
}


/* Construct and send a data, ack, or nak frame. */
static int send_frame(sper_frame_kind fk, u_int16_t bn)
{
    struct ether_header eh, *peh;
    struct sk_buff *skb;
    sper_phdr fr;  /* ptr to nstream header in the pdu */
    struct net_device *dev;
    sper_buf *pbuf=NULL;

    if (fk == sper_data)
    {
	pbuf = &(sper_out_buf[bn]);
	skb = pbuf->skb;
	dev = pbuf->dev;
	if (skb == NULL || dev == NULL)
	{
	    printk("Tx: cannot get skb(%p) or dev(%p) for sper data\n", skb, dev);
	    return SPER_SKB_ERR;
	}

	if (ath_hal_sperdbg & SPER_DBG_LEVEL6)
	{
	    printk("Tx: Data Packet before Nstream, len=0x%x. \n", skb->len);
	    sper_print_pkt(skb->data, 80, skb->len);
	}
	/* Save the original ethernet header */
	memcpy(&eh, skb->data, sizeof(struct ether_header));
	skb_pull(skb, sizeof(struct ether_header));

	/* unshare skb buffer */
	skb = skb_unshare(skb, GFP_ATOMIC);
	if (skb == NULL) {
	    printk("Tx: cannot unshare skb for nstreme\n");
	    return SPER_SKB_ERR;
	}

	/* check if there is enough headroom. For now, we just care about
	 the Ethernet and SPER headers. The rest will be handled in wlan
	 driver. */
	if (skb_headroom(skb) < sizeof(struct ether_header)+SPER_SPER_HDR)
	{
	    struct sk_buff *tmp = skb;
	    skb = skb_realloc_headroom(skb, sizeof(struct ether_header)+SPER_SPER_HDR);
	    dev_kfree_skb(tmp);
	    if (skb == NULL)
	    {
		printk("Tx: cannot expand storage for nstreme\n");
		return SPER_SKB_ERR;
	    }
	}

	/* Add nstream header */
	fr = (sper_phdr) skb_push(skb, SPER_SPER_HDR);
	fr->ssn = pbuf->ssn;
	fr->cmd = SPER_CMD_DATA_BIT;
	if (sper_first_packet == sper_true)
	{
	    fr->cmd |= SPER_CMD_INIT_BIT;
	    sper_first_packet = sper_false;
	}

	/* restore ethernet header */
	peh = (struct ether_header *) skb_push(skb, sizeof(struct ether_header));
	memcpy(peh, &eh, sizeof(struct ether_header));

	/* Clone the skb for possible re-transmission. */
	pbuf->skb = skb_clone(skb, GFP_ATOMIC);

	/* start the timer to wait for the ACK */
	if (ath_hal_sperdbg & SPER_DBG_LEVEL2) {
	    printk("%s: sper_start_txFr_timer: skb[%d]: %p, skb: %p\n",
		   __func__, bn, pbuf->skb, skb);
	}
	sper_start_txFr_timer();
	pbuf->cnt = 0;
    }
    /* Re-transmit a packet */
    else if (fk == sper_retry)
    {
	pbuf = &(sper_out_buf[bn]);
	skb = pbuf->skb;
	dev = pbuf->dev;
	if (skb == NULL || dev == NULL)
	{
	    printk("Tx: cannot get skb(%p) or dev(%p) for sper retry\n", skb, dev);
	    return SPER_SKB_ERR;
	}

	if (ath_hal_sperdbg & SPER_DBG_LEVEL1)
	{
	    printk("Retry: Data Packet before Nstream. \n");
	    sper_print_pkt(skb->data, 80, skb->len);
	}

	fr = (sper_phdr) (skb->data + sizeof(struct ether_header));

	/* Clone the skb for possible re-transmission. */
	pbuf->skb = skb_clone(skb, GFP_ATOMIC);
    }
    /* Send a ACK only frame */
    else
    {
	u_int16_t fl;  /* frame length == 802.3 "length" value */

	dev = sper_ack_timer.dev;
	//  fl = SPER_802_11_HDR + SPER_LLC_HDR + SPER_SPER_HDR + SPER_CHECK_SUM;
	/* allocate a longer buffer */
	fl = 128;
	skb = dev_alloc_skb(fl);
	if (skb != NULL)
	{
	    /* reserve the headroom for wlan headers. */
	    skb_reserve(skb, 64);
	    skb->dev = dev;

	    /* Prepare the ehternet header. */
	    peh = (struct ether_header *) (skb->data);
	    sper_ack_eth_hdr(dev, peh);
	    skb->mac.raw = skb->data;
	    skb_put(skb, sizeof(struct ether_header));
	    skb->nh.raw = skb->data + sizeof(struct ether_header);
	    skb->protocol = __constant_htons(ETH_P_802_2);

	    /* Fill the Nstream Header */
	    skb_put(skb, SPER_SPER_HDR);
	    fr = (sper_phdr)(skb->data + sizeof(struct ether_header));
	    memset(fr, 0, SPER_SPER_HDR);
	    fr->cmd = SPER_CMD_ACK_ONLY;
	    if (sper_first_packet == sper_true)
	    {
		fr->cmd |= SPER_CMD_INIT_BIT;
		sper_first_packet = sper_false;
	    }
	}
	else
	{
	    printk("Tx: cannot allocate ack skb for nstreme\n");
	    return SPER_SKB_ERR;
	}
	if (ath_hal_sperdbg & SPER_DBG_LEVEL2)
	{
	    printk("Tx ACK Packet. \n");
	    sper_print_pkt(skb->data, 80, skb->len);
	}
	if (ath_hal_sperdbg & SPER_DBG_LEVEL1)
	    printk ("Tx: Sending ACK to frame dev=0x%x \n", (u_int32_t)dev);

	/* Let's get the ni and AC for this packet.*/
	if (sper_hardstart_chk(skb, dev) != SPER_SUCCESS) {
	    printk("Tx ACK Packet failed.\n");
            dev_kfree_skb(skb);
	    return SPER_WLL_LINK_ERR;
	}
    }

    /* Construct the rest of Nstreame header */
    fr->dsap = SPER_DATA_SAP;
    fr->ssap = SPER_DATA_SAP;

    /* fill rsn and plm fields */
    sper_get_plm(fr);

    /* Swap the ssn and rsn to conform with network byte order. */
    if (fk != sper_retry)
	fr->ssn = sper_bswap16(fr->ssn);
    else
    { /* Raise one priority level for retry packets. */
	skb->priority = skb->priority + 1;
	if (skb->priority > WME_AC_VO)
	    skb->priority = WME_AC_VO;
    }

    fr->rsn = sper_bswap16(fr->rsn);
    if (ath_hal_sperdbg & SPER_DBG_LEVEL3)
    {
	printk(" Tx: Packet after Nstream, len=0x%x. \n", skb->len);
	sper_print_pkt(skb->data, 80, skb->len);
    }

    /* transmit the frame */
    sper_hardstart_xmit(skb, dev, (void *)pbuf);

    /* waiting for the frame to arrive at receiver. */
    if (fk != sper_ack)
	pbuf->pai = sper_false;

    return SPER_SUCCESS;
}

int sper_hardstart(struct sk_buff *skb, struct net_device *dev)
{
    int rtn;
    //struct ieee80211vap *vap = dev->priv;
    struct ether_header *peh;

    /* Check if we can transmit the frame or not. */
    if ((rtn = sper_hardstart_chk(skb, dev)) != SPER_SUCCESS)
	return SPER_SUCCESS;

    peh = (struct ether_header *)skb->data;

    switch (ntohs(peh->ether_type))
    {
    case ETHERTYPE_ACK:
    case ETHERTYPE_ACK1:
	return sper_hardstart_xmit(skb, dev, NULL);
    }

    if (ath_hal_sperdbg & SPER_DBG_LEVEL1)
	printk ("Tx window: ack_exp=%d, next_fr=%d \n",
		out_ack_expected, out_next_tx_frame);

    /* check if the sending window is open. If it's open,
     accept, save, and transmit the new frame*/
    if (sper_between(out_ack_expected, out_next_tx_frame,
		     (out_ack_expected+SPER_MAX_BUF1)%(SPER_MAX_SEQ)))
    {
	u_int16_t bn = out_next_tx_frame%SPER_MAX_BUF; /* buffer number */

	/* fill the sper descriptor */
	sper_out_buf[bn].skb = skb;
	sper_out_buf[bn].ssn = out_next_tx_frame;
	sper_out_buf[bn].dev = dev;
	//    sper_out_buf[bn].time = 0;

	/* transmit the frame */
	if (ath_hal_sperdbg & SPER_DBG_LEVEL1) {
	    printk("Sending a data frame before polling_Q. \n");
	}
	if (send_frame(sper_data, bn) == SPER_SUCCESS)
	    sper_inc(out_next_tx_frame); /* advance upper window edge */
    }

    /* Otherwise, simply release the new frame*/
    else
    {
	printk("%s: ignore data packet, Tx window full: ack_exp=%d, next_fr %u\n",
			  __func__, out_ack_expected, out_next_tx_frame);
	if (skb != NULL)
	{
	    /* should we return an error code to tell the upper layer
	     that the buffers are full? */
	    dev_kfree_skb(skb);
	    return 0;
	}
    }
    return 0;
}

/* Since TSF is only avaible in HAL layer, a callback function is used
 to update the Tx time stamp when the packet goes to the HAL layer. */
int sper_update_tstamp(struct sk_buff *skb, u_int32_t tsf)
{
    struct sper_header *pfr;  /* ptr to nstream header in the pdu */
    u_int16_t bn;

    /* Retrieve SPER header. */
    pfr = (sper_phdr) (skb->data + sizeof(struct ether_header));

    /* Update Tx time stamp for SPER packets only. */
    if ((pfr->dsap==0x11) && (pfr->ssap==0x11)
	&& ((pfr->cmd & SPER_CMD_DATA_BIT)==SPER_CMD_DATA_BIT))
    {
	bn = (sper_bswap16(pfr->ssn))%SPER_TXBUF;
	sper_out_buf[bn].time = tsf + SPER_TURN_AROUND;
    }

    return 0;
}
EXPORT_SYMBOL(sper_update_tstamp);

/* Deliver the data packet to the higher layer. */
int sper_deliver_data(struct ieee80211_node *ni, struct sk_buff *skb)
{

    if (ath_hal_sperdbg & SPER_DBG_LEVEL1) {
	struct ether_header *peh;

	peh = (struct ether_header *) (skb->data);

	switch (ntohs(peh->ether_type)) {
	case ETHERTYPE_INIT:
	    printf("Found an init pkt. \n");
	    break;
	case ETHERTYPE_CHAT:
	    printf("Found an chat pkt. \n");
	    break;
	case ETHERTYPE_WAKE:
	    printf("Found an wake pkt. \n");
	    break;
	}
    }
#ifdef USE_HEADERLEN_RESV
    skb->protocol = ath_eth_type_trans(skb, skb->dev);
#else
    skb->protocol = eth_type_trans(skb, skb->dev);
#endif
    netif_receive_skb(skb);

    return SPER_SUCCESS;
}

/* a data or control frame has arrived. */
int sper_frame_arrival(struct ieee80211_node *ni, struct sk_buff *skb)
{
    struct ether_header eh, *peh;
    struct sper_header fr, *pfr;  /* ptr to nstream header in the pdu */
    sper_buf *pbuf;
    int bn;
    struct ieee80211vap *vap = skb->dev->priv;
    struct ieee80211com *ic = vap->iv_ic;

    /* Pass the non-Nstream frames directly to upper layer. */
    peh = (struct ether_header *) (skb->data);
    pfr = (sper_phdr) (skb->data + sizeof(struct ether_header));
    if ((ic->ic_flags_ext & IEEE80211_FEXT_SPER) &&
	((pfr->dsap==0x11) && (pfr->ssap==0x11)))
    {
	if (ath_hal_sperdbg & SPER_DBG_LEVEL3)
	{
	    printk("Rx: Nstream Packet, cmd=0x%x. \n", pfr->cmd);
	    sper_print_pkt(skb->data, 40, skb->len);
	}
	/* Pull the nstream header out of the incoming packet*/
	memcpy(&eh, skb->data, sizeof(struct ether_header));
	pfr = (sper_phdr) skb_pull(skb, sizeof(struct ether_header));
	if (pfr == NULL)
	{
	    dev_kfree_skb(skb);
	    return SPER_SKB_ERR;
	}
	memcpy(&fr, skb->data, SPER_SPER_HDR);
	pfr = &fr;
	pfr->ssn = sper_bswap16(pfr->ssn);
	pfr->rsn = sper_bswap16(pfr->rsn);
	skb_pull(skb, SPER_SPER_HDR);

	/* Place Ethernet header back. */
	peh = (struct ether_header *) skb_push(skb, sizeof(struct ether_header));
	memcpy(peh, &eh, sizeof(struct ether_header));
	if (ath_hal_sperdbg & SPER_DBG_LEVEL6)
	{
	    printk("Rx: Packet without Nstream header ssn=0x%x, rsn=0x%x. \n", pfr->ssn, pfr->rsn);
	    sper_print_pkt(skb->data, 40, skb->len);
	}

	/* handle piggybacked ack */
	if (pfr->cmd & SPER_CMD_ACK_BIT)
	{
	    if (ath_hal_sperdbg & SPER_DBG_LEVEL2)
		printk ("Rx: Handle ACK info \n");
	    while (sper_between(out_ack_expected, pfr->rsn, out_next_tx_frame))
	    {
		sper_seq_nr num;
		num = out_ack_expected % SPER_MAX_BUF;
		pbuf = &sper_out_buf[num];
		/* frame arrived intact, release the buffer and un-arm the re-transmit timer. */
		if(pbuf->skb != NULL)
		{
		    dev_kfree_skb(pbuf->skb);
		    pbuf->skb = NULL;
		}
		pbuf->pai = sper_true;

		/* advance lower edge of senders window */
		sper_inc(out_ack_expected);
	    }
	    //      if (out_ack_expected != pfr->rsn)
	    //       printk("Protocol error: ack_exp!=rsn, ack_exp=0x%x, rsn=0x%x. \n", out_ack_expected, pfr->rsn);
	    /* Stop the re-transmission timer if all the packets are
	     received by the other side. */
	    if (out_ack_expected == out_next_tx_frame)
	    {
		del_timer(&sper_retx_timer.timer);
		sper_retx_timer.status = sper_timer_init;
	    }
	}

	/* Save SPER information from the Rx packet for single packet retry. */
	sper_retx_info.time = ni->ni_rstamp;
	memcpy(&(sper_retx_info.sper), pfr, sizeof(struct sper_header));
	sper_retx_info.handle = 1;

	/* handle plm information. */
	if (pfr->cmd & SPER_CMD_PLM_BIT)
	{
	    if (ath_hal_sperdbg & SPER_DBG_LEVEL2)
		printk ("Rx: Handle PLM info \n");
	    sper_rx_plm_info.handle = 1;
	    sper_rx_plm_info.time = ni->ni_rstamp;
	    memcpy(&(sper_rx_plm_info.sper), pfr, sizeof(struct sper_header));
	    //      sper_handle_plm(&sper_rx_plm_info);
	}

	/* Handle received data frame. */
	if (pfr->cmd & SPER_CMD_DATA_BIT)
	{
	    if (ath_hal_sperdbg & SPER_DBG_LEVEL2)
		printk ("Rx: Handle data packet \n");

	    /* Start the acknowledgement timer for any frame with data. */
	    //      printk("dev saved in frame_arrival, dev=0x%x \n", (u_int32_t)skb->dev);
	    sper_ack_timer.dev = skb->dev;
	    sper_start_rxAck_timer((struct ether_header *) skb->data);

	    /* Check if the frame is within the expected window, and isn't
	     repeated. */
	    bn= pfr->ssn % SPER_MAX_BUF;
	    pbuf = &sper_in_buf[bn];
	    if (sper_between(in_frame_expected, pfr->ssn, in_max_rx_frame)
		&& (pbuf->pai == sper_false))
	    {
		/* mark the packet as arrived. */
		pbuf->pai = sper_true;

		/* insert data */
		pbuf->ssn = pfr->ssn;
		pbuf->skb = skb;
		pbuf->dev = skb->dev;

		/* Pass frames and advance window. */
		bn = in_frame_expected % SPER_MAX_BUF;
		while (sper_in_buf[bn].pai == sper_true)
		{
		    pbuf = &sper_in_buf[bn];
		    skb = pbuf->skb;
		    /* Remove the packet from our queue, and hand it over
		     to the upper layer. */
		    sper_deliver_data(ni, skb);
		    if (ath_hal_sperdbg & SPER_DBG_LEVEL1)
			printk ("Rx: Packet %d is delivered to upper layer \n", pbuf->ssn);

		    /* post delivery handling */
		    pbuf->pai = sper_false;
		    pbuf->skb = NULL;
		    sper_no_packet_rcv = sper_false;
		    /* advance lower edge of receiver's window */
		    sper_inc(in_frame_expected);
		    /* advance upper edge of receiver's window */
		    sper_inc(in_max_rx_frame);
		    bn = in_frame_expected % SPER_MAX_BUF;
		}
	    }
	    else /* Release the buffer for the repeated packet. */
	    {
		dev_kfree_skb(skb);
		if (ath_hal_sperdbg & SPER_DBG_LEVEL1)
		    printk ("Rx: Repeated pkt to frame %d \n", pfr->ssn);
	    }
	}
	else /* Release the buffer for ACK only packet. */
	{
	    dev_kfree_skb(skb);
	    if (ath_hal_sperdbg & SPER_DBG_LEVEL1)
		printk ("Rx: ACK_only pkt to frame %d \n", pfr->rsn);
	}

	if (ath_hal_sperdbg & SPER_DBG_LEVEL1)
	    printk ("Rx window: fr_exp=%d, fr_max=%d \n", in_frame_expected, in_max_rx_frame);

    }
    else
    {
	if (ath_hal_sperdbg & SPER_DBG_LEVEL1)
	{
	    printk("Rx: Non-Nstream Packet. \n");
	}
	sper_deliver_data(ni, skb);
    }

    return SPER_SUCCESS;
}

