/**
 * Continuous transmit.
 * The code is taken from madwifi-dfs branch,
 * and tried to make it compatible with the callbacs to tx99 in madwifi driver
 **/

#include "ath_tx99.h"

#ifndef AUTOCONF_INCLUDED
#include <linux/config.h>
#endif
#include <linux/version.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/delay.h>

#include "if_media.h"
#include "if_llc.h"
#include "net80211/if_athproto.h"
#include <net80211/ieee80211_var.h>
#include "if_athvar.h"

#include "ah_devid.h"
#include "ah_desc.h"

#ifdef ATH_PCI          /* PCI BUS */
#include "if_ath_pci.h"
#error "PCI"
#endif                  /* PCI BUS */
#ifdef ATH_AHB          /* AHB BUS */
#include "if_ath_ahb.h"
#endif                  /* AHB BUS */

#define	ATH_DEBUG_TXCONT 	0x10000000
#define	DPRINTF(sc, _m, _fmt, ...) do {		\
	if (sc->sc_debug & (_m))				\
		printk(_fmt, __VA_ARGS__);			\
} while (0)

static	char *mod_info = "ath_txcont";
static char *device = "wifi0";		/* ath device to use for testing */
static struct net_device *txcont_dev = NULL;


static int  txcont_on(struct ath_txcont* txcont);
static void txcont_off(struct ath_txcont *txcont);
static void txcont_detach(struct ath_txcont *txcont);


static void
txcont_attach(struct net_device *dev)
{
	struct ath_softc *sc = dev->priv;
	struct ath_txcont *txcont;

	printk("%s: device %s\n", __func__, dev->name);

	txcont = kmalloc(sizeof(struct ath_txcont), GFP_KERNEL);
	if (txcont == NULL) {
		printk(KERN_INFO "%s: no memory for txcont state\n", __func__);
		return;
	}
	memset(txcont, 0, sizeof(*txcont));
	txcont->stop = txcont_off;
	txcont->start = txcont_on;
	txcont->detach = txcont_detach;
	txcont->dev = dev;
	sc->sc_tx99 = txcont;
}


static void
txcont_detach(struct ath_txcont *txcont)
{
	struct net_device *dev = txcont->dev;
	struct ath_softc *sc = dev->priv;

 	printk("%s: device %s\n", __func__, dev->name);

	sc->sc_tx99 = NULL;
	txcont_off(txcont);
	kfree(txcont);
	dev_put(dev);		/* release ref */
	txcont_dev = NULL;
}


static int
ar_device(int devid)
{
	switch (devid) {
	case AR5210_DEFAULT:
	case AR5210_PROD:
	case AR5210_AP:
		return 5210;
	case AR5211_DEFAULT:
	case AR5311_DEVID:
	case AR5211_LEGACY:
	case AR5211_FPGA11B:
		return 5211;
	case AR5212_DEFAULT:
	case AR5212_DEVID:
	case AR5212_FPGA:
	case AR5212_DEVID_IBM:
	case AR5212_AR5312_REV2:
	case AR5212_AR5312_REV7:
	case AR5212_AR2313_REV8:
	case AR5212_AR2315_REV6:
	case AR5212_AR2315_REV7:
	case AR5212_AR2317_REV1:
	case AR5212_DEVID_0014:
	case AR5212_DEVID_0015:
	case AR5212_DEVID_0016:
	case AR5212_DEVID_0017:
	case AR5212_DEVID_0018:
	case AR5212_DEVID_0019:
	case AR5212_AR2413:
	case AR5212_AR5413:
	case AR5212_AR5424:
	case AR5212_DEVID_FF19:
		return 5212;
	case AR5213_SREV_1_0:
	case AR5213_SREV_REG:
	case AR_SUBVENDOR_ID_NOG:
	case AR_SUBVENDOR_ID_NEW_A:
		return 5213;
	default:
		return 0; /* unknown */
	}
}
static const char *hal_status_desc[] = {
	"No error",
	"No hardware present or device not yet supported",
	"Memory allocation failed",
	"Hardware didn't respond as expected",
	"EEPROM magic number invalid",
	"EEPROM version invalid",
	"EEPROM unreadable",
	"EEPROM checksum invalid",
	"EEPROM read problem",
	"EEPROM mac address invalid",
	"EEPROM size not supported",
	"Attempt to change write-locked EEPROM",
	"Invalid parameter to function",
	"Hardware revision not supported",
	"Hardware self-test failed",
	"Operation incomplete"
};


static const char*
ath_get_hal_status_desc(HAL_STATUS status)
{
	if ((status > 0) && (status < (sizeof(hal_status_desc) / sizeof(char *))))
		return hal_status_desc[status];
	else
		return "";
}

/* Adjust the ratecode used for continuous transmission to the closest rate to the one specified (rounding down) */
static int
ath_get_txcont_adj_ratecode(struct ath_softc *sc)
{
	struct ieee80211com *ic 	= &sc->sc_ic;
	struct ieee80211vap *vap 	= TAILQ_FIRST(&ic->ic_vaps);
	const HAL_RATE_TABLE *rt    = sc->sc_currates;
	int closest_rate_ix         = 0;
	int req_rate 				= 0;
	int j;

	if (vap && (vap->iv_fixed_rate != IEEE80211_FIXED_RATE_NONE) )
		req_rate = vap->iv_fixed_rate / 2;

	if (0 != req_rate) {
		/* Find closest rate to specified rate */
		for (j = 0; j < rt->rateCount; j++) {
			if (((req_rate * 1000) >= rt->info[j].rateKbps) && 
				(rt->info[j].rateKbps >= rt->info[closest_rate_ix].rateKbps)) {
				closest_rate_ix = j;
			}
		}
	}
	
	if (0 == req_rate) {
	/* Diagnostic */
		printk(KERN_INFO "%s: %s: Using default rate of %dM.\n", sc->sc_dev->name, 
				__func__, (rt->info[closest_rate_ix].rateKbps / 1000));
	} else if (req_rate == (rt->info[closest_rate_ix].rateKbps / 1000)) {
		printk(KERN_INFO "%s: %s: Using requested rate of %dM.\n", 
				sc->sc_dev->name, __func__, req_rate);
	} else {
		printk(KERN_INFO "%s: %s: Rounded down requested rate of %dM to %dM.\n", 
				sc->sc_dev->name, __func__, req_rate, 
				(rt->info[closest_rate_ix].rateKbps / 1000));
	}

	return rt->info[closest_rate_ix].rateCode;
}


static void
ath_tx_draintxq(struct ath_softc *sc, struct ath_txq *txq)
{
	struct ath_buf *bf;
	struct sk_buff *skb;
#ifdef ATH_SUPERG_FF
	struct sk_buff *tskb;
	int i;
#endif
	int exit = 0;

	/*
	 * NB: this assumes output has been stopped and
	 *     we do not need to block ath_tx_tasklet
	 */
	for (;;) {
		ATH_TXQ_LOCK(txq);
		bf = STAILQ_FIRST(&txq->axq_q);
		if (bf == NULL) {
			txq->axq_link = NULL;
			exit = 1;
		} else
		ATH_TXQ_REMOVE_HEAD(txq, bf_list);
		ATH_TXQ_UNLOCK(txq);
		if (exit)
			break;
		skb = bf->bf_skb->next;
		bus_unmap_single(sc->sc_bdev,
			bf->bf_skbaddr, bf->bf_skb->len, BUS_DMA_TODEVICE);
		dev_kfree_skb(bf->bf_skb);
#ifdef ATH_SUPERG_FF
		i=0;
		while (skb) {
			tskb = skb->next;
			bus_unmap_single(sc->sc_bdev,
				 bf->bf_skbaddrff[i++],skb->len, BUS_DMA_TODEVICE);
			dev_kfree_skb(skb);
			skb = tskb;
		}
#endif
		if (bf->bf_node)
		ieee80211_free_node(bf->bf_node);

		bf->bf_skb = NULL;
		bf->bf_node = NULL;

		ATH_TXBUF_LOCK_BH(sc);
		STAILQ_INSERT_TAIL(&sc->sc_txbuf, bf, bf_list);
		ATH_TXBUF_UNLOCK_BH(sc);
	}
}

static void
ath_tx_stopdma(struct ath_softc *sc, struct ath_txq *txq)
{
	struct ath_hal *ah = sc->sc_ah;

	(void) ath_hal_stoptxdma(ah, txq->axq_qnum);
	DPRINTF(sc, ATH_DEBUG_TXCONT, "%s: tx queue [%u] %p, link %p\n",
	    __func__, txq->axq_qnum,
	    (caddr_t) ath_hal_gettxbuf(ah, txq->axq_qnum), txq->axq_link);
}

/*
 * Drain the transmit queues and reclaim resources.
 */
static inline void
ath_draintxq(struct ath_softc *sc)
{
	struct ath_hal *ah = sc->sc_ah;
	int i;
	int npend = 0;

	if (!sc->sc_invalid) {
		(void) ath_hal_stoptxdma(ah, sc->sc_bhalq);
		DPRINTF(sc, ATH_DEBUG_TXCONT, "%s: beacon queue %p\n", __func__,
		    (caddr_t) ath_hal_gettxbuf(ah, sc->sc_bhalq));
		for (i = 0; i < HAL_NUM_TX_QUEUES; i++)
			if (ATH_TXQ_SETUP(sc, i)) {
				ath_tx_stopdma(sc, &sc->sc_txq[i]);

				/* The TxDMA may not really be stopped.
				 * Double check the hal tx pending count */
				npend += ath_hal_numtxpending(ah, sc->sc_txq[i].axq_qnum);
			}
	}

	if (npend) {
		HAL_STATUS status;
		/* TxDMA not stopped, reset the hal */
		DPRINTF(sc, ATH_DEBUG_TXCONT, "%s: Unable to stop TxDMA. Reset HAL!\n", __func__);
		if (!ath_hal_reset(ah, sc->sc_ic.ic_opmode, &sc->sc_curchan, AH_TRUE, &status))
			printk("%s: unable to reset hardware; hal status %u\n", __func__, status);
	}

	sc->sc_dev->trans_start = jiffies;
	netif_start_queue(sc->sc_dev);		/* XXX move to callers */
	for (i = 0; i < HAL_NUM_TX_QUEUES; i++)
		if (ATH_TXQ_SETUP(sc, i)) {
			ath_tx_draintxq(sc, &sc->sc_txq[i]);
		}
}

/*
 * Disable the receive h/w in preparation for a reset.
 */
static inline void
ath_stoprecv(struct ath_softc *sc)
{
	struct ath_hal *ah = sc->sc_ah;
	u_int64_t tsf;

	ath_hal_stoppcurecv(ah);	/* disable PCU */

	ath_hal_setrxfilter(ah, 0);	/* clear recv filter */
	ath_hal_stopdmarecv(ah);	/* disable DMA engine */
	mdelay(3);			/* 3ms is long enough for 1 frame */
	tsf = ath_hal_gettsf64(ah);
	sc->sc_rxlink = NULL;		/* just in case */
}


static inline int
ath_set_ack_bitrate(struct ath_softc *sc, int high)
{
	struct ath_hal              *ah = sc->sc_ah;
	if (ar_device(sc->devid) == 5212 || ar_device(sc->devid) == 5213) {
		/* set ack to be sent at low bit-rate */
		/* registers taken from the OpenBSD 5212 HAL */
#define AR5K_AR5212_STA_ID1                     0x8004
#define AR5K_AR5212_STA_ID1_ACKCTS_6MB          0x01000000
#define AR5K_AR5212_STA_ID1_BASE_RATE_11B       0x02000000
		u_int32_t v = AR5K_AR5212_STA_ID1_BASE_RATE_11B | AR5K_AR5212_STA_ID1_ACKCTS_6MB;
		if (high) {
			OS_REG_WRITE(ah, AR5K_AR5212_STA_ID1, OS_REG_READ(ah, AR5K_AR5212_STA_ID1) & ~v);
		} else {
			OS_REG_WRITE(ah, AR5K_AR5212_STA_ID1, OS_REG_READ(ah, AR5K_AR5212_STA_ID1) | v);
		}
#undef AR5K_AR5212_STA_ID1
#undef AR5K_AR5212_STA_ID1_BASE_RATE_11B
#undef AR5K_AR5212_STA_ID1_ACKCTS_6MB
		return 0;
	} else 
		DPRINTF(sc, ATH_DEBUG_TXCONT, "%s Devid %d not supported ?\n", __func__, sc->devid);
	return 1;
}


/*
Configure the radio for continuous transmission
*/
static int
txcont_configure_radio(struct ieee80211com *ic)
{
	struct net_device           *dev = ic->ic_dev;
	struct ath_softc            *sc = dev->priv;
	struct ath_hal              *ah = sc->sc_ah;
	struct ieee80211_wme_state  *wme = &ic->ic_wme;
	struct ieee80211vap         *vap = TAILQ_FIRST(&ic->ic_vaps);

	HAL_STATUS status;
	int q;

	if (IFF_RUNNING != (ic->ic_dev->flags & IFF_RUNNING)) {
		printk(KERN_ERR "%s: %s: Cannot enable txcont when interface is"
			" not in running state.\n", dev->name, __func__);
		return 1;
	}
	if (vap == NULL) {
		printk(KERN_ERR "%s: Cannot enable txcont when no VAP is created.\n", __func__);
		return 1;
	}
	
	if (vap->iv_opmode != IEEE80211_M_HOSTAP) {
		printk(KERN_ERR "%s: Cannot enable txcont on STA VAP.\n"
							"First VAP must be created in AP mode for txcont using.\n", __func__);
		return 1;
	}
	

	if (ic->ic_bsschan == IEEE80211_CHAN_ANYC) {
		if (vap->iv_des_chan == IEEE80211_CHAN_ANYC)
			vap->iv_des_chan = ic->ic_curchan;
		vap->iv_newstate(vap, IEEE80211_S_INIT, -1);
		vap->iv_newstate(vap, IEEE80211_S_SCAN, 0);
		vap->iv_newstate(vap, IEEE80211_S_RUN,  0);
	}

	/* stop calibrate timer */
	del_timer_sync(&sc->sc_cal_ch);
	/* reset radio and do clear reconfigure */
	ath_hal_intrset(ah, 0);
	ic->ic_reset(dev);
	{
		int ac;

		/* Prepare to reconfigure */
		ic->ic_caps  &= ~IEEE80211_C_SHPREAMBLE;
		ic->ic_coverageclass = 0;
		ic->ic_flags &= ~IEEE80211_F_DOTH;
		ic->ic_flags &= ~IEEE80211_F_SHPREAMBLE;
		ic->ic_flags &= ~IEEE80211_F_TXPOW_FIXED;
		ic->ic_flags |= IEEE80211_F_USEBARKER;
		ic->ic_flags_ext &= ~IEEE80211_FEXT_COUNTRYIE;
		ic->ic_flags_ext &= ~IEEE80211_FEXT_MARKDFS;
		ic->ic_flags_ext &= ~IEEE80211_FEXT_REGCLASS;
		ic->ic_protmode = IEEE80211_PROT_NONE;
		ic->ic_roaming = IEEE80211_ROAMING_DEVICE;
		vap->iv_flags &= ~IEEE80211_F_BGSCAN;
		vap->iv_flags &= ~IEEE80211_F_DROPUNENC;
		vap->iv_flags &= ~IEEE80211_F_NOBRIDGE;
		vap->iv_flags &= ~IEEE80211_F_PRIVACY;
		vap->iv_flags |= IEEE80211_F_PUREG;
		vap->iv_flags |= IEEE80211_F_WME;
		vap->iv_flags_ext &= ~IEEE80211_FEXT_UAPSD;
		vap->iv_flags_ext &= ~IEEE80211_FEXT_WDS;
		vap->iv_ic->ic_flags |= IEEE80211_F_WME; /* needed by ic_reset */
		vap->iv_mcast_rate = 54000;
		vap->iv_uapsdinfo = 0;
		vap->iv_ath_cap |= IEEE80211_ATHC_BURST;
		vap->iv_ath_cap |= IEEE80211_ATHC_FF;
		vap->iv_ath_cap &= ~IEEE80211_ATHC_AR;
		vap->iv_ath_cap &= ~IEEE80211_ATHC_COMP;
		vap->iv_des_ssid[0].len = 0;
		vap->iv_des_nssid = 1;
		sc->sc_txantenna = sc->sc_defant = sc->sc_mcastantenna = sc->sc_rxotherant = 1;
		sc->sc_diversity = 0;
		memset(vap->iv_des_ssid[0].ssid, 0, IEEE80211_ADDR_LEN);
		ath_hal_setdiversity(sc->sc_ah, 0);

		for (ac = 0; ac < WME_NUM_AC; ac++) {
			/* AIFSN = 1 */
			wme->wme_wmeBssChanParams.cap_wmeParams[ac].wmep_aifsn     =
			    wme->wme_bssChanParams.cap_wmeParams[ac].wmep_aifsn        =
			    wme->wme_wmeChanParams.cap_wmeParams[ac].wmep_aifsn        =
			    wme->wme_chanParams.cap_wmeParams[ac].wmep_aifsn           =
			    1;

			/*  CWMIN = 1                                                */
			wme->wme_wmeBssChanParams.cap_wmeParams[ac].wmep_logcwmin  =
			    wme->wme_bssChanParams.cap_wmeParams[ac].wmep_logcwmin     =
			    wme->wme_wmeChanParams.cap_wmeParams[ac].wmep_logcwmin     =
			    wme->wme_chanParams.cap_wmeParams[ac].wmep_logcwmin        =
			    1;

			/*  CWMAX = 1                                                */
			wme->wme_wmeBssChanParams.cap_wmeParams[ac].wmep_logcwmax  =
			    wme->wme_bssChanParams.cap_wmeParams[ac].wmep_logcwmax     =
			    wme->wme_wmeChanParams.cap_wmeParams[ac].wmep_logcwmax     =
			    wme->wme_chanParams.cap_wmeParams[ac].wmep_logcwmax        =
			    1;

			/*  ACM = 1                                                  */
			wme->wme_wmeBssChanParams.cap_wmeParams[ac].wmep_acm       =
			    wme->wme_bssChanParams.cap_wmeParams[ac].wmep_acm          =
			    0;

			/*  NOACK = 1                                                */
			wme->wme_wmeChanParams.cap_wmeParams[ac].wmep_noackPolicy  =
			    wme->wme_chanParams.cap_wmeParams[ac].wmep_noackPolicy     =
			    1;

			/*  TXOPLIMIT = 8192 */
			wme->wme_wmeBssChanParams.cap_wmeParams[ac].wmep_txopLimit =
			    wme->wme_bssChanParams.cap_wmeParams[ac].wmep_txopLimit    =
			    wme->wme_wmeChanParams.cap_wmeParams[ac].wmep_txopLimit    =
			    wme->wme_chanParams.cap_wmeParams[ac].wmep_txopLimit       =
			    IEEE80211_US_TO_TXOP(8192);
		}
//		ieee80211_cancel_scan(vap);	/* anything current */
//		ieee80211_wme_updateparams(vap);
		/*  reset the WNIC */
		if (!ath_hal_reset(ah, ic->ic_opmode, &sc->sc_curchan, AH_TRUE, &status)) {
			printk(KERN_ERR "%s: ath_hal_reset failed: '%s' "
					"(HAL status %u) in %s at %s:%d\n", 
					dev->name, ath_get_hal_status_desc(status), 
					status,  __func__, __FILE__, __LINE__);
		}
//		ath_update_txpow(sc);
//		ath_radar_update(sc);		
//		ath_radar_pulse_flush(sc);

#ifdef ATH_SUPERG_DYNTURBO
		/*  Turn on dynamic turbo if necessary -- before we get into our own implementation -- and before we configures */
		if ((!IEEE80211_IS_CHAN_STURBO(ic->ic_bsschan)) && 
				(IEEE80211_ATHC_TURBOP & vap->iv_ath_cap) && 
				(IEEE80211_IS_CHAN_ANYG(ic->ic_bsschan) || 
				 IEEE80211_IS_CHAN_A(ic->ic_bsschan))) {
			u_int32_t newflags = ic->ic_bsschan->ic_flags;
			if (IEEE80211_ATHC_TURBOP & vap->iv_ath_cap) {
				DPRINTF(sc, ATH_DEBUG_TXCONT, "%s: Enabling dynamic turbo...\n",
						dev->name);
				ic->ic_ath_cap |= IEEE80211_ATHC_BOOST;
				sc->sc_ignore_ar = 1;
				newflags |= IEEE80211_CHAN_TURBO;
			} else {
				DPRINTF(sc, ATH_DEBUG_TXCONT,
						"%s: Disabling dynamic turbo...\n", dev->name);
				ic->ic_ath_cap &= ~IEEE80211_ATHC_BOOST;
				newflags &= ~IEEE80211_CHAN_TURBO;
			}
			ieee80211_dturbo_switch(ic, newflags);
			/*  Keep interupts off, just in case... */
			ath_hal_intrset(ah, 0);
		}
#endif /* #ifdef ATH_SUPERG_DYNTURBO */
		/* clear pending tx frames picked up after reset */
		ath_draintxq(sc);
		/* stop receive side */
		ath_stoprecv(sc);
		ath_hal_setrxfilter(ah, 0);
		ath_hal_setmcastfilter(ah, 0, 0);
		ath_set_ack_bitrate(sc, /*sc->sc_ackrate*/ 0);
//		netif_wake_queue(dev);		/* restart xmit */

		if (ar_device(sc->devid) == 5212 || ar_device(sc->devid) == 5213) {
			/* registers taken from openhal */
#define AR5K_AR5212_TXCFG				0x0030
#define AR5K_AR5212_TXCFG_TXCONT_ENABLE			0x00000080
#define AR5K_AR5212_RSSI_THR				0x8018
#define AR5K_AR5212_PHY_NF				0x9864
#define AR5K_AR5212_ADDAC_TEST				0x8054
#define AR5K_AR5212_DIAG_SW				0x8048
#define AR5K_AR5212_DIAG_SW_IGNOREPHYCS			0x00100000
#define AR5K_AR5212_DIAG_SW_IGNORENAV			0x00200000
#define AR5K_AR5212_DCU_GBL_IFS_SIFS			0x1030
#define AR5K_AR5212_DCU_GBL_IFS_SIFS_M			0x0000ffff
#define AR5K_AR5212_DCU_GBL_IFS_EIFS			0x10b0
#define AR5K_AR5212_DCU_GBL_IFS_EIFS_M			0x0000ffff
#define AR5K_AR5212_DCU_GBL_IFS_SLOT			0x1070
#define AR5K_AR5212_DCU_GBL_IFS_SLOT_M			0x0000ffff
#define AR5K_AR5212_DCU_GBL_IFS_MISC			0x10f0
#define	AR5K_AR5212_DCU_GBL_IFS_MISC_LFSR_SLICE		0x00000007
#define	AR5K_AR5212_DCU_GBL_IFS_MISC_TURBO_MODE		0x00000008
#define	AR5K_AR5212_DCU_GBL_IFS_MISC_SIFS_DUR_USEC	0x000003f0
#define	AR5K_AR5212_DCU_GBL_IFS_MISC_USEC_DUR		0x000ffc00
#define	AR5K_AR5212_DCU_GBL_IFS_MISC_DCU_ARB_DELAY	0x00300000
#define	AR5K_AR5212_DCU_MISC_POST_FR_BKOFF_DIS		0x00200000
#define	AR5K_AR5212_DCU_CHAN_TIME_DUR			0x000fffff
#define	AR5K_AR5212_DCU_CHAN_TIME_ENABLE		0x00100000
#define	AR5K_AR5212_QCU(_n, _a)		                (((_n) << 2) + _a)
#define	AR5K_AR5212_DCU(_n, _a)		                AR5K_AR5212_QCU(_n, _a)
#define AR5K_AR5212_DCU_MISC(_n)			AR5K_AR5212_DCU(_n, 0x1100)
#define AR5K_AR5212_DCU_CHAN_TIME(_n)			AR5K_AR5212_DCU(_n, 0x10c0)
			/* NB: This section of direct hardware access contains 
			 * a continuous transmit mode derived by reverse 
			 * engineering. Many of the settings may be unnecessary 
			 * to achieve the end result. Additional testing, 
			 * selectively commenting out register writes below may 
			 * result in simpler code with the same results. */

			/*  Set RSSI threshold to extreme, hear nothing */
			OS_REG_WRITE(ah, AR5K_AR5212_RSSI_THR, 0xffffffff);
			/*  Blast away at noise floor, assuming AGC has 
			 *  already set it... we want to trash it. */
			OS_REG_WRITE(ah, AR5K_AR5212_PHY_NF,   0xffffffff);
			/* Enable continuous transmit mode / DAC test mode */
			OS_REG_WRITE(ah, AR5K_AR5212_ADDAC_TEST, 
					OS_REG_READ(ah, AR5K_AR5212_ADDAC_TEST) | 1);
			/* Ignore real and virtual carrier sensing, and reception */
			OS_REG_WRITE(ah, AR5K_AR5212_DIAG_SW, 
					OS_REG_READ(ah, AR5K_AR5212_DIAG_SW) | 
					AR5K_AR5212_DIAG_SW_IGNOREPHYCS | 
					AR5K_AR5212_DIAG_SW_IGNORENAV);
			/*  Set SIFS to rediculously small value...  */
			OS_REG_WRITE(ah, AR5K_AR5212_DCU_GBL_IFS_SIFS, 
					(OS_REG_READ(ah, AR5K_AR5212_DCU_GBL_IFS_SIFS) &
					 ~AR5K_AR5212_DCU_GBL_IFS_SIFS_M) | 1);
			/*  Set EIFS to rediculously small value...  */
			OS_REG_WRITE(ah, AR5K_AR5212_DCU_GBL_IFS_EIFS, 
					(OS_REG_READ(ah, AR5K_AR5212_DCU_GBL_IFS_EIFS) & 
					 ~AR5K_AR5212_DCU_GBL_IFS_EIFS_M) | 1);
			/*  Set slot time to rediculously small value...  */
			OS_REG_WRITE(ah, AR5K_AR5212_DCU_GBL_IFS_SLOT, 
					(OS_REG_READ(ah, AR5K_AR5212_DCU_GBL_IFS_SLOT) & 
					 ~AR5K_AR5212_DCU_GBL_IFS_SLOT_M) | 1);
			OS_REG_WRITE(ah, AR5K_AR5212_DCU_GBL_IFS_MISC,
			    OS_REG_READ(ah, AR5K_AR5212_DCU_GBL_IFS_MISC) & 
			    ~AR5K_AR5212_DCU_GBL_IFS_MISC_SIFS_DUR_USEC & 
			    ~AR5K_AR5212_DCU_GBL_IFS_MISC_USEC_DUR & 
			    ~AR5K_AR5212_DCU_GBL_IFS_MISC_DCU_ARB_DELAY & 
			    ~AR5K_AR5212_DCU_GBL_IFS_MISC_LFSR_SLICE);

			/*  Disable queue backoff (default was like 256 or 0x100) */
			for (q = 0; q < 4; q++) {
				OS_REG_WRITE(ah, AR5K_AR5212_DCU_MISC(q), AR5K_AR5212_DCU_MISC_POST_FR_BKOFF_DIS);
				/*  Set the channel time (burst time) to the highest setting the register can take, forget this compliant 8192 limit... */
				OS_REG_WRITE(ah, AR5K_AR5212_DCU_CHAN_TIME(q), AR5K_AR5212_DCU_CHAN_TIME_ENABLE | AR5K_AR5212_DCU_CHAN_TIME_DUR);
			}
			/*  Set queue full to continuous */
			OS_REG_WRITE(ah, AR5K_AR5212_TXCFG, OS_REG_READ(ah, AR5K_AR5212_TXCFG) | AR5K_AR5212_TXCFG_TXCONT_ENABLE);
#undef AR5K_AR5212_TXCFG
#undef AR5K_AR5212_TXCFG_TXCONT_ENABLE
#undef AR5K_AR5212_RSSI_THR
#undef AR5K_AR5212_PHY_NF
#undef AR5K_AR5212_ADDAC_TEST
#undef AR5K_AR5212_DIAG_SW
#undef AR5K_AR5212_DIAG_SW_IGNOREPHYCS
#undef AR5K_AR5212_DIAG_SW_IGNORENAV
#undef AR5K_AR5212_DCU_GBL_IFS_SIFS
#undef AR5K_AR5212_DCU_GBL_IFS_SIFS_M
#undef AR5K_AR5212_DCU_GBL_IFS_EIFS
#undef AR5K_AR5212_DCU_GBL_IFS_EIFS_M
#undef AR5K_AR5212_DCU_GBL_IFS_SLOT
#undef AR5K_AR5212_DCU_GBL_IFS_SLOT_M
#undef AR5K_AR5212_DCU_GBL_IFS_MISC
#undef AR5K_AR5212_DCU_GBL_IFS_MISC_LFSR_SLICE
#undef AR5K_AR5212_DCU_GBL_IFS_MISC_TURBO_MODE
#undef AR5K_AR5212_DCU_GBL_IFS_MISC_SIFS_DUR_USEC
#undef AR5K_AR5212_DCU_GBL_IFS_MISC_USEC_DUR
#undef AR5K_AR5212_DCU_GBL_IFS_MISC_DCU_ARB_DELAY
#undef AR5K_AR5212_DCU_MISC_POST_FR_BKOFF_DIS
#undef AR5K_AR5212_DCU_CHAN_TIME_DUR
#undef AR5K_AR5212_DCU_CHAN_TIME_ENABLE
#undef AR5K_AR5212_QCU
#undef AR5K_AR5212_DCU
#undef AR5K_AR5212_DCU_MISC
#undef AR5K_AR5212_DCU_CHAN_TIME
		} else 
			DPRINTF(sc, ATH_DEBUG_TXCONT, "%s Devid %d not supported ?\n", __func__, sc->devid);

		/* Disable beacons and beacon miss interrupts */
		sc->sc_beacons = 0;
		sc->sc_imask &= ~(HAL_INT_SWBA | HAL_INT_BMISS);

	}
//	ath_hal_intrset(ah, sc->sc_imask);
	return 0;
}


static __inline void
ath_desc_swap(struct ath_desc *ds)
{
#ifdef AH_NEED_DESC_SWAP
	ds->ds_link = cpu_to_le32(ds->ds_link);
	ds->ds_data = cpu_to_le32(ds->ds_data);
	ds->ds_ctl0 = cpu_to_le32(ds->ds_ctl0);
	ds->ds_ctl1 = cpu_to_le32(ds->ds_ctl1);
	ds->ds_hw[0] = cpu_to_le32(ds->ds_hw[0]);
	ds->ds_hw[1] = cpu_to_le32(ds->ds_hw[1]);
#endif
}


static void
hal_tx_start(struct ath_hal *ah, u_int q)
{
#define	AR_Q_TXE	0x0840	/* MAC Transmit Queue enable */
#define	AR_Q_TXD	0x0880	/* MAC Transmit Queue disable */
	TRACE();
	if (OS_REG_READ(ah, AR_Q_TXD) & (1 << q)) {
		MYDEBUG("queue has TXD bit set");
	}

	TRACE();
	OS_REG_WRITE(ah, AR_Q_TXE, 1 << q);
#undef	AR_Q_TXE
#undef	AR_Q_TXD
	TRACE();
}


/*
 * Insert a buffer on a txq
 *
 */
static inline void
ath_tx_txqaddbuf(struct ath_softc *sc, struct ieee80211_node *ni,
		 struct ath_txq *txq, struct ath_buf *bf,
		 struct ath_desc *lastds, int framelen)
{
	struct ath_hal *ah = sc->sc_ah;

	/*
	 * Insert the frame on the outbound list and
	 * pass it on to the hardware.
	 */
	ATH_TXQ_LOCK_IRQ(txq);

	ATH_TXQ_INSERT_TAIL(txq, bf, bf_list);
	DPRINTF(sc, ATH_DEBUG_TXCONT, "%s: txq depth = %d\n", __func__, txq->axq_depth);
	if (txq->axq_link == NULL) {
		ath_hal_puttxbuf(ah, txq->axq_qnum, bf->bf_daddr);
		DPRINTF(sc, ATH_DEBUG_TXCONT, "%s: TXDP[%u] = %llx (%p)\n",
			__func__,
			txq->axq_qnum, ito64(bf->bf_daddr), bf->bf_desc);
	} else {
#ifdef AH_NEED_DESC_SWAP
		*txq->axq_link = cpu_to_le32(bf->bf_daddr);
#else
		*txq->axq_link = bf->bf_daddr;
#endif
		DPRINTF(sc, ATH_DEBUG_TXCONT, "%s: link[%u] (%p)=%llx (%p)\n",
			__func__,
			txq->axq_qnum, txq->axq_link,
			ito64(bf->bf_daddr), bf->bf_desc);
	}
	txq->axq_link = &lastds->ds_link;
	TRACE();
//	ath_hal_txstart(ah, txq->axq_qnum);
	hal_tx_start(ah, txq->axq_qnum);
	sc->sc_dev->trans_start = jiffies;

	ATH_TXQ_UNLOCK_IRQ(txq);

	sc->sc_devstats.tx_packets++;
	sc->sc_devstats.tx_bytes += framelen;
	TRACE();
}



/* Queue a self-looped packet for the specified hardware queue. */
static void
txcont_queue_packet(struct ieee80211com *ic, struct ath_txq* txq)
{
	struct net_device *dev             = ic->ic_dev;
	struct ath_softc *sc               = dev->priv;
	struct ath_hal *ah                 = sc->sc_ah;
	struct ath_buf *bf                 = NULL;
	struct sk_buff *skb                = NULL;
	unsigned int i;
	/* maximum supported size, subtracting headers and required slack */
	unsigned int datasz                = 4028;
	struct ieee80211_frame* wh         = NULL;
	unsigned char *data                = NULL;
	unsigned char *crc                 = NULL;

	if (IFF_RUNNING != (ic->ic_dev->flags & IFF_RUNNING)) {
		printk(KERN_ERR "%s: %s: Refusing to queue self linked frame "
				"when txcont is not enabled.\n", dev->name, __func__);
		return;
	}

	ath_hal_intrset(ah, 0);
	{
		bf  = STAILQ_FIRST(&sc->sc_txbuf);
		STAILQ_REMOVE_HEAD(&sc->sc_txbuf, bf_list);
		skb = alloc_skb(datasz + sizeof(struct ieee80211_frame) + IEEE80211_CRC_LEN, GFP_ATOMIC);
		wh  = (struct ieee80211_frame*)skb_put(skb, sizeof(struct ieee80211_frame));
		if (NULL == skb) {
			printk(KERN_ERR "%s: %s: alloc_skb returned null!\n", dev->name, __func__);
			BUG();
		}
		if (NULL == bf) {
			printk(KERN_ERR "%s: %s: STAILQ_FIRST(&sc->sc_txbuf) returned null!\n", dev->name, __func__);
			BUG();
		}

		/*  Define the SKB format */
		data   = skb_put(skb, datasz);

		/*  NB: little endian */
		
		/*  11110000 (protocol = 00, type = 00 "management", 
		 *  subtype = 1111 "reserved/undocumented" */
		wh->i_fc[0]    = 0xf0;
		/* leave out to/from DS, frag., retry, pwr mgt, more data, 
		 * protected frame, and order bit */
		wh->i_fc[1]    = 0x00;
		/* NB: Duration is left at zero, for broadcast frames. */
		wh->i_dur[0]      = 0;
		wh->i_dur[1]      = 0;
		/*  DA (destination address) */
		wh->i_addr1[0] = wh->i_addr1[1] = wh->i_addr1[2] = 
			wh->i_addr1[3] = wh->i_addr1[4] = wh->i_addr1[5] = 0xff;
		/*  BSSID */
		wh->i_addr2[0] = wh->i_addr2[1] = wh->i_addr2[2] = 
			wh->i_addr2[3] = wh->i_addr2[4] = wh->i_addr2[5] = 0xff;
		/*  SA (source address) */
		wh->i_addr3[0] = wh->i_addr3[1] = wh->i_addr3[2] = 
			wh->i_addr3[3] = wh->i_addr3[4] = wh->i_addr3[5] = 0x00;
		/*  Sequence is zero for now, let the hardware assign this or 
		 *  not, depending on how we setup flags (below) */
		wh->i_seq[0]   = 0x00; 
		wh->i_seq[1]   = 0x00;

		/*  Initialize the data */
		if (datasz % 4)	BUG();
		for (i = 0; i < datasz; i+=4) {
			data[i+0] = 0x00;
			data[i+1] = 0xff;
			data[i+2] = 0x00;
			data[i+3] = 0xff;
		}

		/*  Add space for the CRC */
		crc    = skb_put(skb, IEEE80211_CRC_LEN);

		/*  Clear  */
		crc[0] = crc[1] = crc[2] = crc[3] = 0x00;

		/*  Initialize the selfed-linked frame */
		bf->bf_skb      = skb;
		bf->bf_skbaddr  = bus_map_single(sc->sc_bdev, bf->bf_skb->data, 
				bf->bf_skb->len, BUS_DMA_TODEVICE);
		bf->bf_flags    = HAL_TXDESC_CLRDMASK | HAL_TXDESC_NOACK;
		bf->bf_node     = NULL;
		bf->bf_desc->ds_link = bf->bf_daddr;
		bf->bf_desc->ds_data = bf->bf_skbaddr;

		ath_hal_setuptxdesc(ah,
		    bf->bf_desc,		     /* the descriptor */
		    skb->len,			     /* packet length */
		    sizeof(struct ieee80211_frame),  /* header length */
		    HAL_PKT_TYPE_NORMAL,	     /* Atheros packet type */
		    ic->ic_txpowlimit/*sc->sc_tx99_power*/,	     /* txpower in 0.5dBm increments, range 0-n depending upon card typically 60-100 max */
		    ath_get_txcont_adj_ratecode(sc),  /* series 0 rate */
		    0,				     /* series 0 retries */
		    HAL_TXKEYIX_INVALID,	     /* key cache index */
		    sc->sc_txantenna,		     /* antenna mode */
		    bf->bf_flags,		     /* flags */
		    0,				     /* rts/cts rate */
		    0,				     /* rts/cts duration */
		    0,				     /* comp icv len */
		    0,				     /* comp iv len */
		    ATH_COMP_PROC_NO_COMP_NO_CCS     /* comp scheme */
		    );

		ath_hal_filltxdesc(ah,
		    bf->bf_desc,	/* Descriptor to fill */
		    skb->len,		/* buffer length */
		    AH_TRUE,		/* is first segment */
		    AH_TRUE,		/* is last segment */
		    bf->bf_desc		/* first descriptor */
		    );

		/*  Byteswap (as necessary) */
		ath_desc_swap(bf->bf_desc);
		/*  queue the self-linked frame */
		ath_tx_txqaddbuf(sc, NULL,	/* node */
		    txq,			/* hardware queue */
		    bf,				/* atheros buffer */
		    bf->bf_desc,		/* last descriptor */
		    bf->bf_skb->len		/* frame length */
		    );
	TRACE();
//		ath_hal_txstart(ah, txq->axq_qnum);
		hal_tx_start(ah, txq->axq_qnum);
	TRACE();
	}
//	ath_hal_intrset(ah, sc->sc_imask);
}

/* Turn on continuous transmission */
static int
txcont_on(struct ath_txcont *txcont)
{
	struct net_device *dev = txcont->dev;
	struct ath_softc *sc = dev->priv;
	struct ieee80211com *ic = &sc->sc_ic;

	if (IFF_RUNNING != (ic->ic_dev->flags & IFF_RUNNING)) {
		printk(KERN_ERR "%s: %s: Cannot enable txcont when interface is not in running state.\n", dev->name, __func__);
		return -EFAULT;
	}

	if (txcont_configure_radio(ic))
		return -EFAULT;

	txcont_queue_packet(ic, sc->sc_ac2q[WME_AC_BE]);
//	txcont_queue_packet(ic, sc->sc_ac2q[WME_AC_BK]);
//	txcont_queue_packet(ic, sc->sc_ac2q[WME_AC_VI]);
//	txcont_queue_packet(ic, sc->sc_ac2q[WME_AC_VO]);
	printk("%s txcont started on %s\n", mod_info, dev->name);
	return 0;
}

/* Turn off continuous transmission */
static void
txcont_off(struct ath_txcont *txcont)
{
	struct net_device *dev = txcont->dev;
	struct ath_softc *sc = dev->priv;
	struct ieee80211com *ic = &sc->sc_ic;
	struct ieee80211vap *vap	= TAILQ_FIRST(&ic->ic_vaps);
	
	printk("%s stop txcont on %s\n", mod_info, dev->name);
	if (!sc->sc_invalid) {
		ic->ic_reset(dev);
	}
	if(vap && (vap->iv_opmode != IEEE80211_M_WDS) )
		sc->sc_beacons = 1;
}



/*
 * Module glue.
 */
MODULE_AUTHOR("UBNT");
MODULE_DESCRIPTION("txcont support for Atheros devices");
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
MODULE_PARM(device, "s");
#else
module_param(device, charp, 0444);
#endif


static int __init
init_ath_txcont(void)
{	
	printk(KERN_INFO "%s: continuous transmit mode for Atheros devices\n", mod_info);
	if (device == NULL) {
		printk(KERN_INFO "%s: no device specified for testing\n",
			mod_info);
		return -ENOSYS;
	}
	txcont_dev = dev_get_by_name(device);
	if (txcont_dev == NULL) {
		printk(KERN_INFO "%s: device %s not found\n", mod_info, device);
		return -ENOSYS;
	}
	txcont_attach(txcont_dev);
	return 0;
}
module_init(init_ath_txcont);

static void __exit
exit_ath_txcont(void)
{
	if (txcont_dev != NULL) {
		struct ath_softc *sc = txcont_dev->priv;
		if (sc->sc_tx99 != NULL)
			txcont_detach(sc->sc_tx99);
	}
	printk(KERN_INFO "%s: unloaded\n", mod_info);
}
module_exit(exit_ath_txcont);
