/*
 * Scatterlist Cryptographic API.
 *
 * Copyright (c) 2002 James Morris <jmorris@intercode.com.au>
 * Copyright (c) 2002 David S. Miller (davem@redhat.com)
 *
 * Portions derived from Cryptoapi, by Alexander Kjeldaas <astor@fast.no>
 * and Nettle, by Niels Mller.
 *
 * HW crypto extensions added by Eugene Surovegin <ebs@ebshome.net>, 2004
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 */
#include <linux/init.h>
#include <linux/crypto.h>
#include <linux/errno.h>
#include <linux/rwsem.h>
#include <linux/slab.h>
#include "internal.h"

struct list_head crypto_alg_list[CRYPTO_PRIO_NUM] = {
	LIST_HEAD_INIT(crypto_alg_list[0]),
#ifdef CONFIG_CRYPTO_HW		
	LIST_HEAD_INIT(crypto_alg_list[1])
#endif	
};
DECLARE_RWSEM(crypto_alg_sem);

static inline int crypto_alg_prio_get(struct crypto_alg *alg)
{
	return alg->cra_flags & CRYPTO_ALG_HW ? CRYPTO_PRIO_HW : CRYPTO_PRIO_SW;
}

static inline int crypto_alg_get(struct crypto_alg *alg)
{
	return try_inc_mod_count(alg->cra_module);
}

static inline void crypto_alg_put(struct crypto_alg *alg)
{
	if (alg->cra_module)
		__MOD_DEC_USE_COUNT(alg->cra_module);
}

static inline struct crypto_alg* __crypto_alg_lookup(const char* name, 
	int start_prio)
{
	struct crypto_alg *q;
	int i;
	for (i = start_prio; i < CRYPTO_PRIO_NUM; ++i)
		list_for_each_entry(q, &crypto_alg_list[i], cra_list) {
			if (!(strcmp(q->cra_name, name))) {
				if (crypto_alg_get(q))
					return q;
				break;
			}
		}
	return NULL;
}

struct crypto_alg *crypto_alg_lookup(const char *name)
{
	struct crypto_alg *alg = NULL;

	if (!name)
		return NULL;
	
	down_read(&crypto_alg_sem);
	alg = __crypto_alg_lookup(name, CRYPTO_PRIO_HW);
	up_read(&crypto_alg_sem);
	return alg;
}

static int crypto_init_flags(struct crypto_tfm *tfm, u32 flags)
{
	tfm->crt_flags = 0;
	
	switch (crypto_tfm_alg_type(tfm)) {
	case CRYPTO_ALG_TYPE_CIPHER:
		return crypto_init_cipher_flags(tfm, flags);
		
	case CRYPTO_ALG_TYPE_DIGEST:
		return crypto_init_digest_flags(tfm, flags);
		
	case CRYPTO_ALG_TYPE_COMPRESS:
		return crypto_init_compress_flags(tfm, flags);
	
	default:
		break;
	}
	
	BUG();
	return -EINVAL;
}

static int crypto_init_ops(struct crypto_tfm *tfm)
{
	switch (crypto_tfm_alg_type(tfm)) {
	case CRYPTO_ALG_TYPE_CIPHER:
		return crypto_init_cipher_ops(tfm);
		
	case CRYPTO_ALG_TYPE_DIGEST:
		return crypto_init_digest_ops(tfm);
		
	case CRYPTO_ALG_TYPE_COMPRESS:
		return crypto_init_compress_ops(tfm);
	
	default:
		break;
	}
	
	BUG();
	return -EINVAL;
}

static void crypto_exit_ops(struct crypto_tfm *tfm)
{
	switch (crypto_tfm_alg_type(tfm)) {
	case CRYPTO_ALG_TYPE_CIPHER:
		crypto_exit_cipher_ops(tfm);
		break;
		
	case CRYPTO_ALG_TYPE_DIGEST:
		crypto_exit_digest_ops(tfm);
		break;
		
	case CRYPTO_ALG_TYPE_COMPRESS:
		crypto_exit_compress_ops(tfm);
		break;
	
	default:
		BUG();
		
	}
}

struct crypto_tfm *crypto_alloc_tfm(const char *name, u32 flags)
{
	struct crypto_tfm *tfm = NULL;
	struct crypto_alg *alg;
	size_t ctx_size;

	alg = crypto_alg_mod_lookup(name);
	if (alg == NULL)
		goto out;
	
	ctx_size = alg->cra_ctxsize + 
		   (alg->cra_helper ? alg->cra_helper->cra_ctxsize : 0);
	tfm = kmalloc(sizeof(*tfm) + ctx_size, GFP_KERNEL);
	if (tfm == NULL)
		goto out_put;

	memset(tfm, 0, sizeof(*tfm) + ctx_size);
	
	tfm->__crt_alg = alg;
	
	if (crypto_init_flags(tfm, flags))
		goto out_free_tfm;
		
	if (crypto_init_ops(tfm)) {
		crypto_exit_ops(tfm);
		goto out_free_tfm;
	}

	goto out;

out_free_tfm:
	kfree(tfm);
	tfm = NULL;
out_put:
	crypto_alg_put(alg);
out:
	return tfm;
}

void crypto_free_tfm(struct crypto_tfm *tfm)
{
	crypto_exit_ops(tfm);
	crypto_alg_put(tfm->__crt_alg);
	kfree(tfm);
}

#ifdef CONFIG_CRYPTO_HW
static inline int crypto_alg_compat(struct crypto_alg* a1, struct crypto_alg* a2)
{
    return (a1->cra_flags & CRYPTO_ALG_TYPE_MASK) == (a2->cra_flags & CRYPTO_ALG_TYPE_MASK)
	    && a1->cra_blocksize == a2->cra_blocksize;
}

static void crypto_update_alg_helpers(void)
{
	struct crypto_alg *q;
	list_for_each_entry(q, &crypto_alg_list[CRYPTO_PRIO_HW], cra_list) {
		if (!q->cra_helper){
			q->cra_helper = __crypto_alg_lookup(q->cra_name, CRYPTO_PRIO_SW);
			if (q->cra_helper && !crypto_alg_compat(q, q->cra_helper)){
				crypto_alg_put(q->cra_helper);
				q->cra_helper = 0;
			}
		}
	}
}
#else
static void crypto_update_alg_helpers(void)
{ }
#endif /* CONFIG_CRYPTO_HW */

int crypto_register_alg(struct crypto_alg *alg)
{
	int ret = 0, prio = crypto_alg_prio_get(alg);
	struct crypto_alg *q;
	
	down_write(&crypto_alg_sem);
	
	list_for_each_entry(q, &crypto_alg_list[prio], cra_list) {
		if (!(strcmp(q->cra_name, alg->cra_name))) {
			ret = -EEXIST;
			goto out;
		}
	}
	
	list_add_tail(&alg->cra_list, &crypto_alg_list[prio]);
	
	/* Rescan hw alg list and fill helpers if needed */
	crypto_update_alg_helpers();
out:	
	up_write(&crypto_alg_sem);
	return ret;
}

int crypto_unregister_alg(struct crypto_alg *alg)
{
	int ret = -ENOENT, prio = crypto_alg_prio_get(alg);
	struct crypto_alg *q;
	
	BUG_ON(!alg->cra_module);
	
	down_write(&crypto_alg_sem);
	list_for_each_entry(q, &crypto_alg_list[prio], cra_list) {
		if (alg == q) {
			if (alg->cra_helper){
				crypto_alg_put(alg->cra_helper);
				alg->cra_helper = 0;
			}	
			list_del(&alg->cra_list);
			ret = 0;
			goto out;
		}
	}
out:	
	up_write(&crypto_alg_sem);
	return ret;
}

int crypto_alg_available(const char *name, u32 flags)
{
	int ret = 0;
	struct crypto_alg *alg = crypto_alg_mod_lookup(name);
	
	if (alg) {
		crypto_alg_put(alg);
		ret = 1;
	}
	
	return ret;
}

static int __init init_crypto(void)
{
	printk(KERN_INFO "Initializing Cryptographic API\n");
	crypto_init_proc();
	return 0;
}

__initcall(init_crypto);

EXPORT_SYMBOL_GPL(crypto_register_alg);
EXPORT_SYMBOL_GPL(crypto_unregister_alg);
EXPORT_SYMBOL_GPL(crypto_alloc_tfm);
EXPORT_SYMBOL_GPL(crypto_free_tfm);
EXPORT_SYMBOL_GPL(crypto_alg_available);
