/*
 * Cryptographic API.
 *
 * Digest operations.
 *
 * Copyright (c) 2002 James Morris <jmorris@intercode.com.au>
 *
 * HW crypto extensions added by Eugene Surovegin <ebs@ebshome.net>, 2004
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 */
#include <linux/crypto.h>
#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/highmem.h>
#include <asm/scatterlist.h>
#include "internal.h"

static void init(struct crypto_tfm *tfm)
{
	tfm->__crt_alg->cra_digest.dia_init(crypto_tfm_ctx(tfm));
}

static inline void __update(struct crypto_tfm *tfm,
                	    struct scatterlist *sg, unsigned int nsg, 
			    struct crypto_alg *alg, void* ctx)
{
	unsigned int i;

	for (i = 0; i < nsg; i++) {

		struct page *pg = sg[i].page;
		unsigned int offset = sg[i].offset;
		unsigned int l = sg[i].length;

		do {
			unsigned int bytes_from_page = min(l, ((unsigned int)
							   (PAGE_SIZE)) - 
							   offset);
			char *p = crypto_kmap(pg, 0) + offset;

			alg->cra_digest.dia_update(ctx, p, bytes_from_page);
			crypto_kunmap(p, 0);
			crypto_yield(tfm);
			offset = 0;
			pg++;
			l -= bytes_from_page;
		} while (l > 0);
	}
}

static void update(struct crypto_tfm *tfm,
                   struct scatterlist *sg, unsigned int nsg)
{
	__update(tfm, sg, nsg, tfm->__crt_alg, crypto_tfm_ctx(tfm));
}

static void final(struct crypto_tfm *tfm, u8 *out)
{
	tfm->__crt_alg->cra_digest.dia_final(crypto_tfm_ctx(tfm), out);
}

static inline int __setkey(struct crypto_alg *alg, const u8 *key, 
			   unsigned int keylen, void *ctx)
{
	u32 flags;
	if (alg->cra_digest.dia_setkey == NULL)
		return -ENOSYS;
	return alg->cra_digest.dia_setkey(ctx, key, keylen, &flags);
}

static int setkey(struct crypto_tfm *tfm, const u8 *key, unsigned int keylen)
{
	return __setkey(tfm->__crt_alg, key, keylen, crypto_tfm_ctx(tfm));
}

static inline void __digest(struct crypto_tfm *tfm,
                	    struct scatterlist *sg, unsigned int nsg, u8 *out, 
			    struct crypto_alg *alg, void* ctx)
{
	unsigned int i;

	tfm->crt_digest.dit_init(tfm);
		
	for (i = 0; i < nsg; i++) {
		char *p = crypto_kmap(sg[i].page, 0) + sg[i].offset;
		alg->cra_digest.dia_update(ctx, p, sg[i].length);
		crypto_kunmap(p, 0);
		crypto_yield(tfm);
	}

	tfm->crt_digest.dit_final(tfm, out);
}

static void digest(struct crypto_tfm *tfm,
                   struct scatterlist *sg, unsigned int nsg, u8 *out)
{
	__digest(tfm, sg, nsg, out, tfm->__crt_alg, crypto_tfm_ctx(tfm));
}

#if defined(CONFIG_CRYPTO_HW)
static void hlp_init(struct crypto_tfm *tfm)
{
	tfm->__crt_alg->cra_helper->cra_digest.dia_init(crypto_tfm_hlp_ctx(tfm));
}

static void hlp_update(struct crypto_tfm *tfm,
                   struct scatterlist *sg, unsigned int nsg)
{
	__update(tfm, sg, nsg, tfm->__crt_alg->cra_helper, crypto_tfm_hlp_ctx(tfm));
}

static void hlp_final(struct crypto_tfm *tfm, u8 *out)
{
	tfm->__crt_alg->cra_helper->cra_digest.dia_final(crypto_tfm_hlp_ctx(tfm), out);
}

static int hlp_setkey(struct crypto_tfm *tfm, const u8 *key, unsigned int keylen)
{
	return __setkey(tfm->__crt_alg->cra_helper, key, keylen, crypto_tfm_hlp_ctx(tfm));
}

static void hw_digest(struct crypto_tfm *tfm,
                      struct scatterlist *sg, unsigned int nsg, u8 *out)
{
	int ret = tfm->__crt_alg->cra_digest.dia_hw_digest(crypto_tfm_ctx(tfm), sg, nsg, out);
	if (unlikely(ret))
		__digest(tfm, sg, nsg, out, tfm->__crt_alg->cra_helper, crypto_tfm_hlp_ctx(tfm));
}		   
#endif /* CONFIG_CRYPTO_HW */

int crypto_init_digest_flags(struct crypto_tfm *tfm, u32 flags)
{
	return flags ? -EINVAL : 0;
}

int crypto_init_digest_ops(struct crypto_tfm *tfm)
{
	struct digest_tfm *ops = &tfm->crt_digest;

#if defined(CONFIG_CRYPTO_HW)	
	struct crypto_alg *alg = tfm->__crt_alg;
	printk(KERN_INFO "crypto_init_digest_ops\n");
	if (alg->cra_flags & CRYPTO_ALG_HW){
		if (unlikely(!alg->cra_helper)){
			printk(KERN_ERR"crypto_init_digest_ops: empty cra_helper\n");
			return -EINVAL;
		}
		ops->dit_digest = hw_digest;
		ops->dit_init   = hlp_init;
		ops->dit_update	= hlp_update;
		ops->dit_final  = hlp_final;
		ops->dit_setkey = hlp_setkey;

		goto out;
	}
#endif	
	ops->dit_init	= init;
	ops->dit_update	= update;
	ops->dit_final	= final;
	ops->dit_digest	= digest;
	ops->dit_setkey	= setkey;
out:
	return crypto_alloc_hmac_block(tfm);
}

void crypto_exit_digest_ops(struct crypto_tfm *tfm)
{
#if defined(CONFIG_CRYPTO_HW)
	struct digest_alg *dia = &tfm->__crt_alg->cra_digest;
	int hw = tfm->__crt_alg->cra_flags & CRYPTO_ALG_HW;
	int ret=0;

	if (hw && dia->dia_hw_hmac_cleanup)
		ret = dia->dia_hw_hmac_cleanup(crypto_tfm_ctx(tfm));
	if (ret)
		printk(KERN_INFO "crypto_exit_digest_ops: failed to delete hw context!\n");
#endif
	crypto_free_hmac_block(tfm);
}
