/*
 */
#include <linux/config.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/pci_ids.h>
#include <linux/crypto.h>

#include "ixp4xx_engine.h"
#include "ixp4xx_cipher.h"
#include "ixp4xx_digest.h"
#include "ixp4xx_ll.h"
#include "ixp4xx_dbg.h"

#define DRIVER_VERSION "1.0"

#define HW_CCMP_ENABLED	1

#define MD5_DIGEST_SIZE		16
#define MD5_BLOCK_SIZE		64

#define SHA1_DIGEST_SIZE	20
#define SHA1_BLOCK_SIZE		64

#define AES_MIN_KEY_SIZE	16
#define AES_MAX_KEY_SIZE	32
#define AES_BLOCK_SIZE		16

#define CCMP_MIN_KEY_SIZE	16
#define CCMP_MAX_KEY_SIZE	32
#define CCMP_BLOCK_SIZE		1

#define DES_KEY_SIZE		8
#define DES_BLOCK_SIZE		8

#define DES3_EDE_KEY_SIZE	(3 * DES_KEY_SIZE)
#define DES3_EDE_BLOCK_SIZE	DES_BLOCK_SIZE

#define ARC4_MIN_KEY_SIZE       1
#define ARC4_MAX_KEY_SIZE       256
#define ARC4_BLOCK_SIZE         1

static void ixp4xx_dump_cipher_context(struct ixp4xx_cipher_context* c) 
{
	if (!c) {
		printk(KERN_DEBUG "invalid context is given.\n");
	} else {
		printk(KERN_DEBUG "encoding context id: %d\n", c->enc_CtxId);
		printk(KERN_DEBUG "decoding context id: %d\n", c->dec_CtxId);
	}
}

#ifdef DEBUG
static void ixp4xx_dump(void *buf, int len)
{
    int i;

    for (i = 0 ; i < len; i++)
    {
	printk("%02x", ((u8*)buf)[i]);
	if (i%2)
	    printk(" ");
	if (15 == i%16)
	    printk("\n");
    }
    printk("\n");
}
#endif

static int ixp4xx_md5_digest(void* ctx, struct scatterlist *sg, unsigned int nsg, u8* out) 
{
	struct ixp4xx_cipher_context* p = ctx;			
	printk(KERN_DEBUG "'%s' called.\n", __FUNCTION__);
	ixp4xx_dump_cipher_context(p);
	return -1;
}

static int ixp4xx_md5_hmac(void* ctx,  u8* key, unsigned int *keylen, struct scatterlist *sg, unsigned int nsg, u8* out) 
{
	return ixp4xx_hmac(IXP4XX_HMAC_MD5, ctx, key, *keylen, sg, nsg, out);
}

static int ixp4xx_sha1_digest(void* ctx, struct scatterlist *sg, unsigned int nsg, u8* out) 
{
	struct ixp4xx_cipher_context* p = ctx;			
	printk(KERN_DEBUG "'%s' called.\n", __FUNCTION__);
	ixp4xx_dump_cipher_context(p);
	return -1;
}

static int ixp4xx_sha1_hmac(void* ctx,  u8* key, unsigned int* keylen, struct scatterlist *sg, unsigned int nsg, u8* out) 
{
	return ixp4xx_hmac(IXP4XX_HMAC_SHA1, ctx, key, *keylen, sg, nsg, out);
}

static int ixp4xx_sha1_hmac_cleanup(void* ctx)
{
#ifdef DEBUG
	printk(KERN_DEBUG "'%s' called.\n", __FUNCTION__);
#endif
	return ixp4xx_hmac_cleanup(ctx);
}

static int ixp4xx_md5_hmac_cleanup(void* ctx)
{
#ifdef DEBUG
	printk(KERN_DEBUG "'%s' called.\n", __FUNCTION__);
#endif
	return ixp4xx_hmac_cleanup(ctx);
}

/* Crypto API entry points */
#define HASH_IMPL(name, algo, block_size, digest_size)			\
static struct crypto_alg name##_alg = {					\
	.cra_name	= #name,					\
	.cra_flags	= CRYPTO_ALG_TYPE_DIGEST | CRYPTO_ALG_HW,	\
	.cra_blocksize	= block_size,					\
	.cra_ctxsize	= sizeof(struct ixp4xx_cipher_context) + digest_size,\
	.cra_module	= THIS_MODULE,					\
	.cra_list	= LIST_HEAD_INIT(name##_alg.cra_list),		\
	.cra_u		= { 						\
		.digest = {						\
			.dia_digestsize	= digest_size,			\
			.dia_hw_digest 	= ixp4xx_##name##_digest,	\
			.dia_hw_hmac 	= ixp4xx_##name##_hmac,		\
			.dia_hw_hmac_cleanup = ixp4xx_##name##_hmac_cleanup	\
		}							\
	}								\
}

HASH_IMPL(md5,  MAC_ALGO_MD5, MD5_BLOCK_SIZE,  MD5_DIGEST_SIZE);
HASH_IMPL(sha1, MAC_ALGO_SHA, SHA1_BLOCK_SIZE, SHA1_DIGEST_SIZE);

void set_cipher_mode(u32 * flags, u16 * mode)
{
	*mode &= ~ENC_MODE_MASK;
	switch ( *flags & CRYPTO_TFM_MODE_MASK) {
		case CRYPTO_TFM_MODE_CBC:
			*mode |= ENC_MODE_CBC;
			break;
		case CRYPTO_TFM_MODE_ECB:
			*mode |= ENC_MODE_ECB;
			break;
		case CRYPTO_TFM_MODE_CFB:
			*mode |= ENC_MODE_CFB;
			break;
		case CRYPTO_TFM_MODE_CTR:
			*mode |= ENC_MODE_CTR;
			break;
		case CRYPTO_TFM_MODE_CCM:
			*mode |= ENC_MODE_CCM;
			break;
		default:
			printk("Unsupported TFM mode!\n");
			*flags |= CRYPTO_TFM_RES_BAD_FLAGS;
			return;
	}
}

static int ixp4xx_cipher_unset_key(struct ixp4xx_cipher_context* ctx)
{
	int ret; 

	if ((ctx->enc_CtxId != 0) && (ctx->dec_CtxId != 0)) {
		/* delete hw context */
		ret = ixp4xx_unsetkey(ctx);
	} else 
		/* no contexts registered - just return ok */
		ret=0;
#ifdef DEBUG
	if (ret)
		printk(KERN_INFO "ixp4xx_unsetkey returned %d\n", ret);
#endif

	return ret;
}

static int ixp4xx_cipher_setkey(struct ixp4xx_cipher_context* ctx)
{
	int ret = 0;

	/* unregister any contexts */
	if ((ctx->enc_CtxId != 0) && (ctx->dec_CtxId != 0)) {
		ret = ixp4xx_unsetkey(ctx);
		if (ret)
			printk(KERN_INFO "Failed to UNregister HW ctx\n");
	}

	if ((ctx->enc_CtxId == 0) && (ctx->dec_CtxId == 0)) {
		/* should fill ctx->enc_CtxId and ctx->dec_CtxId */
		ret = ixp4xx_setkey(ctx); 
		if (ret)
			printk(KERN_INFO "Failed to register HW ctx\n");
	}
	return ret;
}

static int ixp4xx_ccmp_cipher_setkey(struct ixp4xx_cipher_context* ctx)
{
	int ret = 0;

	/* unregister any contexts */
	if ((ctx->enc_CtxId != 0) && (ctx->dec_CtxId != 0)) {
		ret = ixp4xx_unsetkey(ctx);
		if (ret)
			printk(KERN_INFO "Failed to UNregister HW ctx\n");
	}

	if ((ctx->enc_CtxId == 0) && (ctx->dec_CtxId == 0)) {
		/* should fill ctx->enc_CtxId and ctx->dec_CtxId */
		ret = ixp4xx_ccmp_hw_setkey(ctx); 
		if (ret)
			printk(KERN_INFO "Failed to register HW ctx\n");
	}
	return ret;
}


static int ixp4xx_aes_setkey(void *ctx, const u8 *key, unsigned int keylen, u32 *flags)
{
	struct ixp4xx_cipher_context* p = ctx;
	//int ret = 0;

	if ((key == NULL) && (keylen == -1)) 
		return ixp4xx_cipher_unset_key(p);

	switch (keylen){
		case 16:	
			p->mode = ENC_AES_KEY_128 | ENC_ALGO_AES;
			break;
		case 24:
			p->mode = ENC_AES_KEY_192 | ENC_ALGO_AES;
			break;
		case 32:
			p->mode = ENC_AES_KEY_256 | ENC_ALGO_AES;
			break;
		default:
			*flags |= CRYPTO_TFM_RES_BAD_KEY_LEN;
			return -EINVAL;
	}

	set_cipher_mode(flags, &p->mode);

	memcpy(p->key, key, keylen);

	p->mode &= ~ENC_ALGO_MASK;
	p->mode |= ENC_ALGO_AES;
	p->keylen = keylen;
	p->blocksize = AES_BLOCK_SIZE;

	return ixp4xx_cipher_setkey(p);
}

static int ixp4xx_ccmp_setkey(void *ctx, const u8 *key, unsigned int keylen, u32 *flags)
{
	struct ixp4xx_cipher_context* p = ctx;
	//int ret = 0;

	if ((key == NULL) && (keylen == -1)) 
		return ixp4xx_cipher_unset_key(p);

	switch (keylen){
		case 16:	
			p->mode = ENC_AES_KEY_128 | ENC_ALGO_AES;
			break;
		case 24:
			p->mode = ENC_AES_KEY_192 | ENC_ALGO_AES;
			break;
		case 32:
			p->mode = ENC_AES_KEY_256 | ENC_ALGO_AES;
			break;
		default:
			*flags |= CRYPTO_TFM_RES_BAD_KEY_LEN;
			return -EINVAL;
	}

	set_cipher_mode(flags, &p->mode);

	memcpy(p->key, key, keylen);

	p->mode &= ~ENC_ALGO_MASK;
	p->mode |= ENC_ALGO_AES;
	p->keylen = keylen;
	p->blocksize = CCMP_BLOCK_SIZE;

	return ixp4xx_ccmp_cipher_setkey(p);
}

static int ixp4xx_des_setkey(void *ctx, const u8 *key, unsigned int keylen, u32 *flags)
{
	struct ixp4xx_cipher_context* p = ctx;
	//int ret;

	if ((key == NULL) && (keylen == -1)) 
		return ixp4xx_cipher_unset_key(p);

	BUG_ON(keylen != DES_KEY_SIZE);

	set_cipher_mode(flags, &p->mode);

	memcpy(p->key, key, keylen);

	p->mode &= ~ENC_ALGO_MASK;
	p->mode |= ENC_ALGO_DES;
	p->keylen = keylen;
	p->blocksize = DES_BLOCK_SIZE;

	return ixp4xx_cipher_setkey(p);
}

static int ixp4xx_des3_ede_setkey(void *ctx, const u8 *key, unsigned int keylen, u32 *flags)
{
	struct ixp4xx_cipher_context* p = ctx;
	// int ret;

	if ((key == NULL) && (keylen == -1)) 
		return ixp4xx_cipher_unset_key(p);

	BUG_ON(keylen != DES3_EDE_KEY_SIZE);

	set_cipher_mode(flags, &p->mode);

	memcpy(p->key, key, keylen);
	p->mode &= ~ENC_ALGO_MASK;
	p->mode |= ENC_ALGO_3DES;
	p->keylen = keylen;
	p->blocksize = DES3_EDE_BLOCK_SIZE;

	return ixp4xx_cipher_setkey(p);
}

static int ixp4xx_arc4_setkey(void *ctx, const u8 *key, unsigned int keylen, u32 *flags)
{
	struct ixp4xx_cipher_context* p = ctx;
	//int ret;

	if ((key == NULL) && (keylen == -1)) 
		return ixp4xx_cipher_unset_key(p);

	set_cipher_mode(flags, &p->mode);

	memcpy(p->key, key, keylen);
	p->mode &= ~ENC_ALGO_MASK;
	p->mode |= ENC_ALGO_ARC4;
	p->keylen = keylen;
	p->blocksize = ARC4_BLOCK_SIZE;

	if (p->keylen != 16)
		return 0;

	/* only alloc if there is no ctx allocated */
	if ((p->enc_CtxId == 0) && (p->dec_CtxId == 0)) 
		return ixp4xx_cipher_setkey(p);

	return 0;
}

static int ixp4xx_crypt(void* ctx, struct scatterlist* dst, struct scatterlist* src, 
		      unsigned int nbytes, int enc, u32 mode, const u8* iv)
{
	struct ixp4xx_cipher_context* p = ctx;
#ifdef DEBUG
	printk(KERN_DEBUG "'%s' called, '%s' mode.\n",
	       __FUNCTION__, enc ? "encrypt" : "decrypt");
#endif
	return ixp4xx_cipher(p, dst, src, nbytes, enc, iv);
}	

static int ixp4xx_ccmp_crypt(void* ctx, struct scatterlist* dst, struct scatterlist* src, 
		      unsigned int nbytes, int enc, u32 mode, const u8* iv)
{
	struct ixp4xx_cipher_context* p = ctx;

	printk("'%s' called, '%s' mode.\n",
	       __FUNCTION__, enc ? "encrypt" : "decrypt");

	printk("dst = %p, src = %p, nbytes = %d, iv = %p\n",
					dst, src, nbytes, iv);

	return ixp4xx_ccmp(p, dst, src, nbytes, enc, iv);
}	

#define CIPHER_IMPL(name, block, keymin, keymax, ctxsize)		\
static struct crypto_alg name##_alg = {					\
	.cra_name	= #name,					\
	.cra_flags	= CRYPTO_ALG_TYPE_CIPHER | CRYPTO_ALG_HW,	\
	.cra_blocksize	= block,					\
	.cra_ctxsize	= sizeof(struct ixp4xx_digest_context) + ctxsize,\
	.cra_module	= THIS_MODULE,					\
	.cra_list	= LIST_HEAD_INIT(name##_alg.cra_list),		\
	.cra_u		= {						\
		.cipher = {						\
			.cia_min_keysize = keymin,			\
			.cia_max_keysize = keymax,			\
			.cia_setkey	 = ixp4xx_##name##_setkey,	\
			.cia_hw_crypt	 = ixp4xx_crypt,		\
		}							\
	}								\
}

CIPHER_IMPL(aes,      AES_BLOCK_SIZE,      AES_MIN_KEY_SIZE,  AES_MAX_KEY_SIZE,  AES_MAX_KEY_SIZE);
CIPHER_IMPL(des,      DES_BLOCK_SIZE,      DES_KEY_SIZE,      DES_KEY_SIZE,      DES_KEY_SIZE);
CIPHER_IMPL(des3_ede, DES3_EDE_BLOCK_SIZE, DES3_EDE_KEY_SIZE, DES3_EDE_KEY_SIZE, DES3_EDE_KEY_SIZE);
CIPHER_IMPL(arc4,     ARC4_BLOCK_SIZE,     ARC4_MIN_KEY_SIZE, ARC4_MAX_KEY_SIZE, ARC4_MAX_KEY_SIZE);


#ifdef HW_CCMP_ENABLED

#if 0 /* software algo - must be */
/* just dummy functions for sw algo */
static void ccmp_encrypt(void *ctx_arg, u8 *out, const u8 *in)
{
		printk("ccmp_encrypt: unexpected call to software algo!\n");
		return;
}

static void ccmp_decrypt(void *ctx_arg, u8 *out, const u8 *in)
{
		printk("ccmp_decrypt: unexpected call to software algo!\n");
		return;
}
#endif

static struct crypto_alg ccmp_alg = {					
	.cra_name	= "ccmp",					
	.cra_flags	= CRYPTO_ALG_TYPE_CIPHER | CRYPTO_ALG_HW,	
	.cra_blocksize	= CCMP_BLOCK_SIZE,					
	.cra_ctxsize	= sizeof(struct ixp4xx_digest_context) + CCMP_MAX_KEY_SIZE,
	.cra_module	= THIS_MODULE,					
	.cra_list	= LIST_HEAD_INIT(ccmp_alg.cra_list),		
	.cra_u		= {						
		.cipher = {						
			.cia_min_keysize = CCMP_MIN_KEY_SIZE,			
			.cia_max_keysize = CCMP_MAX_KEY_SIZE,		
			.cia_setkey	 = ixp4xx_ccmp_setkey,	
			.cia_hw_crypt	 = ixp4xx_ccmp_crypt		
		}							
	}								
};

#endif /* HW_CCMP_ENABLED */

static void __init ixp4xx_register_algo(struct crypto_alg* alg) 
{
	int status;

	if (!alg) 
		BUG();

	if ((status = crypto_register_alg(alg)) != 0) 
		printk(KERN_ERR "Failed to register %s algo '%s'\n", 
						alg->cra_flags & CRYPTO_ALG_HW?"h/w":"s/w",
						alg->cra_name);
	else
		printk(KERN_INFO "%s '%s' registered successfully.\n", 
						alg->cra_flags & CRYPTO_ALG_HW?"H/w":"S/w",
						alg->cra_name);
}

static int __init ixp4xx_init(void)
{
	if (ixp4xx_engine_init()) 
		return -1;

	printk(KERN_INFO "IXP4XX HW crypto driver v" DRIVER_VERSION "\n");
	/* Register kernel Cryptographic API handlers */

	ixp4xx_register_algo(&md5_alg);
	ixp4xx_register_algo(&sha1_alg);
	ixp4xx_register_algo(&aes_alg);
	ixp4xx_register_algo(&des_alg);
	ixp4xx_register_algo(&des3_ede_alg);
	ixp4xx_register_algo(&arc4_alg);

#ifdef HW_CCMP_ENABLED
	/* register dummy software ccmp algo - otherwise we can not register hw algo */
	// ixp4xx_register_algo(&ccmp_sw_alg);
	ixp4xx_register_algo(&ccmp_alg);
#endif

	return 0;
}

static void __exit ixp4xx_unregister_algo(struct crypto_alg* alg) 
{
	int status;

	if (!alg) 
		BUG();

	if ((status = crypto_unregister_alg(alg)) != 0) 
		printk(KERN_ERR "Failed to unregister %s '%s', status: %d\n", 
						alg->cra_flags & CRYPTO_ALG_HW?"h/w":"s/w",
						alg->cra_name, status);
	else
		printk(KERN_INFO "%s '%s' unregistered successfully.\n", 
						alg->cra_flags & CRYPTO_ALG_HW?"H/w":"S/w",
						alg->cra_name);
}

static void __exit ixp4xx_fini(void)
{
	ixp4xx_unregister_algo(&md5_alg);
	ixp4xx_unregister_algo(&sha1_alg);
	ixp4xx_unregister_algo(&aes_alg);
	ixp4xx_unregister_algo(&des_alg);
	ixp4xx_unregister_algo(&des3_ede_alg);
	ixp4xx_unregister_algo(&arc4_alg);

#ifdef HW_CCMP_ENABLED
	/* unregister hw and sw ccmp algo -- hw first */
	ixp4xx_unregister_algo(&ccmp_alg);
	// ixp4xx_unregister_algo(&ccmp_sw_alg);
#endif

	ixp4xx_engine_stop();
}

module_init(ixp4xx_init);
module_exit(ixp4xx_fini);

MODULE_AUTHOR("Gemtek");
MODULE_DESCRIPTION("Driver for IXP4XX crypto engine, v"DRIVER_VERSION);
MODULE_LICENSE("GPL");
