#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <asm/semaphore.h>

#include "IxQMgr.h"
#include "IxNpeDl.h"
#include "IxOsServices.h"
#include "ixp425.h"
#include "IxOsCacheMMU.h"
#include "IxCryptoAcc.h"
#include "ix_ossl.h"
#include "IxTypes.h"
#include "IxCryptoAcc.h"

#include "ixp4xx.h"
#include "ixp4xx_ll.h"
#include "ixp4xx_digest.h"


static void ixp4xx_digest_context_callback(       
	UINT32 cryptoCtxId,
        IX_MBUF *mbuf, 
        IxCryptoAccStatus status)
{
	struct ixp4xx_crypto_context* icc = 0;

	if (mbuf) 
		icc = mbuf->priv;

	if (!icc) { 
		printk(KERN_ERR "h/w request crypto context is NULL.\n");
		BUG();
	} 

#ifdef DEBUG
	printk(KERN_DEBUG "'%s' called, with %d context id, status: %d\n",
	       __FUNCTION__, cryptoCtxId, status);
#endif

	switch (status) {
	case IX_CRYPTO_ACC_STATUS_SUCCESS:
		if (mbuf) 
			ixp4xx_free_mbuf(mbuf, mbuf->m_len);


		if (icc->completion) 
			complete(icc->completion);

		icc->callback = 1;

		break;
	case IX_CRYPTO_ACC_STATUS_WAIT:
		if (mbuf)
			ixp4xx_free_mbuf(mbuf, mbuf->m_len);
#ifdef DEBUG
		printk(KERN_DEBUG "h/w digest context registration is in progress. [WAIT]\n");
#endif
		break;
	default:
		printk(KERN_DEBUG "h/w digest context registration status: %d\n", status);
	};
}


static void ixp4xx_digest_perform_callback( 
        UINT32 cryptoCtxId,
        IX_MBUF *pSrcMbuf, 
        IX_MBUF *pDestMbuf,
        IxCryptoAccStatus status)
{
	struct ixp4xx_crypto_context* icc = 0;

	if (pSrcMbuf) 
		icc = pSrcMbuf->priv;

	if (!icc) { 
		printk(KERN_ERR "h/w request crypto context is NULL.\n");
		BUG();
	} 

	if (icc->completion) 
		complete(icc->completion);

	icc->callback = 1;

#ifdef DEBUG
	printk(KERN_DEBUG "'%s' called, with %d context id, status: %d\n",
	       __FUNCTION__, cryptoCtxId, status);
#endif
}

void ixp4xx_digest_ctx_init(IxCryptoAccCtx * cryptoAccCtx)
{
	memset(cryptoAccCtx, 0, sizeof(*cryptoAccCtx));

	cryptoAccCtx->operation = IX_CRYPTO_ACC_OP_TYPE_OF_OPERATION;
	cryptoAccCtx->cipherCtx.cipherAlgo = IX_CRYPTO_ACC_CIPHER_NULL;
	cryptoAccCtx->cipherCtx.cipherMode = IX_CRYPTO_ACC_MODE_NULL;
	cryptoAccCtx->authCtx.authAlgo =IX_CRYPTO_ACC_AUTH_NULL;
}

#define MINIMUM_DIGEST_MBUF_SIZE	64
int ixp4xx_register_digest_ctx(int *ctxid, int digest, 
			       IxCryptoAccCtx* hwctx, 
			       u8* key, size_t keylen) 
{
	int status;
	struct ixp4xx_crypto_context icc = { hwctx , 0 };
	IX_MBUF *mbuf1, *mbuf2;

	/* primary and secondary mbuf for var. chaining, m_data is of 
	 * IX_CRYPTO_AUTHENTICATION_DATA_BLOCK_LENGTH. Latter mbufs are
	 * freed via callback, see ixp4xx_digest_context_callback().
	 */
	mbuf1 = ixp4xx_alloc_mbuf(MINIMUM_DIGEST_MBUF_SIZE);
	mbuf2 = ixp4xx_alloc_mbuf(MINIMUM_DIGEST_MBUF_SIZE);

	if (!mbuf1 || !mbuf2) {
		ixp4xx_free_mbuf(mbuf1, MINIMUM_DIGEST_MBUF_SIZE);
		ixp4xx_free_mbuf(mbuf2, MINIMUM_DIGEST_MBUF_SIZE);

		return -ENOMEM;
	}

	if (!hwctx || !ctxid) return -EINVAL;

	ixp4xx_digest_ctx_init(hwctx);
	hwctx->operation = IX_CRYPTO_ACC_OP_AUTH_CALC;
#ifdef DEBUG
	printk(KERN_DEBUG "key length: %d, key: \n", keylen);

	if (key &&  keylen)
		ixp4xx_hex_dump(key, keylen);
	else
		printk(KERN_DEBUG "invalid key: %p or keylen: %d\n", key, keylen);
#endif

	switch (digest) {
	case IXP4XX_HMAC_MD5:
		hwctx->authCtx.authAlgo = IX_CRYPTO_ACC_AUTH_MD5;
		hwctx->authCtx.authDigestLen = IX_CRYPTO_ACC_MD5_DIGEST_128;
		hwctx->authCtx.authKeyLen = IX_CRYPTO_ACC_MD5_KEY_128;
		memcpy(hwctx->authCtx.key.authKey, key, keylen);
		hwctx->authCtx.aadLen = 0;
		break;
	case IXP4XX_HMAC_SHA1:
		hwctx->authCtx.authAlgo = IX_CRYPTO_ACC_AUTH_SHA1;
		hwctx->authCtx.authKeyLen = IX_CRYPTO_ACC_SHA1_KEY_160;
		hwctx->authCtx.authDigestLen = IX_CRYPTO_ACC_SHA1_DIGEST_160;
		memcpy(hwctx->authCtx.key.authKey, key, keylen);
		hwctx->authCtx.aadLen = 0;
		break;
	default:
		return -EINVAL;
	};

	/* Oh no,in-place operation, please ... oh well, it doesn't crash
	 * though if not-in-place operation */
	hwctx->useDifferentSrcAndDestMbufs = 1;	

	mbuf1->priv = &icc;
	mbuf2->priv = &icc;
	/* Register the context, as a note , primary and secondory mbuf's are 
	 * used internally for a variables chaining & so on. For
	 * authentication/digest calculations these will be populated
	 * internally by the library and freed via callback. 
	 *
	 * So we are providing here empty mbufs ...  No, at least of 64 bytes 
	 * (as required in ixCryptoRegisterAuthAlgoRegister function.
	 */
	status = ixCryptoAccCtxRegister(hwctx,
					mbuf1,
					mbuf2,
					ixp4xx_digest_context_callback,
					ixp4xx_digest_perform_callback,
					ctxid);


	if (status == IX_CRYPTO_ACC_STATUS_SUCCESS) {
		ixp4xx_crypto_wait(&icc);

		if (!icc.callback) {
			printk(KERN_DEBUG "h/w digest context register callback timeout.\n");	
		}
	} else {
		printk("Digest context registration failed with status(%d)\n", status);
		return IX_FAIL;
	}


	return 0;
}

int ixp4xx_perform_digest(int ctxid, void *data, size_t dlength, void* out, size_t olength) 
{
	IX_MBUF *mbuf1, *mbuf2;
	DECLARE_COMPLETION(completion);
	struct ixp4xx_crypto_context icc = { 0, 0, 0 };
	int status; 

#define IX_CRYPTO_MAX_AUTH_LENGTH       0xFFBF
	if (dlength > IX_CRYPTO_MAX_AUTH_LENGTH) {
#ifdef DEBUG
		printk(KERN_DEBUG "digest data size is %d, max: %d\n", 
		       dlength, 
		       IX_CRYPTO_MAX_AUTH_LENGTH);
#endif
		/* try s/w ipsec implementation ... otherwise device will hang up! */
		return -1;
	}
	
#ifdef DEBUG
	printk(KERN_DEBUG "dumping data: \n");
	ixp4xx_hex_dump(data, dlength);
#endif
	mbuf1 = ixp4xx_alloc_mbuf(0);
	if (!mbuf1) {
		return -ENOMEM;
	}

	mbuf2 = ixp4xx_alloc_mbuf(dlength + olength);
	if (!mbuf2) {
		ixp4xx_free_mbuf(mbuf1, 0);
		return -ENOMEM;
	}
	/* Doesn't work in 100% cases  ... sometimes crashes. In-place
	 * operation is disabled. Below a description of a hack, if you 
	 * want to enable, don't forget to change 1 to 0, 
	 * 
	 *    hwctx->useDifferentSrcAndDestMbufs = 1;	
	 *
	 * Hack, for a in-place digest operation, in order to avoid copying data,
	 * it is assumed that data, of size dlength, has engough room to store 
	 * checksum, of size olength. 
	 *
	 * [data] + [digest of olength extra bytes]. Which is the case for
	 * skb_buff today. So we are lucky and therefore no need to copy to 
	 * allocate secondary mbuf of dlength + olength size in bytes. Digest
	 * calculation is in place operation now.
	 */

	mbuf1->m_data = data;
	mbuf1->m_len  = dlength;

	/* zero checksum */
	memset(mbuf2->m_data + dlength, 0, olength);

	/* assign request context. */
	mbuf1->priv = &icc;
	mbuf2->priv = &icc;

	if (!in_interrupt()) 
		icc.completion = &completion;

	status = ixCryptoAccAuthCryptPerform(ctxid, 
					     mbuf1,
					     mbuf2, 
					     0,
					     dlength,
					     0,
					     dlength,
					     dlength,	/* offset in data, to store digest */
					     0);	/* ignored if not encrypted/decrypted */

	if (status == IX_CRYPTO_ACC_STATUS_SUCCESS) {
		ixp4xx_crypto_wait(&icc);

		if (!icc.callback) {
			printk(KERN_DEBUG "h/w digest perform callback timeout.\n");	
		} else {
#ifdef DEBUG 
	 		ixp4xx_hex_dump(mbuf2->m_data + dlength, olength);
#endif
			memcpy(out, mbuf2->m_data + dlength, olength);
			status = 0;
		}
	} else {
		printk(KERN_DEBUG "h/w digest perform failed with status(%d)\n", status);
	}


	ixp4xx_free_mbuf(mbuf1, 0);
	ixp4xx_free_mbuf(mbuf2, dlength + olength);

	return status;
}

int ixp4xx_hmac(int type, 
		struct ixp4xx_digest_context* ctx,  
		u8* key, unsigned int keylen, 
		struct scatterlist *sg, unsigned int nsg, 
		u8* out)
{
#if 0
	return -1;
#else
	int out_len = (type == IXP4XX_HMAC_MD5) ? IX_CRYPTO_ACC_MD5_DIGEST_128 : IX_CRYPTO_ACC_SHA1_DIGEST_160;
	int status;

	if (nsg > 1) 
		BUG();

	if ((status = ixp4xx_hmac_init(type, ctx, key, keylen)))  {
		printk(KERN_DEBUG "using s/w crypto fallback.\n");
		return status;
	}

#ifdef DEBUG	
	printk(KERN_DEBUG "digest: %s, '%s' called, key : %d bytes.\n", 
	       type == IXP4XX_HMAC_MD5 ? "md5" : "sha1", 
	       __FUNCTION__,
	       keylen);

	if (!sg->address || !sg->length) 
		return -EINVAL;
#endif

#ifdef DEBUG
	printk(KERN_DEBUG "sg n: %d, sg->address: %p, page_address(sg->page) + offset: %p, length: %d\n",
	       nsg, sg->address, page_address(sg->page)+sg->offset, sg->length);
#endif

 
	status = ixp4xx_perform_digest(ctx->ctxid, 
				       page_address(sg->page)+sg->offset, sg->length, 
				       out, out_len);
	if (status) {
#ifdef DEBUG
		printk(KERN_DEBUG "h/w digest perform on context(%d), failed.\n", ctx->ctxid);     
#endif
	}

	return status;
#endif
}

int ixp4xx_hmac_init(int type, struct ixp4xx_digest_context* ctx, u8* key, size_t keylen) 
{
	if (ctx->ctxid != 0) return 0;

	ctx->hwctx = kmalloc( sizeof(*ctx->hwctx), in_interrupt() ? GFP_ATOMIC : GFP_KERNEL);
	if (!ctx->hwctx) 
		return -ENOMEM;

	if (ixp4xx_register_digest_ctx(&ctx->ctxid, type, ctx->hwctx, key, keylen)) {
		/* errmmm ... */
		printk(KERN_DEBUG "h/w digest context registration failed.\n");
		return -1;
	} else {
#ifdef DEBUG
		/* yahooo ! we've got Context id ! */
		printk(KERN_DEBUG "h/w digest context id: %d\n", ctx->ctxid);
#endif	
	}

	return 0;
}

int ixp4xx_hmac_cleanup(struct ixp4xx_digest_context* ctx)
{
	int status;

	if (ctx->ctxid == 0) return  0;
	
	/* unregister h/w ctx first, then free mbuf* */
	status = ixCryptoAccCtxUnregister(ctx->ctxid);
	if (status != IX_SUCCESS) {
		printk(KERN_DEBUG "failed unregister the context: %d\n", ctx->ctxid);
	}

	kfree(ctx->hwctx);
	ctx->hwctx = 0;
	ctx->ctxid = 0;

	return 0;
}


