/*
 * $Id: redboot.c,v 1.6 2001/10/25 09:16:06 dwmw2 Exp $
 *
 * Parse RedBoot-style Flash Image System (FIS) tables and
 * produce a Linux partition array to match.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/init.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>

struct fis_image_desc {
    unsigned char name[16];      // Null terminated name
    unsigned long flash_base;    // Address within FLASH of image
    unsigned long mem_base;      // Address in memory where it executes
    unsigned long size;          // Length of image
    unsigned long entry_point;   // Execution entry point
    unsigned long data_length;   // Length of actual data
    unsigned char _pad[256-(16+7*sizeof(unsigned long))];
    unsigned long desc_cksum;    // Checksum over image descriptor
    unsigned long file_cksum;    // Checksum over image data
};

struct fis_list {
	struct fis_image_desc *img;
	struct fis_list *next;
};

static inline int redboot_checksum(struct fis_image_desc *img)
{
	/* RedBoot doesn't actually write the desc_cksum field yet AFAICT */
	return 1;
}

#undef REDBOOT_CONCAT_DIRECTORY_AND_CONFIG

#define REDBOOT_NAME		"RedBoot"
#undef REDBOOT_FIX_CONFIG_SIZE
#define REDBOOT_CONFIG_NAME	"RedBoot config"
#define FISDIR_FIX_CONFIG_SIZE
#define FIS_DIRECTORY_NAME	"FIS directory"
#define WORKING_FLASH_NAME      "Working flash"

/* AR531x platforms usualy have one or two reserved erase-block-sized blocks 
 * at the end of FLASH memory (radio/board configuration data) */
#ifndef CONFIG_MTD_REDBOOT_RESERVED
#warning "CONFIG_MTD_REDBOOT_RESERVED not set, setting to 2"
#define CONFIG_MTD_REDBOOT_RESERVED 2
#endif

#ifdef REDBOOT_CONCAT_DIRECTORY_AND_CONFIG
void show_fis_part_info(struct fis_image_desc * f)
{
	/*
	unsigned char name[16];      // Null terminated name
	unsigned long flash_base;    // Address within FLASH of image
	unsigned long mem_base;      // Address in memory where it executes
	unsigned long size;          // Length of image
	unsigned long entry_point;   // Execution entry point
	unsigned long data_length;   // Length of actual data
	unsigned char _pad[256-(16+7*sizeof(unsigned long))];
	unsigned long desc_cksum;    // Checksum over image descriptor
	unsigned long file_cksum;    // Checksum over image data
	*/

//	printk("DEBUG: name = %s, flash_base = %ld, mem_base = %ld, size = %ld,\n",
//			f->name, f->flash_base, f->mem_base, f->size);

//	printk("  entry_point = %ld, data_length = %ld, desc_cksum = %ld, file_cksum = %ld\n",
//			f->entry_point, f->data_length, f->desc_cksum, f->file_cksum);
	return;
}
#endif

int parse_redboot_partitions(struct mtd_info *master, struct mtd_partition **pparts, unsigned long origin)
{
	int nrparts = 0;
	struct fis_image_desc *buf;
	struct mtd_partition *parts;
	struct fis_list *fl = NULL, *tmp_fl;
	/* FIS directory and RedBoot config */
	struct fis_list *fl_dir = NULL, *fl_cfg = NULL;
	int ret, i;
	size_t retlen;
	char *names;
	int namelen = 0;
	int directory = 1;
	int fisdir_idx = -1, redboot_idx = -1;

	buf = kmalloc(PAGE_SIZE, GFP_KERNEL);

	if (!buf)
		return -ENOMEM;
retry:

	/* Read the start of the last erase block */
	ret = master->read(master, master->size - 
			 (master->erasesize * (directory + 1)),
			   PAGE_SIZE, &retlen, (void *)buf);

	if (ret)
		goto out;

	if (retlen != PAGE_SIZE) {
		ret = -EIO;
		goto out;
	}

	/* RedBoot image could appear in any of the first three slots */
	for (i = 0; i < 3; i++) {
		if (!memcmp(buf[i].name, REDBOOT_NAME, 8))
			break;
	}
	if (i == 3) {
		/* Didn't find it */
		if (master->size - (master->erasesize * (directory + 2)) > 0) {
			/* maybe next block? */
			directory++;
			goto retry;
		}
		printk(KERN_NOTICE "No RedBoot partition table detected in %s\n",
		       master->name);
		ret = 0;
		goto out;
	}

	for (i = 0; i < PAGE_SIZE / sizeof(struct fis_image_desc); i++) {
		struct fis_list *new_fl, **prev;

		if (buf[i].name[0] == 0xff)
			break;
		if (!redboot_checksum(&buf[i]))
			break;

		new_fl = kmalloc(sizeof(struct fis_list), GFP_KERNEL);
		namelen += strlen(buf[i].name)+1;
		if (!new_fl) {
			ret = -ENOMEM;
			goto out;
		}
		new_fl->img = &buf[i];
		buf[i].flash_base &= master->size-1;

		/* I'm sure the JFFS2 code has done me permanent damage.
		 * I now think the following is _normal_
		 */
		prev = &fl;
		while(*prev && (*prev)->img->flash_base < new_fl->img->flash_base)
			prev = &(*prev)->next;
		new_fl->next = *prev;
		*prev = new_fl;
#ifdef REDBOOT_FIX_CONFIG_SIZE
		if (strncmp(buf[i].name, REDBOOT_CONFIG_NAME,
								strlen(REDBOOT_CONFIG_NAME)+1) == 0) {
				/* this is RedBoot config partition */
				printk("mtd: adjusted dev mtd%d size from %d to %d bytes\n",
								i, buf[i].size, master->erasesize);
				buf[i].size = master->erasesize;
		}
#endif
#ifdef FISDIR_FIX_CONFIG_SIZE
		if (strncmp(buf[i].name, FIS_DIRECTORY_NAME,
								strlen(FIS_DIRECTORY_NAME)+1) == 0) {
				/* this is FIS directory partition */
				printk("mtd: adjusted dev mtd%d size from %lu to %u bytes\n",
								i, buf[i].size, master->erasesize);
				buf[i].size = master->erasesize;
		}
#endif


#define FIS_DIRECTORY_NAME	"FIS directory"
#define REDBOOT_CONFIG_NAME	"RedBoot config"

		/* check if it is "FIS directory" */
		if (strncmp(buf[i].name, FIS_DIRECTORY_NAME, 
					strlen(FIS_DIRECTORY_NAME)+1) == 0) 
			fl_dir = new_fl;
		/* check if it is "RedBoot config" */
		if (strncmp(buf[i].name, REDBOOT_CONFIG_NAME, 
					strlen(FIS_DIRECTORY_NAME)+1) == 0) 
			fl_cfg = new_fl;

		nrparts++;
	}

#ifdef REDBOOT_CONCAT_DIRECTORY_AND_CONFIG

#define REDBOOT_COMBINED_PART_AND_CFG	"FIS and config"

	/* Concatenate FIS directory and RedBoot config into one 
	 * "Redboot directory and config" partition */

	if (fl_dir)
		show_fis_part_info(fl_dir->img);
	else
		printk("dir NOT FOUND!\n");

	if (fl_cfg)
		show_fis_part_info(fl_cfg->img);
	else
		printk("cfg NOT FOUND!\n");

	/* check if they are exactly in one erase block */

	if (fl_dir && fl_cfg && (fl_dir->img->size + fl_cfg->img->size == master->erasesize)) {
		struct fis_list *new_fl, **prev;

		/* and create dir+cfg */
		new_fl = kmalloc(sizeof(struct fis_list), GFP_KERNEL);
		namelen += strlen(REDBOOT_COMBINED_PART_AND_CFG)+1;
		if (!new_fl) {
			ret = -ENOMEM;
			goto out;
		}
		new_fl->img = kmalloc(sizeof(struct fis_image_desc), GFP_KERNEL);
		if (!new_fl->img) {
			ret = -ENOMEM;
			goto out;
		}

		memmove(new_fl->img, fl_dir->img, sizeof(struct fis_image_desc));
		memmove(new_fl->img->name, REDBOOT_COMBINED_PART_AND_CFG, 
				strlen(REDBOOT_COMBINED_PART_AND_CFG)+1);
		
		new_fl->img->size = fl_dir->img->size + fl_cfg->img->size;

		buf[i].flash_base &= master->size-1;


		/* print information about new partition */
		show_fis_part_info(new_fl->img);


		/* I'm sure the JFFS2 code has done me permanent damage.
		 * I now think the following is _normal_
		 */
		prev = &fl;
		while(*prev && (*prev)->img->flash_base < new_fl->img->flash_base)
			prev = &(*prev)->next;
		new_fl->next = *prev;
		*prev = new_fl;

		printk("Combined partition <%s> for RedBoot FIS directory and config created.\n",
				new_fl->img->name);

		nrparts++;

		/* free directory and config  -  OPTIONAL */
	} else
		if (fl_dir && fl_cfg)
			printk("Unexpected cfg or dir size ! (dir size: %ld, cfg size: %ld, erase size: %d\n",
					fl_dir->img->size, fl_cfg->img->size, master->erasesize);
		else
			printk("cfg or dir not found.\n");


#endif /* REDBOOT_CONCAT_DIRECTORY_AND_CONFIG */

#define REDBOOT_EEPROM_NAME	"EEPROM"
	/************************************************************************
	 * create EEPROM partition at the end of flash, 1 or 2 (best guess) erase blocks 
	 */ 
	{
		struct fis_list *new_fl, **prev;
		unsigned long reserved = CONFIG_MTD_REDBOOT_RESERVED;


		new_fl = kmalloc(sizeof(struct fis_list), GFP_KERNEL);
		namelen += strlen(REDBOOT_EEPROM_NAME)+1;
		if (!new_fl) {
			ret = -ENOMEM;
			goto out;
		}
		new_fl->img = kmalloc(sizeof(struct fis_image_desc), GFP_KERNEL);
		if (!new_fl->img) {
			ret = -ENOMEM;
			goto out;
		}

		if (fl_dir && fl_cfg) {
			long res_size = 0;
			res_size = (master->size - fl_cfg->img->flash_base - fl_cfg->img->size);
			if ((res_size > 0) && (res_size % master->erasesize == 0)) {
				reserved = res_size / master->erasesize;
			}
		}

		new_fl->img->size = master->erasesize * reserved;
		new_fl->img->flash_base = master->size - 
				(master->erasesize * reserved);
		memmove(new_fl->img->name, REDBOOT_EEPROM_NAME, 
						strlen(REDBOOT_EEPROM_NAME)+1);
		
		new_fl->img->flash_base &= master->size-1;

		/* I'm sure the JFFS2 code has done me permanent damage.
		 * I now think the following is _normal_
		 */
		prev = &fl;
		while(*prev && (*prev)->img->flash_base < new_fl->img->flash_base)
			prev = &(*prev)->next;
		new_fl->next = *prev;
		*prev = new_fl;

		nrparts++;
	} /* create eeprom partition */
	

	if (fl->img->flash_base)
		nrparts++;

	for (tmp_fl = fl; tmp_fl->next; tmp_fl = tmp_fl->next) {
		if (tmp_fl->img->flash_base + tmp_fl->img->size + master->erasesize <= tmp_fl->next->img->flash_base)
			nrparts++;
	}
	parts = kmalloc(sizeof(*parts)*(nrparts + 1) + namelen, GFP_KERNEL); //UBNT add one for all except RedBoot, FIS Dir, FIS Cfg and EEPROM

	if (!parts) {
		ret = -ENOMEM;
		goto out;
	}
	names = (char *)&parts[nrparts + 1];
	memset(parts, 0, sizeof(*parts)*(nrparts + 1) + namelen);
	i=0;

	if (fl->img->flash_base) {
	       parts[0].name = "unallocated space";
	       parts[0].size = fl->img->flash_base;
	       parts[0].offset = 0;
	}
	for ( ; i<nrparts; i++) {
		parts[i].size = fl->img->size;
		parts[i].offset = fl->img->flash_base;
		parts[i].name = names;

		strcpy(names, fl->img->name);
		names += strlen(names)+1;
		if (strcmp(fl->img->name, REDBOOT_NAME) == 0)
                        redboot_idx = i;
		else if (strcmp(fl->img->name, FIS_DIRECTORY_NAME) == 0)
                        fisdir_idx = i;

		if(fl->next && fl->img->flash_base + fl->img->size + master->erasesize <= fl->next->img->flash_base) {
			i++;
			parts[i].offset = parts[i-1].size + parts[i-1].offset;
			parts[i].size = fl->next->img->flash_base - parts[i].offset;
			parts[i].name = "unallocated space";
		}
		tmp_fl = fl;
		fl = fl->next;
		kfree(tmp_fl);
	}
	ret = nrparts;
	if ((redboot_idx >= 0) && (fisdir_idx > 0))
	{
		parts[ret].offset = parts[redboot_idx].offset + parts[redboot_idx].size;
		parts[ret].size = parts[fisdir_idx].offset - parts[ret].offset;
		parts[ret].name = WORKING_FLASH_NAME;
            	++ret;
	}
	*pparts = parts;
 out:
	while (fl) {
		struct fis_list *old = fl;
		fl = fl->next;
		kfree(old);
	}
	kfree(buf);
	return ret;
}

/* EXPORT_SYMBOL(parse_redboot_partitions); */
static struct mtd_part_parser redboot_parser = {
	.owner = THIS_MODULE,
	.parse_fn = parse_redboot_partitions,
	.name = REDBOOT_NAME,
};


static int __init redboot_parser_init(void)
{
	return register_mtd_parser(&redboot_parser);
}

static void __exit redboot_parser_exit(void)
{
	deregister_mtd_parser(&redboot_parser);
}

module_init(redboot_parser_init);
module_exit(redboot_parser_exit);



MODULE_LICENSE("GPL");
MODULE_AUTHOR("Red Hat, Inc. - David Woodhouse <dwmw2@cambridge.redhat.com>");
MODULE_DESCRIPTION("Parsing code for RedBoot Flash Image System (FIS) tables");
