/*
    lm77.c - Part of lm_sensors, Linux kernel modules for hardware
             monitoring
    Copyright (c) 1998, 1999  Frodo Looijaard <frodol@dds.nl>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <linux/module.h>
#include <linux/slab.h>
#include <linux/i2c.h>
#include <linux/i2c-proc.h>
#include <linux/init.h>
#define LM_DATE "20041007"
#define LM_VERSION "2.8.8"
#include "lm77.h"

/* Addresses to scan */
static unsigned short normal_i2c[] = { SENSORS_I2C_END };
static unsigned short normal_i2c_range[] = { 0x48, 0x4f, SENSORS_I2C_END };
static unsigned int normal_isa[] = { SENSORS_ISA_END };
static unsigned int normal_isa_range[] = { SENSORS_ISA_END };

/* Insmod parameters */
SENSORS_INSMOD_1(lm77);

/* Many LM77 constants specified below */

/* The LM77 registers */
#define LM77_REG_TEMP_INPUT     0x00
#define LM77_REG_CONF           0x01
#define LM77_REG_TEMP_HYST      0x02
#define LM77_REG_TEMP_CRIT      0x03
#define LM77_REG_TEMP_MIN       0x04
#define LM77_REG_TEMP_MAX       0x05


/* Each client has this additional data */
struct lm77_data {
	struct i2c_client client;
	int sysctl_id;

	struct semaphore update_lock;
	char valid;		/* !=0 if following fields are valid */
	unsigned long last_updated;	/* In jiffies */

	int temp_input, temp_crit, 
		temp_min, temp_max, temp_hyst; /* Register values */

	u8 alarms;
};

static int lm77_attach_adapter(struct i2c_adapter *adapter);
static int lm77_detect(struct i2c_adapter *adapter, int address,
		       unsigned short flags, int kind);
static void lm77_init_client(struct i2c_client *client);
static int lm77_detach_client(struct i2c_client *client);

static int lm77_read_value(struct i2c_client *client, u8 reg);
static int lm77_write_value(struct i2c_client *client, u8 reg, u16 value);
static void lm77_temp(struct i2c_client *client, int operation,
		      int ctl_name, int *nrels_mag, long *results);
static void lm77_crit(struct i2c_client *client, int operation,
		      int ctl_name, int *nrels_mag, long *results);
static void lm77_hyst(struct i2c_client *client, int operation,
		      int ctl_name, int *nrels_mag, long *results);
static void lm77_min(struct i2c_client *client, int operation,
		      int ctl_name, int *nrels_mag, long *results);
static void lm77_max(struct i2c_client *client, int operation,
		      int ctl_name, int *nrels_mag, long *results);
static void lm77_alarms(struct i2c_client *client, int operation,
		      int ctl_name, int *nrels_mag, long *results);
static void lm77_update_client(struct i2c_client *client);


/* This is the driver that will be inserted */
static struct i2c_driver lm77_driver = {
	.owner		= THIS_MODULE,
	.name		= "LM77 sensor chip driver",
	.id		= I2C_DRIVERID_LM75,
	.flags		= I2C_DF_NOTIFY,
	.attach_adapter	= lm77_attach_adapter,
	.detach_client	= lm77_detach_client,
};

/* -- SENSORS SYSCTL START -- */

#define LM77_SYSCTL_TEMP 120000	/* Degrees Celcius * 1000 */
#define LM77_SYSCTL_CRIT  80000	/* Degrees Celcius * 1000 */
#define LM77_SYSCTL_HYST    200	/* Degrees Celcius * 1000 */
#define LM77_SYSCTL_MIN   10000	/* Degrees Celcius * 1000 */
#define LM77_SYSCTL_MAX   64000	/* Degrees Celcius * 1000 */
#define LM77_SYSCTL_ALRM      0	/* Counter */

/* -- SENSORS SYSCTL END -- */

/* These files are created for each detected LM77. This is just a template;
   though at first sight, you might think we could use a statically
   allocated list, we need some way to get back to the parent - which
   is done through one of the 'extra' fields which are initialized
   when a new copy is allocated. */
static ctl_table lm77_dir_table_template[] = {
	{LM77_SYSCTL_TEMP, "temp", NULL, 0, 0644, NULL, &i2c_proc_real,
	 &i2c_sysctl_real, NULL, &lm77_temp},
	{LM77_SYSCTL_CRIT, "crit", NULL, 0, 0644, NULL, &i2c_proc_real,
	 &i2c_sysctl_real, NULL, &lm77_crit},
	{LM77_SYSCTL_HYST, "hyst", NULL, 0, 0644, NULL, &i2c_proc_real,
	 &i2c_sysctl_real, NULL, &lm77_hyst},
	{LM77_SYSCTL_MIN, "min", NULL, 0, 0644, NULL, &i2c_proc_real,
	 &i2c_sysctl_real, NULL, &lm77_min},
	{LM77_SYSCTL_MAX, "max", NULL, 0, 0644, NULL, &i2c_proc_real,
	 &i2c_sysctl_real, NULL, &lm77_max},
	{LM77_SYSCTL_ALRM, "alarms", NULL, 0, 0644, NULL, &i2c_proc_real,
	 &i2c_sysctl_real, NULL, &lm77_alarms},
	{0}
};

static int lm77_id = 0;

static int lm77_attach_adapter(struct i2c_adapter *adapter)
{
	return i2c_detect(adapter, &addr_data, lm77_detect);
}

/* This function is called by i2c_detect */
int lm77_detect(struct i2c_adapter *adapter, int address,
		unsigned short flags, int kind)
{
	int i;
	struct i2c_client *new_client;
	struct lm77_data *data;
	int err = 0;
	const char *type_name, *client_name;

	if (i2c_is_isa_adapter(adapter)) {
		printk
		    ("lm77.o: lm77_detect called for an ISA bus adapter?!?\n");
		return 0;
	}

	if (!i2c_check_functionality(adapter, I2C_FUNC_SMBUS_BYTE_DATA |
				     I2C_FUNC_SMBUS_WORD_DATA))
		    goto error0;

	/* OK. For now, we presume we have a valid client. We now create the
	   client structure, even though we cannot fill it completely yet.
	   But it allows us to access lm77_{read,write}_value. */
	if (!(data = kmalloc(sizeof(struct lm77_data), GFP_KERNEL))) {
		err = -ENOMEM;
		goto error0;
	}

	new_client = &data->client;
	new_client->addr = address;
	new_client->data = data;
	new_client->adapter = adapter;
	new_client->driver = &lm77_driver;
	new_client->flags = 0;

	/* Now, we do the remaining detection. There is no identification-
	   dedicated register so we have to rely on several tricks:
	   unused bits, registers cycling over 8-address boundaries,
	   addresses 0x04-0x07 returning the last read value.
	   The cycling+unused addresses combination is not tested,
	   since it would significantly slow the detection down and would
	   hardly add any value. */
	
	if (kind < 0) {
		int i, cur, conf, hyst, crit, min, max;

		/* addresses cycling */
		cur = i2c_smbus_read_word_data(new_client, 0);
		conf = i2c_smbus_read_byte_data(new_client, 1);
		hyst = i2c_smbus_read_word_data(new_client, 2);
		crit = i2c_smbus_read_word_data(new_client, 3);
		min = i2c_smbus_read_word_data(new_client, 4);
		max = i2c_smbus_read_word_data(new_client, 5);
		for (i = 8; i <= 0xff; i += 8)
			if (i2c_smbus_read_byte_data(new_client, i + 1) != conf
			    || i2c_smbus_read_word_data(new_client, i + 2) != hyst
			    || i2c_smbus_read_word_data(new_client, i + 3) != crit
			    || i2c_smbus_read_word_data(new_client, i + 4) != min
			    || i2c_smbus_read_word_data(new_client, i + 5) != max)
				goto exit_free;

		/* sign bits */
		if (((cur & 0x00f0) != 0xf0 && (cur & 0x00f0) != 0x0)
		    || ((hyst & 0x00f0) != 0xf0 && (hyst & 0x00f0) != 0x0)
		    || ((crit & 0x00f0) != 0xf0 && (crit & 0x00f0) != 0x0)
		    || ((min & 0x00f0) != 0xf0 && (min & 0x00f0) != 0x0)
		    || ((max & 0x00f0) != 0xf0 && (max & 0x00f0) != 0x0))
			goto exit_free;

		/* unused bits */
		if (conf & 0xe0)
			goto exit_free;

		/* 0x06 and 0x07 return the last read value */
		cur = i2c_smbus_read_word_data(new_client, 0);
		if (i2c_smbus_read_word_data(new_client, 6) != cur
		    || i2c_smbus_read_word_data(new_client, 7) != cur)
			goto exit_free;
		hyst = i2c_smbus_read_word_data(new_client, 2);
		if (i2c_smbus_read_word_data(new_client, 6) != hyst
		    || i2c_smbus_read_word_data(new_client, 7) != hyst)
			goto exit_free;
		min = i2c_smbus_read_word_data(new_client, 4);
		if (i2c_smbus_read_word_data(new_client, 6) != min
		    || i2c_smbus_read_word_data(new_client, 7) != min)
			goto exit_free;

	}
	
	/* Determine the chip type - only one kind supported! */
	if (kind <= 0)
		kind = lm77;

	if (kind == lm77) {
		type_name = "lm77";
		client_name = "LM77 chip";
	} else {
		pr_debug("lm77.o: Internal error: unknown kind (%d)?!?", kind);
		goto error1;
	}

	/* Fill in the remaining client fields and put it into the global list */
	strcpy(new_client->name, client_name);

	new_client->id = lm77_id++;
	data->valid = 0;
	init_MUTEX(&data->update_lock);

	/* Tell the I2C layer a new client has arrived */
	if ((err = i2c_attach_client(new_client)))
		goto error3;

	/* Register a new directory entry with module sensors */
	if ((i = i2c_register_entry(new_client, type_name,
					lm77_dir_table_template)) < 0) {
		err = i;
		goto error4;
	}
	data->sysctl_id = i;

	lm77_init_client(new_client);
	return 0;

/* OK, this is not exactly good programming practice, usually. But it is
   very code-efficient in this case. */

      error4:
	i2c_detach_client(new_client);
      error3:
      exit_free:
      error1:
	kfree(data);
      error0:
	return err;
}

static int lm77_detach_client(struct i2c_client *client)
{
	struct lm77_data *data = client->data;

	i2c_deregister_entry(data->sysctl_id);
	i2c_detach_client(client);
	kfree(client->data);
	return 0;
}

/* All registers are word-sized, except for the configuration register.
   LM77 uses a high-byte first convention, which is exactly opposite to
   the usual practice. */
static int lm77_read_value(struct i2c_client *client, u8 reg)
{
	if (reg == LM77_REG_CONF)
		return i2c_smbus_read_byte_data(client, reg);
	else
		return swab16(i2c_smbus_read_word_data(client, reg));
}

/* All registers are word-sized, except for the configuration register.
   LM77 uses a high-byte first convention, which is exactly opposite to
   the usual practice. */
static int lm77_write_value(struct i2c_client *client, u8 reg, u16 value)
{
	if (reg == LM77_REG_CONF)
		return i2c_smbus_write_byte_data(client, reg, value);
	else
		return i2c_smbus_write_word_data(client, reg, swab16(value));
}

static void lm77_init_client(struct i2c_client *client)
{
	/* Initialize the LM77 chip */
	int conf = lm77_read_value(client, LM77_REG_CONF);
	if (conf & 1)
		lm77_write_value(client, LM77_REG_CONF, conf & 0xfe);
}

static void lm77_update_client(struct i2c_client *client)
{
	struct lm77_data *data = client->data;

	down(&data->update_lock);

	if ((jiffies - data->last_updated > HZ + HZ / 2) ||
	    (jiffies < data->last_updated) || !data->valid) {
		pr_debug("Starting lm77 update\n");

		data->temp_input = lm77_read_value(client, LM77_REG_TEMP_INPUT);
		data->temp_hyst = lm77_read_value(client, LM77_REG_TEMP_HYST);
		data->temp_crit = lm77_read_value(client, LM77_REG_TEMP_CRIT);
		data->temp_min = lm77_read_value(client, LM77_REG_TEMP_MIN);
		data->temp_max = lm77_read_value(client, LM77_REG_TEMP_MAX);
		data->alarms = lm77_read_value(client, LM77_REG_TEMP_INPUT) & 0x0007;
		data->last_updated = jiffies;
		data->valid = 1;
	}

	up(&data->update_lock);
}


void lm77_temp(struct i2c_client *client, int operation, int ctl_name,
	       int *nrels_mag, long *results)
{
	struct lm77_data *data = client->data;
	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 1;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm77_update_client(client);
		results[0] = LM77_TEMP_FROM_REG(data->temp_input);
		*nrels_mag = 1;
	} 
}

void lm77_crit(struct i2c_client *client, int operation, int ctl_name,
	       int *nrels_mag, long *results)
{
	struct lm77_data *data = client->data;
	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 1;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm77_update_client(client);
		results[0] = LM77_TEMP_FROM_REG(data->temp_crit);
		*nrels_mag = 1;
	} else if (operation == SENSORS_PROC_REAL_WRITE) {
		if (*nrels_mag >= 1) {
			data->temp_crit = LM77_TEMP_TO_REG(results[0]);
			lm77_write_value(client, LM77_REG_TEMP_CRIT,
					 data->temp_crit);
		}
	}
}

void lm77_hyst(struct i2c_client *client, int operation, int ctl_name,
	       int *nrels_mag, long *results)
{
	struct lm77_data *data = client->data;
	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 1;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm77_update_client(client);
		results[0] = LM77_TEMP_FROM_REG(data->temp_hyst);
		*nrels_mag = 1;
	} else if (operation == SENSORS_PROC_REAL_WRITE) {
		if (*nrels_mag >= 1) {
			data->temp_hyst = LM77_TEMP_TO_REG(results[0]);
			lm77_write_value(client, LM77_REG_TEMP_HYST,
					 data->temp_hyst);
		}
	}
}
void lm77_min(struct i2c_client *client, int operation, int ctl_name,
	       int *nrels_mag, long *results)
{
	struct lm77_data *data = client->data;
	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 1;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm77_update_client(client);
		results[0] = LM77_TEMP_FROM_REG(data->temp_min);
		*nrels_mag = 1;
	}
}
void lm77_max(struct i2c_client *client, int operation, int ctl_name,
	       int *nrels_mag, long *results)
{
	struct lm77_data *data = client->data;
	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 1;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm77_update_client(client);
		results[0] = LM77_TEMP_FROM_REG(data->temp_max);
		*nrels_mag = 1;
	}
}
void lm77_alarms(struct i2c_client *client, int operation, int ctl_name,
	       int *nrels_mag, long *results)
{
	struct lm77_data *data = client->data;
	if (operation == SENSORS_PROC_REAL_INFO)
		*nrels_mag = 1;
	else if (operation == SENSORS_PROC_REAL_READ) {
		lm77_update_client(client);
		results[0] = LM77_TEMP_FROM_REG(data->alarms);
		*nrels_mag = 1;
	}
}

static int __init sm_lm77_init(void)
{
	printk(KERN_INFO "lm77.o version %s (%s)\n", LM_VERSION, LM_DATE);
	return i2c_add_driver(&lm77_driver);
}

static void __exit sm_lm77_exit(void)
{
	i2c_del_driver(&lm77_driver);
}

MODULE_AUTHOR("Frodo Looijaard <frodol@dds.nl>");
MODULE_DESCRIPTION("LM77 driver");
MODULE_LICENSE("GPL");

module_init(sm_lm77_init);
module_exit(sm_lm77_exit);
