#ifndef _NET_ESP_H
#define _NET_ESP_H

#include <net/xfrm.h>
#include <asm/scatterlist.h>

#define ESP_NUM_FAST_SG		4

struct esp_data
{
	struct scatterlist		sgbuf[ESP_NUM_FAST_SG];

	/* Confidentiality */
	struct {
		u8			*key;		/* Key */
		int			key_len;	/* Key length */
		u8			*ivec;		/* ivec buffer */
		/* ivlen is offset from enc_data, where encrypted data start.
		 * It is logically different of crypto_tfm_alg_ivsize(tfm).
		 * We assume that it is either zero (no ivec), or
		 * >= crypto_tfm_alg_ivsize(tfm). */
		int			ivlen;
		int			padlen;		/* 0..255 */
		struct crypto_tfm	*tfm;		/* crypto handle */
	} conf;

	/* Integrity. It is active when icv_full_len != 0 */
	struct {
		u8			*key;		/* Key */
		int			key_len;	/* Length of the key */
		u8			*work_icv;
		int			icv_full_len;
		int			icv_trunc_len;
		void			(*icv)(struct esp_data*,
		                               struct sk_buff *skb, int nfrags,
		                               int offset, int len, u8 *icv);
		struct crypto_tfm	*tfm;
	} auth;
};

extern int skb_to_sgvec(struct sk_buff *skb, struct scatterlist *sg, int offset, int len);
extern int skb_cow_data(struct sk_buff *skb, int tailbits, struct sk_buff **trailer);
extern void *pskb_put(struct sk_buff *skb, struct sk_buff *tail, int len);

#if defined(CONFIG_CRYPTO_HW)
static inline int
esp_hmac_digest_sg(struct esp_data *esp, struct sk_buff *skb, int nfrags, 
               int offset, int len)
{
       struct scatterlist sgbuf[nfrags > 4 ? 0 : nfrags];
       struct scatterlist *sg = sgbuf;

       if (unlikely(nfrags > 4)) {
               sg = kmalloc(sizeof(struct scatterlist) * nfrags, GFP_ATOMIC);
               if (!sg)
                       return -ENOMEM;
       }
       skb_to_sgvec(skb, sg, offset, len);
       crypto_hmac(esp->auth.tfm, esp->auth.key, &esp->auth.key_len, sg, 
                   nfrags, esp->auth.work_icv);
       if (unlikely(sg != sgbuf))
               kfree(sg);

       return 0;
}
#endif /* CONFIG_CRYPTO_HW */

static inline void
esp_hmac_digest_walk(struct esp_data *esp, struct sk_buff *skb,  
                int offset, int len)
{
	struct crypto_tfm *tfm = esp->auth.tfm;

	crypto_hmac_init(tfm, esp->auth.key, &esp->auth.key_len);
	skb_icv_walk(skb, tfm, offset, len, crypto_hmac_update);
       crypto_hmac_final(tfm, esp->auth.key, &esp->auth.key_len, esp->auth.work_icv);
}              

static inline void
esp_hmac_digest(struct esp_data *esp, struct sk_buff *skb, int nfrags, 
               int offset, int len, u8 *auth_data)
{
       memset(auth_data, 0, esp->auth.icv_trunc_len);

#if defined(CONFIG_CRYPTO_HW)  
       if (unlikely(esp_hmac_digest_sg(esp, skb, nfrags, offset, len)))
#endif 
               esp_hmac_digest_walk(esp, skb, offset, len);
               
       memcpy(auth_data, esp->auth.work_icv, esp->auth.icv_trunc_len);
}

#endif
